package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import java.util.Collections;
import java.util.Map;

/**
 * VideoBandwidthProfileOptions enables you to configure how available bandwidth is shared among the
 * {@link RemoteVideoTrack}(s) in a Group Room.
 *
 * @see <a href="https://www.twilio.com/docs/video/tutorials/using-bandwidth-profile-api">Using
 *     Bandwidth Profile API</a>
 */
public class VideoBandwidthProfileOptions {
    @Nullable private final TrackPriority dominantSpeakerPriority;
    @Nullable private final Long maxSubscriptionBitrate;
    @Nullable private final Long maxTracks;
    @Nullable private final BandwidthProfileMode mode;
    @NonNull private final Map<TrackPriority, VideoDimensions> renderDimensions;
    @Nullable private final TrackSwitchOffMode trackSwitchOffMode;

    /**
     * Get the dominant speaker priority specified by {@link
     * Builder#dominantSpeakerPriority(TrackPriority)}. This property is {@code null} by default and
     * can be set to {@code null} via {@link Builder#dominantSpeakerPriority(TrackPriority)}. A
     * {@code null} value results in the Twilio Media Server selecting the Dominant Speaker's
     * priority. This property will only be enabled if the Dominant Speaker feature is enabled via
     * {@link ConnectOptions.Builder#enableDominantSpeaker(boolean)}.
     */
    @Nullable
    public TrackPriority getDominantSpeakerPriority() {
        return dominantSpeakerPriority;
    }

    /**
     * Get the max subscription bitrate in Kilobits per second (Kbps) specified by {@link
     * Builder#maxSubscriptionBitrate(Long)}.
     *
     * <p>This property is 2400 Kbps by default. This property is allowed to be {@code null} if
     * specified in {@link Builder#maxSubscriptionBitrate(Long)}. Providing {@code null} allows the
     * maximum downlink video bandwidth for a {@link Room}.
     */
    @Nullable
    public Long getMaxSubscriptionBitrate() {
        return maxSubscriptionBitrate;
    }

    /**
     * Get the max tracks specified by {@link Builder#maxTracks(Long)} (long)}. This property is
     * {@code null} by default and can be set to {@code null} via {@link Builder#maxTracks(Long)}. A
     * {@code null} value results in no limits on the number of visible {@link RemoteVideoTrack}s.
     */
    @Nullable
    public Long getMaxTracks() {
        return maxTracks;
    }

    /**
     * Get the bandwidth profile mode specified by {@link
     * Builder#trackSwitchOffMode(TrackSwitchOffMode)}. This property is {@code null} by default and
     * can be set to {@code null} via {@link Builder#mode(BandwidthProfileMode)}. A {@code null}
     * value results in the Twilio Media Server selecting the mode.
     */
    @Nullable
    public BandwidthProfileMode getMode() {
        return mode;
    }

    /**
     * Get the render dimensions specified by {@link Builder#renderDimensions(Map)}. This property
     * is an empty {@link Map} by default.
     */
    @NonNull
    public Map<TrackPriority, VideoDimensions> getRenderDimensions() {
        return renderDimensions;
    }

    /**
     * Get the track switch off mode specified by {@link
     * Builder#trackSwitchOffMode(TrackSwitchOffMode)}. This property is {@code null} by default and
     * can be set to {@code null} via {@link Builder#trackSwitchOffMode(TrackSwitchOffMode)}. A
     * {@code null} value results in the Twilio Media Server selecting the track switch off mode.
     */
    @Nullable
    public TrackSwitchOffMode getTrackSwitchOffMode() {
        return trackSwitchOffMode;
    }

    private VideoBandwidthProfileOptions(Builder builder) {
        this.dominantSpeakerPriority = builder.dominantSpeakerPriority;
        this.maxSubscriptionBitrate = builder.maxSubscriptionBitrate;
        this.maxTracks = builder.maxTracks;
        this.mode = builder.mode;
        this.renderDimensions = builder.renderDimensions;
        this.trackSwitchOffMode = builder.trackSwitchOffMode;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        VideoBandwidthProfileOptions that = (VideoBandwidthProfileOptions) o;

        if (dominantSpeakerPriority != that.dominantSpeakerPriority) return false;
        if (!Util.equals(maxSubscriptionBitrate, that.maxSubscriptionBitrate)) return false;
        if (!Util.equals(maxTracks, that.maxTracks)) return false;
        if (mode != that.mode) return false;
        if (!renderDimensions.equals(that.renderDimensions)) return false;
        return trackSwitchOffMode == that.trackSwitchOffMode;
    }

    @Override
    public int hashCode() {
        int result = dominantSpeakerPriority != null ? dominantSpeakerPriority.hashCode() : 0;
        result =
                31 * result
                        + (maxSubscriptionBitrate != null ? maxSubscriptionBitrate.hashCode() : 0);
        result = 31 * result + (maxTracks != null ? maxTracks.hashCode() : 0);
        result = 31 * result + (mode != null ? mode.hashCode() : 0);
        result = 31 * result + renderDimensions.hashCode();
        result = 31 * result + (trackSwitchOffMode != null ? trackSwitchOffMode.hashCode() : 0);
        return result;
    }

    /** Builder class for {@link VideoBandwidthProfileOptions}. */
    public static class Builder {
        @Nullable private TrackPriority dominantSpeakerPriority;
        @Nullable private Long maxSubscriptionBitrate = 2400L;
        @Nullable private Long maxTracks;
        @Nullable private BandwidthProfileMode mode;
        private Map<TrackPriority, VideoDimensions> renderDimensions = Collections.emptyMap();
        @Nullable private TrackSwitchOffMode trackSwitchOffMode;

        /**
         * Set the minimum {@link TrackPriority} of the Dominant Speaker's {@link
         * RemoteVideoTrack}s. This property is {@code null} by default and can be set to {@code
         * null}. A {@code null} value results in the Twilio Media Server selecting the Dominant
         * Speaker's priority. This property will only be enabled if the Dominant Speaker feature is
         * enabled via {@link ConnectOptions.Builder#enableDominantSpeaker(boolean)}.
         *
         * @param priority The track priority to use or {@code null}.
         */
        public Builder dominantSpeakerPriority(@Nullable TrackPriority priority) {
            this.dominantSpeakerPriority = priority;
            return this;
        }

        /**
         * Set the maximum downlink video bandwidth in Kilobits per second (Kbps). This property is
         * 2,400 Kbps by default and can be set to {@code null}. A {@code null} value results in the
         * maximum allowed bitrate for a {@link Room}.
         *
         * @param maxSubscriptionBitrate The maximum bitrate for subscribe {@link RemoteVideoTrack}s
         *     or {@code null}.
         */
        public Builder maxSubscriptionBitrate(@Nullable Long maxSubscriptionBitrate) {
            this.maxSubscriptionBitrate = maxSubscriptionBitrate;
            return this;
        }

        /**
         * Set the maximum number of visible {@link RemoteVideoTrack}s selected first using {@link
         * TrackPriority} and second using an N-Loudest policy. This property is {@code null} by
         * default and can be set to {@code null}. A {@code null} value results in no limits on the
         * number of visible {@link RemoteVideoTrack}s.
         *
         * @param maxTracks The maximum number of video tracks or {@code null}.
         */
        public Builder maxTracks(@Nullable Long maxTracks) {
            this.maxTracks = maxTracks;
            return this;
        }

        /**
         * Set how {@link RemoteVideoTrack}s {@link TrackPriority} values are mapped to bandwidth
         * allocation in Group Rooms. This property is {@code null} by default and can be set to
         * {@code null}. A {@code null} value results in the Twilio Media Server selecting the mode.
         *
         * @param mode The {@link BandwidthProfileMode} to use or {@code null}.
         */
        public Builder mode(@Nullable BandwidthProfileMode mode) {
            this.mode = mode;
            return this;
        }

        /**
         * Set the desired render dimensions of {@link RemoteVideoTrack}s based on {@link
         * TrackPriority} and the {@link RemoteVideoTrack}s of the Dominant Speaker. The bandwidth
         * allocation algorithm will distribute the available downlink bandwidth proportional to the
         * requested render dimensions.
         *
         * @param dimensions A map containing at most one entry for each {@link TrackPriority}.
         */
        public Builder renderDimensions(@NonNull Map<TrackPriority, VideoDimensions> dimensions) {
            Preconditions.checkNotNull(dimensions, "render dimensions must not be null");
            Preconditions.checkArgument(
                    !dimensions.containsKey(null),
                    "render dimensions cannot contain a null track priority key");
            Preconditions.checkArgument(
                    !dimensions.containsValue(null),
                    "render dimensions cannot contain a null video dimensions value");
            this.renderDimensions = dimensions;
            return this;
        }

        /**
         * Configure how {@link RemoteVideoTrack}s are switched off. This property is {@code null}
         * by default and can be set to {@code null}. A {@code null} value results in the Twilio
         * Media Server selecting the track switch off mode.
         *
         * @param mode The {@link TrackSwitchOffMode} to use or {@code null}.
         */
        public Builder trackSwitchOffMode(@Nullable TrackSwitchOffMode mode) {
            this.trackSwitchOffMode = mode;
            return this;
        }

        public VideoBandwidthProfileOptions build() {
            return new VideoBandwidthProfileOptions(this);
        }
    }
}
