/*
 * Copyright (C) 2017 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Handler;
import android.util.Pair;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import tvi.webrtc.voiceengine.WebRtcAudioManager;

/**
 * A Room represents a media session with zero or more remote Participants. Media shared by any one
 * {@link RemoteParticipant} is distributed equally to all other Participants.
 *
 * <p>A subset of Android devices provide an <a
 * href="https://www.khronos.org/opensles/">OpenSLES</a> implementation. Although more efficient,
 * the use of OpenSLES occasionally results in echo. As a result, the Video SDK disables OpenSLES by
 * default unless explicitly enabled. To enable OpenSLES, execute the following before invoking
 * {@link Video#connect(Context, ConnectOptions, Listener)}
 *
 * <p>{@code tvi.webrtc.voiceengine.WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(false)}
 *
 * <p>
 */
public class Room {
    private static final Logger logger = Logger.getLogger(Room.class);

    /*
     * The following fields are statically initialized so the SDK only pays a one time cost of
     * fetching the private members of WebRtcAudioManager. These fields are used to apply the
     * OpenSLES configuration policy.
     */
    private static final Field blacklistDeviceForOpenSLESUsageField;
    private static final Field blacklistDeviceForOpenSLESUsageIsOverriddenField;

    static {
        try {
            blacklistDeviceForOpenSLESUsageField =
                    WebRtcAudioManager.class.getDeclaredField("blacklistDeviceForOpenSLESUsage");
            blacklistDeviceForOpenSLESUsageIsOverriddenField =
                    WebRtcAudioManager.class.getDeclaredField(
                            "blacklistDeviceForOpenSLESUsageIsOverridden");
            blacklistDeviceForOpenSLESUsageField.setAccessible(true);
            blacklistDeviceForOpenSLESUsageIsOverriddenField.setAccessible(true);
        } catch (NoSuchFieldException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    private long nativeRoomDelegate;
    private Context context;
    private String name;
    private String sid;
    private String mediaRegion = null;
    private Room.State state;
    private Map<String, RemoteParticipant> participantMap = new HashMap<>();
    private LocalParticipant localParticipant;
    private RemoteParticipant dominantSpeaker;
    private final Room.Listener listener;
    private final Handler handler;
    private Queue<Pair<Handler, StatsListener>> statsListenersQueue;
    private MediaFactory mediaFactory;

    /*
     * The contract for Room JNI callbacks is as follows:
     *
     * 1. All event callbacks are done on the same thread the developer used to connect to a room.
     * 2. Create and release all native memory on the same thread. In the case of a Room, the
     * RoomDelegate is created and released on the developer thread and the native room, room
     * observer, local participant, and participants are created and released on notifier thread.
     * 3. All Room fields must be mutated on the developer's thread.
     *
     * Not abiding by this contract, may result in difficult to debug JNI crashes,
     * incorrect return values in the synchronous API methods, or missed callbacks.
     */
    private final Room.Listener roomListenerProxy =
            new Room.Listener() {
                @Override
                @AccessedByNative
                public void onConnected(@NonNull final Room room) {
                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onConnected()");
                                Room.this.listener.onConnected(room);
                            });
                }

                @Override
                @AccessedByNative
                public void onConnectFailure(
                        @NonNull final Room room, @NonNull final TwilioException twilioException) {
                    // Release native room
                    releaseRoom();

                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onConnectFailure()");

                                // Update room state
                                Room.this.state = Room.State.DISCONNECTED;

                                // Release native room delegate
                                release();

                                // Notify developer
                                Room.this.listener.onConnectFailure(room, twilioException);
                            });
                }

                @Override
                @AccessedByNative
                public void onReconnecting(
                        @NonNull Room room, @NonNull TwilioException twilioException) {
                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onReconnecting()");

                                // Update room state
                                Room.this.state = State.RECONNECTING;

                                // Notify developer
                                Room.this.listener.onReconnecting(room, twilioException);
                            });
                }

                @Override
                @AccessedByNative
                public void onReconnected(@NonNull Room room) {
                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onReconnected()");

                                // Notify developer
                                Room.this.state = State.CONNECTED;
                                Room.this.listener.onReconnected(room);
                            });
                }

                @Override
                @AccessedByNative
                public void onDisconnected(
                        @NonNull final Room room, @Nullable final TwilioException twilioException) {
                    // Release native room
                    releaseRoom();

                    // Ensure the local participant is released if the disconnect was issued by the
                    // core
                    if (localParticipant != null) {
                        localParticipant.release();
                    }

                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onDisconnected()");

                                // Update room state
                                Room.this.state = Room.State.DISCONNECTED;

                                // Release native room delegate
                                release();

                                // Notify developer
                                Room.this.listener.onDisconnected(room, twilioException);
                            });
                }

                @Override
                @AccessedByNative
                public void onParticipantConnected(
                        @NonNull final Room room, @NonNull final RemoteParticipant participant) {
                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onParticipantConnected()");

                                // Update participants
                                participantMap.put(participant.getSid(), participant);

                                // Notify developer
                                Room.this.listener.onParticipantConnected(room, participant);
                            });
                }

                @Override
                @AccessedByNative
                public void onParticipantDisconnected(
                        @NonNull final Room room,
                        @NonNull final RemoteParticipant remoteParticipant) {
                    // Release participant
                    remoteParticipant.release();

                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onParticipantDisconnected()");

                                // Update participants
                                participantMap.remove(remoteParticipant.getSid());

                                // Notify developer
                                Room.this.listener.onParticipantDisconnected(
                                        room, remoteParticipant);
                            });
                }

                @Override
                @AccessedByNative
                public void onDominantSpeakerChanged(
                        @NonNull Room room, @Nullable RemoteParticipant remoteParticipant) {
                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onDominantSpeakerChanged()");

                                // Update room state
                                Room.this.dominantSpeaker = remoteParticipant;

                                // Notify developer
                                Room.this.listener.onDominantSpeakerChanged(
                                        room, remoteParticipant);
                            });
                }

                @Override
                @AccessedByNative
                public void onRecordingStarted(@NonNull final Room room) {
                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onRecordingStarted()");
                                Room.this.listener.onRecordingStarted(room);
                            });
                }

                @Override
                @AccessedByNative
                public void onRecordingStopped(@NonNull final Room room) {
                    handler.post(
                            () -> {
                                ThreadChecker.checkIsValidThread(handler);
                                logger.d("onRecordingStopped()");
                                Room.this.listener.onRecordingStopped(room);
                            });
                }
            };
    private final StatsListener statsListenerProxy =
            new StatsListener() {
                @Override
                @AccessedByNative
                public void onStats(@NonNull List<StatsReport> statsReports) {
                    final Pair<Handler, StatsListener> statsPair =
                            Room.this.statsListenersQueue.poll();
                    if (statsPair != null) {
                        statsPair.first.post(() -> statsPair.second.onStats(statsReports));
                    }
                }
            };

    Room(
            @NonNull Context context,
            @NonNull String name,
            @NonNull Handler handler,
            @NonNull Listener listener) {
        this.context = context;
        this.name = name;
        this.sid = "";
        this.state = Room.State.DISCONNECTED;
        this.listener = listener;
        this.handler = handler;
        this.statsListenersQueue = new ConcurrentLinkedQueue<>();
        configureOpenSLES();
    }

    /**
     * Returns the name of the current room. This method will return the SID if the room was created
     * without a name.
     */
    @NonNull
    public String getName() {
        return name;
    }

    /** Returns the SID of the current room. */
    @NonNull
    public String getSid() {
        return sid;
    }

    /**
     * Returns the region where media is processed. This property is set in Group Rooms by the time
     * the {@link Room} reaches {@link State#CONNECTED}. This method returns {@code null} under the
     * following conditions:
     *
     * <p>
     *
     * <ul>
     *   <li>The {@link Room} has not reached the {@link Room.State#CONNECTED} state.
     *   <li>The instance represents a peer-to-peer Room.
     * </ul>
     */
    @Nullable
    public String getMediaRegion() {
        return mediaRegion;
    }

    /** Returns the current room state. */
    @NonNull
    public synchronized Room.State getState() {
        return state;
    }

    /**
     * Returns the dominant speaker of the {@link Room}.
     *
     * <p>To enable this feature, add an invocation of {@link
     * ConnectOptions.Builder#enableDominantSpeaker} with {@code true} when building your {@link
     * ConnectOptions}. This method returns {@code null} when one of the following conditions are
     * true:
     *
     * <p>
     *
     * <ul>
     *   <li>The {@link Room} topology is P2P.
     *   <li>The dominant speaker feature was not enabled via {@link
     *       ConnectOptions.Builder#enableDominantSpeaker}.
     *   <li>There is currently no dominant speaker.
     * </ul>
     */
    @Nullable
    public RemoteParticipant getDominantSpeaker() {
        return dominantSpeaker;
    }

    /** Returns whether any media in the Room is being recorded. */
    public synchronized boolean isRecording() {
        return state == Room.State.CONNECTED && nativeIsRecording(nativeRoomDelegate);
    }

    /**
     * Returns all currently connected participants.
     *
     * @return list of participants.
     */
    @NonNull
    public synchronized List<RemoteParticipant> getRemoteParticipants() {
        return new ArrayList<>(participantMap.values());
    }

    /**
     * Returns the current local participant. If the room has not reached {@link
     * Room.State#CONNECTED} then this method will return null.
     */
    @Nullable
    public synchronized LocalParticipant getLocalParticipant() {
        return localParticipant;
    }

    /**
     * Retrieve stats for all media tracks and notify {@link StatsListener} via calling thread. In
     * case where room is in {@link Room.State#DISCONNECTED} state, reports won't be delivered.
     *
     * @param statsListener listener that receives stats reports for all media tracks.
     */
    public synchronized void getStats(@NonNull StatsListener statsListener) {
        Preconditions.checkNotNull(statsListener, "StatsListener must not be null");
        if (state == Room.State.DISCONNECTED) {
            return;
        }
        statsListenersQueue.offer(new Pair<>(Util.createCallbackHandler(), statsListener));
        nativeGetStats(nativeRoomDelegate);
    }

    /** Disconnects from the room. */
    public synchronized void disconnect() {
        if (state != Room.State.DISCONNECTED && nativeRoomDelegate != 0) {
            if (localParticipant != null) {
                localParticipant.release();
            }
            nativeDisconnect(nativeRoomDelegate);
        }
    }

    void onNetworkChanged(Video.NetworkChangeEvent networkChangeEvent) {
        if (nativeRoomDelegate != 0) {
            nativeOnNetworkChange(nativeRoomDelegate, networkChangeEvent);
        }
    }

    /*
     * We need to synchronize access to room listener during initialization and make
     * sure that onConnect() callback won't get called before connect() exits and Room
     * creation is fully completed.
     */
    @SuppressLint("RestrictedApi")
    void connect(@NonNull final ConnectOptions connectOptions) {
        // Check if audio or video tracks have been released
        ConnectOptions.checkAudioTracksReleased(connectOptions.getAudioTracks());
        ConnectOptions.checkVideoTracksReleased(connectOptions.getVideoTracks());

        synchronized (roomListenerProxy) {
            /*
             * Tests are allowed to provide a test MediaFactory to simulate media scenarios on the
             * same device.
             */
            mediaFactory =
                    (connectOptions.getMediaFactory() == null)
                            ? MediaFactory.instance(this, context)
                            : connectOptions.getMediaFactory();
            nativeRoomDelegate =
                    nativeConnect(
                            connectOptions,
                            roomListenerProxy,
                            statsListenerProxy,
                            mediaFactory.getNativeMediaFactoryHandle(),
                            handler);
            state = Room.State.CONNECTING;
        }
    }

    /*
     * Disable OpenSLES unless the developer has explicitly requested to enabled it via
     * WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(false).
     */
    @VisibleForTesting
    static void configureOpenSLES() {
        if (!openSLESEnabled()) {
            WebRtcAudioManager.setBlacklistDeviceForOpenSLESUsage(true);
        }
    }

    @VisibleForTesting
    static boolean openSLESEnabled() {
        try {
            return !(boolean) blacklistDeviceForOpenSLESUsageField.get(null)
                    && (boolean) blacklistDeviceForOpenSLESUsageIsOverriddenField.get(null);
        } catch (Exception e) {
            throw new RuntimeException("Failed to determine if OpenSLES is enabled.");
        }
    }

    /*
     * Called by JNI layer to finalize Room state after connected.
     */
    @SuppressWarnings("unused")
    @AccessedByNative
    private synchronized void setConnected(
            String roomSid,
            String mediaRegion,
            LocalParticipant localParticipant,
            List<RemoteParticipant> remoteParticipants) {
        logger.d("setConnected()");
        this.sid = roomSid;
        this.mediaRegion = mediaRegion.isEmpty() ? null : mediaRegion;
        if (this.name == null || this.name.isEmpty()) {
            this.name = roomSid;
        }
        this.localParticipant = localParticipant;
        for (RemoteParticipant remoteParticipant : remoteParticipants) {
            participantMap.put(remoteParticipant.getSid(), remoteParticipant);
        }
        this.state = Room.State.CONNECTED;
    }

    /*
     * Release all native Room memory in notifier thread and before invoking
     * onDisconnected callback for the following reasons:
     *
     * 1. Ensures that native WebRTC tracks are not null when removing remote sinks.
     * 2. Protects developers from potentially referencing deleted WebRTC tracks in
     *    onDisconnected callback.
     *
     * See GSDK-1007, GSDK-1079, and GSDK-1043 for more details.
     *
     * Thread validation is performed in RoomDelegate.
     */
    private synchronized void releaseRoom() {
        if (nativeRoomDelegate != 0) {
            for (RemoteParticipant remoteParticipant : participantMap.values()) {
                remoteParticipant.release();
            }
            nativeReleaseRoom(nativeRoomDelegate);
            cleanupStatsListenerQueue();
        }
    }

    /*
     * Release the native RoomDelegate from developer thread once the native Room memory
     * has been released.
     */
    private synchronized void release() {
        ThreadChecker.checkIsValidThread(handler);

        if (nativeRoomDelegate != 0) {
            nativeRelease(nativeRoomDelegate);
            nativeRoomDelegate = 0;
            mediaFactory.release(this);
        }
    }

    private void cleanupStatsListenerQueue() {
        for (final Pair<Handler, StatsListener> listenerPair : statsListenersQueue) {
            listenerPair.first.post(
                    () -> listenerPair.second.onStats(new ArrayList<StatsReport>()));
        }
        statsListenersQueue.clear();
    }

    /** Listener definition of room related events. */
    public interface Listener {
        /**
         * Called when a room has succeeded.
         *
         * @param room the connected room.
         */
        void onConnected(@NonNull Room room);

        /**
         * Called when a connection to a room failed.
         *
         * @param room the room that failed to be connected to.
         * @param twilioException an exception describing why connect failed.
         */
        void onConnectFailure(@NonNull Room room, @NonNull TwilioException twilioException);

        /**
         * Called when the {@link LocalParticipant} has experienced a network disruption and the
         * client begins trying to reestablish a connection to a room.
         *
         * <p>The SDK groups network disruptions into two categories: signaling and media. The
         * occurrence of either of these network disruptions will result in the onReconnecting
         * callback. During a media reconnection signaling related methods may continue to be
         * invoked.
         *
         * @param room the room the {@link LocalParticipant} is attempting to reconnect to.
         * @param twilioException An error explaining why the {@link LocalParticipant} is
         *     reconnecting to a room. Errors are limited to {@link
         *     TwilioException#SIGNALING_CONNECTION_DISCONNECTED_EXCEPTION} and {@link
         *     TwilioException#MEDIA_CONNECTION_ERROR_EXCEPTION}.
         */
        void onReconnecting(@NonNull Room room, @NonNull TwilioException twilioException);

        /**
         * Called after the {@link LocalParticipant} reconnects to a room after a network
         * disruption.
         *
         * @param room the room that was reconnected.
         */
        void onReconnected(@NonNull Room room);

        /**
         * Called when a room has been disconnected from.
         *
         * @param room the room that was disconnected from.
         * @param twilioException An exception if there was a problem that caused the room to be
         *     disconnected from. This value will be null is there were no problems disconnecting
         *     from the room.
         */
        void onDisconnected(@NonNull Room room, @Nullable TwilioException twilioException);

        /**
         * Called when a participant has connected to a room.
         *
         * @param room the room the participant connected to.
         * @param remoteParticipant the newly connected participant.
         */
        void onParticipantConnected(
                @NonNull Room room, @NonNull RemoteParticipant remoteParticipant);

        /**
         * Called when a participant has disconnected from a room. The disconnected participant's
         * audio and video tracks will still be available in their last known state. Video tracks
         * sinks are removed when a participant is disconnected.
         *
         * @param room the room the participant disconnected from.
         * @param remoteParticipant the disconnected participant.
         */
        void onParticipantDisconnected(
                @NonNull Room room, @NonNull RemoteParticipant remoteParticipant);

        /**
         * This method is called when the dominant speaker in the {@link Room} changes. Either there
         * is a new dominant speaker, in which case {@link Room#getDominantSpeaker} returns the
         * {@link RemoteParticipant} included in the event or there is no longer a dominant speaker,
         * in which case {@link Room#getDominantSpeaker} returns {@code null}.
         *
         * <p>This method will not be called when one of the following conditions are true:
         *
         * <p>
         *
         * <ul>
         *   <li>The {@link Room} topology is P2P.
         *   <li>The dominant speaker feature was not enabled via {@link
         *       ConnectOptions.Builder#enableDominantSpeaker(boolean)}
         * </ul>
         *
         * @param room The {@link Room} in which the dominant speaker changed.
         * @param remoteParticipant The {@link RemoteParticipant} that is currently the dominant
         *     speaker or {@code null}.
         */
        default void onDominantSpeakerChanged(
                @NonNull Room room, @Nullable RemoteParticipant remoteParticipant) {
            logger.d("onDominantSpeakerChanged");
        }

        /**
         * This method is only called when a {@link Room} which was not previously recording starts
         * recording. If you've joined a {@link Room} which is already recording this event will not
         * be fired.
         *
         * @param room
         */
        void onRecordingStarted(@NonNull Room room);

        /**
         * This method is only called when a {@link Room} which was previously recording stops
         * recording. If you've joined a {@link Room} which is not recording this event will not be
         * fired.
         *
         * @param room
         */
        void onRecordingStopped(@NonNull Room room);
    }

    /** Represents the current state of a {@link Room}. */
    public enum State {
        CONNECTING,
        CONNECTED,
        RECONNECTING,
        DISCONNECTED
    }

    private native long nativeConnect(
            ConnectOptions ConnectOptions,
            Listener listenerProxy,
            StatsListener statsListenerProxy,
            long nativeMediaFactoryHandle,
            Handler handler);

    private native boolean nativeIsRecording(long nativeRoomDelegate);

    private native void nativeGetStats(long nativeRoomDelegate);

    private native void nativeOnNetworkChange(
            long nativeRoomDelegate, Video.NetworkChangeEvent networkChangeEvent);

    private native void nativeDisconnect(long nativeRoomDelegate);

    private native void nativeReleaseRoom(long nativeRoomDelegate);

    private native void nativeRelease(long nativeRoomDelegate);
}
