package com.tinkerpatch.sdk.okhttp3;

import com.tencent.tinker.lib.util.TinkerLog;
import com.tencent.tinker.loader.shareutil.SharePatchFileUtil;
import com.tinkerpatch.sdk.server.model.DataFetcher;
import com.tinkerpatch.sdk.server.model.TinkerClientUrl;
import com.tinkerpatch.sdk.util.ContentLengthInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import okhttp3.Call;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;

/**
 * Created by simsun{github.com/simpleton}.
 */

public class OkHttp3StreamFetcher implements DataFetcher<InputStream> {

    private static final String    TAG           = "Tinker.OkHttp3Fetcher";
    private static       MediaType mediaTypeForm = MediaType.parse("application/x-www-form-urlencoded; charset=utf-8");

    private final TinkerClientUrl tkUrl;
    private OkHttpClient    client;

    private ResponseBody responseBody;
    private InputStream stream;

    /**
     * @param client okhttp3 instance
     * @param tkUrl tinkerpatch request url
     */
    public OkHttp3StreamFetcher(OkHttpClient client, TinkerClientUrl tkUrl) {
        this.client = client;
        this.tkUrl = tkUrl;
    }

    @Override
    public void loadData(final DataCallback<? super InputStream> callback) {
        Request.Builder requestBuilder = new Request.Builder().url(tkUrl.getStringUrl());
        switch (tkUrl.getMethod()) {
            case "GET":
                requestBuilder = requestBuilder.get();
                break;
            case "POST":
                requestBuilder = requestBuilder.post(RequestBody.create(mediaTypeForm, tkUrl.getBody()));
                break;
            default:
                throw new RuntimeException("Unsupported request Method" + tkUrl.getMethod());
        }
        for (Map.Entry<String, String> headerEntry : tkUrl.getHeaders().entrySet()) {
            String key = headerEntry.getKey();
            requestBuilder.addHeader(key, headerEntry.getValue());
        }
        final Request request = requestBuilder.build();

        Call call = client.newCall(request);
        call.enqueue(new okhttp3.Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                TinkerLog.w(TAG, "Okhttp3 failed to obtain result", e);
                callback.onLoadFailed(e);
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                responseBody = response.body();
                if (responseBody != null && response.isSuccessful()) {
                    long contentLength = responseBody.contentLength();
                    TinkerLog.d(TAG, "Okhttp3 got success response: " + response.code() + ", " + response.message());
                    stream = ContentLengthInputStream.obtain(responseBody.byteStream(), contentLength);
                    callback.onDataReady(stream);
                } else {
                    TinkerLog.w(TAG, "Okhttp3 got error response: " + response.code() + ", " + response.message());
                    callback.onLoadFailed(new RuntimeException(String.format("Okhttp3 error:%s", response.message())));
                }
            }
        });
    }

    @Override
    public void cleanup() {
        this.client = null;
        SharePatchFileUtil.closeQuietly(stream);
        SharePatchFileUtil.closeQuietly(responseBody);
    }
}
