/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2016-2017 Shengjie Sim Sun
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package com.tinkerpatch.sdk.build.gradle

import com.tinkerpatch.sdk.build.gradle.extension.ProductFlavors
import com.tinkerpatch.sdk.build.gradle.extension.TinkerPatchSupportExtension
import com.tinkerpatch.sdk.build.gradle.task.TinkerPatchBackupFile
import com.tinkerpatch.sdk.build.gradle.task.TinkerPatchManifestTask
import com.tinkerpatch.sdk.build.gradle.utils.Utils
import org.gradle.api.GradleException
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.Task

/**
 * Registers the plugin's tasks.
 *
 */

class TinkerPatchSupportPlugin implements Plugin<Project> {

    @Override
    public void apply(Project project) {
        project.extensions.create(
                'tinkerpatchSupport', TinkerPatchSupportExtension
        )
        project.tinkerpatchSupport.extensions.create(
                'productFlavors', ProductFlavors, project
        )

        if (!project.plugins.hasPlugin('com.android.application')) {
            throw new GradleException('generateTinkerApk: Android Application plugin required')
        }
        project.afterEvaluate {
            setTinkerConfig(project)
        }
        //然后这里才执行tinker plugin的afterEvaluate
        project.apply plugin: 'com.tencent.tinker.patch'

        //这里的afterEvaluate 需要在tinker plugin的afterEvaluate 之后运行
        project.afterEvaluate {
            if (!project.extensions.tinkerpatchSupport.tinkerEnable) {
                project.logger.warn("Tinker is disabled, ignore invoke");
                return
            }

            def date = new Date().format("MMdd-HH-mm-ss")
            def android = project.extensions.android
            def configuration = project.extensions.tinkerpatchSupport

            List<String> flavors = new ArrayList<>();
            project.android.productFlavors.each {flavor ->
                flavors.add(flavor.name)
            }
            boolean hasFlavors = flavors.size() > 0
            if (hasFlavors) {
                def flavorReleaseTask = project.tasks.create("tinkerPatchAllFlavorsRelease")
                flavorReleaseTask.group = "tinker"

                def flavorDebugTask = project.tasks.create("tinkerPatchAllFlavorsDebug")
                flavorDebugTask.group = "tinker"

                for (String flavor : flavors) {
                    def tinkerReleaseTask = project.tasks.findByName("tinkerPatch${flavor.capitalize()}Release")
                    if(tinkerReleaseTask != null) {
                        flavorReleaseTask.dependsOn tinkerReleaseTask
                    }

                    def tinkerDebugTask = project.tasks.findByName("tinkerPatch${flavor.capitalize()}Debug")
                    if (tinkerDebugTask != null) {
                        flavorDebugTask.dependsOn tinkerDebugTask
                    }
                }
            }

            android.applicationVariants.all { variant ->
                def variantOutput = variant.outputs.first()
                def variantName = variant.name.capitalize()
                //processFlavor1DebugManifest
                def preAssembleTask = project.tasks.findByName("process${variantName}Manifest")
                if (hasFlavors) {
                    preAssembleTask.doFirst {
                        upgradeTinkerIdByFlavor(project, variant.flavorName)
                    }
                }

                TinkerPatchManifestTask manifestTask = project.tasks.create("tinkerpatchSupportProcess${variantName}Manifest", TinkerPatchManifestTask)


                if (variantOutput.processManifest.properties['manifestOutputFile'] != null) {
                    manifestTask.manifestPath = variantOutput.processManifest.manifestOutputFile
                } else if (variantOutput.processResources.properties['manifestFile'] != null) {
                    manifestTask.manifestPath = variantOutput.processResources.manifestFile
                }
                manifestTask.mustRunAfter variantOutput.processManifest

                def tinkerManifestTask = getTinkerManifestTask(project, variantName)
                if (tinkerManifestTask == null) {
                    throw new GradleException("can't find tinkerProcess${variantName}Manifest, you must init tinker plugin first!")
                }
                tinkerManifestTask.mustRunAfter manifestTask
                variantOutput.processResources.dependsOn manifestTask

                if (!Utils.isNullOrNil(configuration.autoBackupApkPath)) {
                    TinkerPatchBackupFile backupTask = new TinkerPatchBackupFile(
                            project : project,
                            variant : variant,
                            buildTime: date,
                    )
                    variant.assemble.doLast {
                        backupTask.backupFile()
                    }
                }
            }
        }
    }

    private void setTinkerConfig(Project project) {
        def tinkerConfiguration = project.extensions.tinkerPatch
        def configuration = project.extensions.tinkerpatchSupport
        configuration.trim()

        tinkerConfiguration.tinkerEnable = configuration.tinkerEnable
        tinkerConfiguration.buildConfig.tinkerId = configuration.appVersion
        tinkerConfiguration.buildConfig.isProtectedApp = configuration.protectedApp
        tinkerConfiguration.buildConfig.supportHotplugComponent = configuration.supportComponent

        if (!Utils.isNullOrNil(configuration.baseProguardMappingFile)) {
            tinkerConfiguration.buildConfig.applyMapping = configuration.baseProguardMappingFile
        }
        if (!Utils.isNullOrNil(configuration.baseResourceRFile)) {
            tinkerConfiguration.buildConfig.applyResourceMapping = configuration.baseResourceRFile
        }
        if (!Utils.isNullOrNil(configuration.baseApkFile)) {
            tinkerConfiguration.oldApk = configuration.baseApkFile
        }
    }

    static Task getTinkerManifestTask(Project project, String variantName) {
        String tinkerManifestTask = "tinkerProcess${variantName}Manifest"
        return project.tasks.findByName(tinkerManifestTask)
    }

    private void upgradeTinkerIdByFlavor(Project project, String flavorName) {
        TinkerPatchSupportExtension configuration = project.extensions.tinkerpatchSupport
        Boolean.parseBoolean()

        if (!Utils.isNullOrNil(flavorName)) {
            def findedFlavor = configuration.productFlavors.flavors.find {
                if (flavorName.equalsIgnoreCase(it.flavorName)) {
                    def tinkerConfiguration = project.extensions.tinkerPatch
                    //全部参数都是可选复写的
                    if (!Utils.isNullOrNil(it.appVersion)) {
                        tinkerConfiguration.buildConfig.tinkerId = it.appVersion
                    }
                    if (!Utils.isNullOrNil(it.baseProguardMappingFile)) {
                        tinkerConfiguration.buildConfig.applyMapping = it.baseProguardMappingFile
                    }
                    if (!Utils.isNullOrNil(it.baseResourceRFile)) {
                        tinkerConfiguration.buildConfig.applyResourceMapping = it.baseResourceRFile
                    }
                    if (!Utils.isNullOrNil(it.baseApkFile)) {
                        tinkerConfiguration.oldApk = it.baseApkFile
                    }
                    if (!Utils.isNullOrNil(it.protectedApp)) {
                        if (it.protectedApp.equalsIgnoreCase("true") || it.protectedApp.equalsIgnoreCase("false")) {
                            tinkerConfiguration.buildConfig.isProtectedApp = Boolean.parseBoolean(it.protectedApp)
                        }
                    }
                    return true
                }
            }
            if (!findedFlavor) {
                setTinkerConfig(project)
            }
        }
        if (Utils.isNullOrNil(project.extensions.tinkerPatch.buildConfig.tinkerId)) {
            throw new GradleException("Can\'t find flavorName: ${flavorName} in your productFlavors, " +
                    "you must add an flavor with flavorName=${flavorName}")
        }
    }
}