package com.tenqube.visual_third.repository;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.graphics.drawable.Drawable;
import android.text.TextUtils;

import com.tenqube.visual_third.db.DatabaseHelper;
import com.tenqube.visual_third.db.Db;
import com.tenqube.visual_third.db.dao.CardDao;
import com.tenqube.visual_third.db.dao.CategoryDao;
import com.tenqube.visual_third.db.dao.CurrencyDao;
import com.tenqube.visual_third.db.dao.NotificationDao;
import com.tenqube.visual_third.db.dao.TransactionDao;
import com.tenqube.visual_third.db.dao.UserCategoryDao;
import com.tenqube.visual_third.entity.Card;
import com.tenqube.visual_third.entity.Category;
import com.tenqube.visual_third.entity.Currency;
import com.tenqube.visual_third.entity.JoinedTransaction;
import com.tenqube.visual_third.entity.Transaction;
import com.tenqube.visual_third.entity.UserCategory;
import com.tenqube.visual_third.entity.VisualNotification;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.model.api.SyncCategoryResponse;
import com.tenqube.visual_third.model.js.Cards;
import com.tenqube.visual_third.model.js.CategoryInfo;
import com.tenqube.visual_third.model.js.InsertTransactionRequest;
import com.tenqube.visual_third.model.js.TransactionByIdsRequest;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.model.js.Transactions;
import com.tenqube.visual_third.model.js.UpdateTransactionRequest;
import com.tenqube.visual_third.model.ui.NotiCatch;

import java.util.ArrayList;
import java.util.Calendar;

import tenqube.parser.constants.Constants;

import static com.tenqube.visual_third.manager.PrefManager.NOTI_CHANNEL;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_ICON_RES;
import static com.tenqube.visual_third.manager.PrefManager.SDK_ENABLED;
import static com.tenqube.visual_third.manager.PrefManager.UID;
import static com.tenqube.visual_third.util.Mapper.toAnalysisTran;
import static com.tenqube.visual_third.util.Mapper.toApiTransactions;
import static com.tenqube.visual_third.util.Mapper.toJsCards;
import static com.tenqube.visual_third.util.Mapper.toJsCategories;
import static com.tenqube.visual_third.util.Mapper.toJsTransactions;
import static com.tenqube.visual_third.util.Mapper.toJsUserCategories;
import static com.tenqube.visual_third.util.Mapper.toSearchTransactionWithEntity;
import static com.tenqube.visual_third.util.Utils.isEmpty;

public class VisualRepository {

    private final Context context;
    private static VisualRepository mInstance;

    private final PrefManager prefManager;
    private final CardDao cardDao;
    private final CategoryDao categoryDao;
    private final TransactionDao transactionDao;
    private final UserCategoryDao userCategoryDao;
    private final CurrencyDao currencyDao;
    private final NotificationDao notificationDao;

    public static VisualRepository getInstance(Context context){
        synchronized (VisualRepository.class) {
            if(mInstance == null){
                mInstance = new VisualRepository(context);
            }
        }
        return mInstance;
    }

    private VisualRepository(Context context) {
        DatabaseHelper dbHelper = DatabaseHelper.getInstance(context);
        Db db = new Db(dbHelper.getWritableDatabase(), dbHelper.getReadableDatabase());

        this.context = context;
        prefManager = PrefManager.getInstance(context);
        this.cardDao = new CardDao(db);
        this.categoryDao = new CategoryDao(db);
        this.transactionDao = new TransactionDao(db);
        this.userCategoryDao = new UserCategoryDao(db);
        this.currencyDao = new CurrencyDao(db);
        this.notificationDao = new NotificationDao(db);
    }

    public ArrayList<NotiCatch> loadNotiCatchedApps() {
        String pkgs = "com.kftc.citismb,com.cu.sbank,com.keb.android.mbank,com.epost.psf.sdsi,com.wooribank.pib.smart,com.wooricard.smartapp,net.ib.android.smcard,com.hyundaicard.appcard,com.lcacApp,nh.smart.card,nh.smart,com.bccard.bcsmartapp,co.kr.kdb.android.smartkdb,com.kftc.citismb-49,com.shinhan.sbanking,com.shinhan.sbanking-116,com.citibank.cardapp,com.feelingk.dyfn.pushagent,com.nh.mobilenoti,com.kbstar.starpush,com.knb.bsp,com.shinhan.smartcaremg,com.IBK.SmartPush.app,com.hanabank.ebk.channel.android.hananbank,kr.co.bnkbank.push.customer,com.kbcard.kbkookmincard,com.wr.alrim,com.kftc.shsmb,com.smg.mgnoti,kr.co.youfirst.checkable,com.hanaskcard.app.touchstamp,com.samsung.android.spay,com.sc.danb.scbankapp,com.yuanta.tradarm,com.kbankwith.smartbank,com.hanaskcard.paycla,com.shinhan.smartcaremgr,kr.co.citibank.citimobile,com.shinhan.mobilebankbook,com.shinhaninvest.mts,kr.co.bsbank.mobilebank,com.epost.psf.ss,com.kbstar.liivkbcambodia,com.wooriwm.txsmart,com.wooriwm.mugsmart,com.skplanet.syrup.notisender,com.kakaobank.channel";
        ArrayList<NotiCatch> notiCatches = new ArrayList<>();
        PackageManager packageManager = context.getPackageManager();
        String[] apps = pkgs.split(",");
        for(String packageName : apps) {
            try {

                ApplicationInfo appInfo = packageManager.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
                Drawable icon = packageManager.getApplicationIcon(packageName);
                String appName = (String) packageManager.getApplicationLabel(appInfo);
                notiCatches.add(new NotiCatch(icon, appName));
            } catch (PackageManager.NameNotFoundException e) {
                e.printStackTrace();
            }
        }

        return notiCatches;
    }

    public Cursor getSum(String fromAt, String toAt, int groupBy) {
        return transactionDao.getSum(fromAt, toAt, groupBy);
    }

    public void settingNotification(int smallIcon, String channel) {
        prefManager.saveIntValue(NOTI_ICON_RES, smallIcon);
        prefManager.saveStringValue(NOTI_CHANNEL, channel);
    }

    public void setEnabled(boolean enabled) {
        prefManager.saveBoolean(SDK_ENABLED, enabled);
    }

    public boolean isExecutedBulk() {
        return prefManager.isEnabled(PrefManager.BULK_EXECUTED, false);
    }

    public void setDeepLink(String url) {
        prefManager.saveStringValue(PrefManager.DEEP_LINK, url);
    }

    public void setUserName(String userName) {
        prefManager.saveStringValue(PrefManager.USER_NAME, userName);
    }

    public boolean shouldSignUp() {
        return TextUtils.isEmpty(prefManager.loadStringValue(UID, ""));
    }

    public void saveSDKInfo(String apiKey, String qualifier) {
        prefManager.saveStringValue(PrefManager.API_KEY, apiKey);
        prefManager.saveStringValue(PrefManager.QUALIFIER, qualifier);
    }

    public void saveServerInfo(SignUpResponse response) {
        // login request succeed, new token generated
        prefManager.saveStringValue(PrefManager.SEARCH_URL, response.getResults().getSearch().getUrl());
        prefManager.saveStringValue(PrefManager.SEARCH_API_KEY, response.getResults().getSearch().getApiKey());
        prefManager.saveStringValue(UID, response.getResults().getAuthorization().getSdk());
        prefManager.saveLongValue(PrefManager.SIGN_UP_TIME, Calendar.getInstance().getTimeInMillis());
        if (response.getResults().getWeb() != null)
            prefManager.saveStringValue(PrefManager.WEB_URL, response.getResults().getWeb().getUrl());
    }

    public double getSum(String fromAt, String toAt) {
        return transactionDao.getSum(fromAt, toAt);
    }

    public boolean isActiveNoti(String name) {
        return notificationDao.isActiveNoti(name);
    }

    public void setActiveNoti(String name, boolean isActive) {
        notificationDao.setActiveNoti(name, isActive);
    }

    public void updateNotiHour(String name, int hour) {
        notificationDao.updateNotiHour(name, hour);
    }

    public ArrayList<VisualNotification> loadNotifications() {
        return notificationDao.loadNotifications();
    }

    public boolean shouldShowTranPopup() {
        return prefManager.isEnabled(PrefManager.TRAN_POPUP, true);
    }

    public void setTranPopup(boolean shouldShow) {
        prefManager.saveBoolean(PrefManager.TRAN_POPUP, shouldShow);
    }

    public Currency loadCurrencyInfo(String from, String to) {
        return currencyDao.loadCurrencyInfo(from, to);
    }

    public void mergeCurrency(String from, String to, float rate, String updatedAt) {
        currencyDao.mergeCurrency(new Currency(0, from, to, rate, updatedAt));
    }

    public void updateDeletedTransactionById(Integer[] ids) {
        transactionDao.updateDeletedTransactionById(ids);
    }

    public void deleteTransactions() {
        ArrayList<Integer> ids = transactionDao.loadDeletedTranIds();
        if(!isEmpty(ids))
            transactionDao.deleteTransactions(ids);
    }

    public void loadAll() {
//        cardDao.loadAll();
//        categoryDao.loadAll();
//        transactionDao.loadAll();
//        userCategoryDao.loadAll();
//        contentsDao.loadAll();
    }

    public int insertTransaction(InsertTransactionRequest insertTransactionRequest) {
        return transactionDao.insertTransaction(insertTransactionRequest);
    }

    public JoinedTransaction loadJoinedTransaction(int tranId) {
        return transactionDao.loadJoinedTransaction(tranId);
    }

    public int loadTranId(String identifier) {
        return transactionDao.loadTranId(identifier);
    }

    public Transaction loadApplyAllTran(String keyword) {
        if(TextUtils.isEmpty(keyword)) return  null;
        return transactionDao.loadApplyAllTran(keyword);
    }

    // load
    public ArrayList<Transactions.Transaction> loadTransactions(TransactionRequest transactionRequest) {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadJoinedTransactions(transactionRequest);
        return toJsTransactions(transactions);
    }

    public ArrayList<com.tenqube.visual_third.model.analysis.Transaction> loadAnalysisTransactions(TransactionRequest transactionRequest) {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadJoinedTransactions(transactionRequest);
        return toAnalysisTran(transactions);
    }

    public ArrayList<Transactions.Transaction> loadTransactions(TransactionByIdsRequest transactionRequest) {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadJoinedTransactions(transactionRequest);
        return toJsTransactions(transactions);
    }

    public com.tenqube.visual_third.model.api.TransactionRequest loadNotSyncedTransactions() {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadNotSyncedTransactions();
        if(!transactions.isEmpty())
            return new com.tenqube.visual_third.model.api.TransactionRequest(toApiTransactions(transactions));
        else
            return null;
    }

    public SearchCompanyRequest loadFailedSearchTransactions() {
        ArrayList<JoinedTransaction> transactions = transactionDao.loadFailedSearchTransactions();
        ArrayList<SearchCompanyRequest.Transaction> searchTransactions = toSearchTransactionWithEntity(transactions);
        if(!isEmpty(searchTransactions))
            return new SearchCompanyRequest(searchTransactions);
        else
            return null;
    }

    public ArrayList<Cards.Card> loadCards() {
        ArrayList<Card> cards = cardDao.loadCards();
        return toJsCards(cards);
    }

    public ArrayList<CategoryInfo.ServerCategory> loadCategories() {
        ArrayList<Category> categories = categoryDao.loadCategories();
        return toJsCategories(categories);
    }

    public Category loadCategory(String categoryCode) {
        return categoryDao.loadCategory(categoryCode);
    }

    public ArrayList<CategoryInfo.UserCategory> loadUserCategories() {
        ArrayList<UserCategory> categories = userCategoryDao.loadUserCategories();
        return toJsUserCategories(categories);
    }

    public void mergeCategory(ArrayList<SyncCategoryResponse.Category> categories) {

        try {
            for(SyncCategoryResponse.Category category : categories) {

                categoryDao.mergeCategory(category);
                // 새로운 카테고리 가 생김
                // 카테고리 아이디 추가

                // 기존 카테고리가 삭제됨
                // 내역 삭제
                // 사용자 카테고리 삭제

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public void updateSyncedTransactions(com.tenqube.visual_third.model.api.TransactionRequest transactionRequest) {
        ArrayList<String> ids = new ArrayList<>();
        ArrayList<com.tenqube.visual_third.model.api.TransactionRequest.Transaction> transactions = transactionRequest.getTransactions();

        for(com.tenqube.visual_third.model.api.TransactionRequest.Transaction tran : transactions) {
            ids.add(tran.getIdentifier());
        }
        transactionDao.updateSyncedTransactions(ids);
    }

    public void updateTransaction(UpdateTransactionRequest updateTransactionRequest) {
        transactionDao.updateTransaction(updateTransactionRequest);
    }

    public void updateRetrySearch(ArrayList<SearchCompanyRequest.Transaction> transactions, boolean shouldRetry) {
        ArrayList<String> ids = new ArrayList<>();

        for(SearchCompanyRequest.Transaction tran : transactions) {
            ids.add(tran.getIdentifier());
        }
        transactionDao.updateRetryTransactions(ids, shouldRetry);


    }

    public int loadCardId(String cardName, int cardType, int cardSubType) {
        int cardId = cardDao.getCardId(cardName, cardType, cardSubType);
        if(cardId == -1) {
            cardId = cardDao.insert(cardName, cardType, cardSubType);
        }
        return cardId;
    }


    /**
     * 파싱된 결과 로컬 디비에 저장하기
     * 1. 카드 아이디 구하기 없으면 저장
     * 2. 사용자 카테고리 아이디
     * 3. 내역 저장하기
     * @param transactions 파싱된 내역
     */
    public void mergeTransactions(ArrayList<com.tenqube.visual_third.model.parser.Transaction> transactions) {

        try {
            for(com.tenqube.visual_third.model.parser.Transaction parsedTran : transactions) {

                // getCardId
                int cardId = loadCardId(parsedTran.getParsedTransaction().cardName, parsedTran.getParsedTransaction().cardType, parsedTran.getParsedTransaction().cardSubType);
                parsedTran.setCardId(cardId);

                // getUserCateId
                int userCateId = userCategoryDao.getCategoryId(parsedTran.getParsedTransaction().dwType == Constants.DWType.DEPOSIT.ordinal() ?
                        "901010" : "101010");

                parsedTran.setUserCateId(userCateId);

                // merge transaction
                transactionDao.mergeTransaction(parsedTran);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }


    }

    /**
     * 검색 후 업데이트 할 부분
     * @param response 검색된 결과 값
     */
    public void updateTransactions(SearchCompanyResponse response) {

        try {
            // userCateId 구하기
            for(SearchCompanyResponse.TranCompany tranCompany : response.getResults()) {
                int userCateId = userCategoryDao.getCategoryId(tranCompany.getCategory().getRepCode());
                transactionDao.updateTransaction(tranCompany, userCateId);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    public void updateTransaction(SearchCompanyResponse.TranCompany tranCompany, boolean isAll) {

        try {
            // userCateId 구하기
            int userCateId = userCategoryDao.getCategoryId(tranCompany.getCategory().getRepCode());

            if(isAll) {
                transactionDao.updateTransactionByKeyword(tranCompany, userCateId);
            } else {
                transactionDao.updateTransaction(tranCompany, userCateId);

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

}
