package com.tenqube.visual_third.ui;

import android.content.Intent;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.DialogFragment;
import android.support.v4.app.Fragment;
import android.support.v4.widget.SwipeRefreshLayout;
import android.text.TextUtils;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.LinearLayout;

import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.VisualServiceImpl;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SignUpRequest;
import com.tenqube.visual_third.model.api.SignUpResponse;
import com.tenqube.visual_third.model.js.DateRequest;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.model.js.TimeRequest;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.web.ErrorImpl;
import com.tenqube.visual_third.web.LogImpl;
import com.tenqube.visual_third.web.RepoImpl;
import com.tenqube.visual_third.web.SystemImpl;
import com.tenqube.visual_third.web.UiImpl;
import com.tenqube.visual_third.web.ViewContractor;
import com.tenqube.visual_third.web.VisualInterface;

import java.util.Calendar;

import static android.app.Activity.RESULT_OK;
import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.manager.PrefManager.UID;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualActivityFragment extends Fragment implements TimePickerFragment.Callback, DatePickerFragment.Callback, ViewContractor {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();
    public static final String ARG_UID = "UID";

    private PrefManager prefManager;
    private VisualApi api;
    private String mFailingUrl;

    private WebView webview;

    private VisualInterface.Repo repoInterface;
    private VisualInterface.UI uiInterface;
    private VisualInterface.System systemInterface;
    private VisualInterface.Error errorInterface;
    private VisualInterface.Log logInterface;

    private SwipeRefreshLayout swipeRefreshLayout;

    private boolean isPageLoaded;

    private String uid;

    public VisualActivityFragment() {
        // Required empty public constructor
    }

    public static VisualActivityFragment newInstance(String uid) {

        VisualActivityFragment fragment = new VisualActivityFragment();
        Bundle bundle = new Bundle();
        bundle.putString(ARG_UID, uid);
        fragment.setArguments(bundle);
        return new VisualActivityFragment();
    }

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        if(getArguments() != null) {
            uid = getArguments().getString(ARG_UID, "");
        }
        api = VisualApi.getInstance(getActivity());
        prefManager = PrefManager.getInstance(getActivity());

        initFabric(getActivity());
        AnswerManager.onKeyMetric(new LogRequest("VisualActivityFragment"));
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_visual_web, container, false);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        try {
            swipeRefreshLayout = (SwipeRefreshLayout) view.findViewById(R.id.swipe_refresh_layout);
            swipeRefreshLayout.setEnabled(false);
            swipeRefreshLayout.setColorSchemeResources(
                    R.color.colorPopupRed,
                    R.color.colorPopupRed,
                    R.color.colorPopupRed);

            swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    swipeRefreshLayout.setRefreshing(false);
                    webview.reload();
                }
            });

            webview = (WebView) view.findViewById(R.id.visual_web_view);

            // javascriptInterface
            errorInterface = new ErrorImpl(getActivity(), this, webview);
            logInterface = new LogImpl(getActivity(), this, webview, errorInterface);
            repoInterface = new RepoImpl(getActivity(), this, webview, errorInterface,
                    VisualRepository.getInstance(getActivity()),
                    api,
                    new AnalysisServiceImpl(getActivity()));

            uiInterface = new UiImpl(getActivity(), this, webview, errorInterface, new BottomDialog(getActivity()));
            systemInterface = new SystemImpl(getActivity(), this, webview, errorInterface);

            webview.addJavascriptInterface(errorInterface, "visualError");
            webview.addJavascriptInterface(repoInterface, "visualRepo");
            webview.addJavascriptInterface(uiInterface, "visualUI");
            webview.addJavascriptInterface(systemInterface, "visualSystem");
            webview.addJavascriptInterface(logInterface, "visualLog");

            // settings
            webview.getSettings().setJavaScriptEnabled(true);
            webview.getSettings().setDomStorageEnabled(true);
            webview.getSettings().setDatabaseEnabled(true);

            // error
            webview.setWebViewClient(new WebViewClient() {
                public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
                    mFailingUrl = failingUrl;
                    loadErrorUrl();
                }
            });

            if(prefManager.loadStringValue(PrefManager.QUALIFIER, "dev").equals("dev"))
                webview.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);

            final LinearLayout bulkContainer = (LinearLayout) view.findViewById(R.id.start_bulk);

            loadUrl();

            bulkContainer.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                        signUp(uid, new OnResultListener() {
                            @Override
                            public void onResult(int signUpResult, String msg) {
                                if(signUpResult != Constants.SignupResult.FAIL) {

                                    loadUrl();

                                    try {
                                        VisualWebActivity.startActivity(getActivity(), "");
                                    } catch (AuthException e) {
                                        e.printStackTrace();
                                    }
                                } else {
                                    loadErrorUrl();
                                }
                            }
                        });
                }
            });

        } catch(Exception e) {
            getActivity().finish();
        }

    }

    private void loadUrl() {
        String url = String.format("http://%1$s.richnco-view.tenqube.kr/%2$s/",
                prefManager.loadStringValue(PrefManager.QUALIFIER, "dev"),
                TextUtils.isEmpty(prefManager.loadStringValue(PrefManager.UID, "")) ? "intro" : "landing");

        webview.loadUrl(url);
    }

    private void signUp(@NonNull final String uid, @NonNull final OnResultListener onResultListener) {
        LOGI(TAG, "가입", mIsDebug);

            final SignUpRequest signUpRequest = new SignUpRequest(uid);

            if(!TextUtils.isEmpty(prefManager.loadStringValue(UID, ""))) {
                onResultListener.onResult(Constants.SignupResult.ALREADY_JOINED, "already joined");
                return;
            }

            AnswerManager.onKeyMetric(new LogRequest("signUp"));
            new Thread(new Runnable() {
                @Override
                public void run() {

                    try {
                        SignUpResponse response = api.signUp(signUpRequest);

                        if (response == null) {
                            onResultListener.onResult(Constants.SignupResult.FAIL, "network error");
                        } else {

                            // login request succeed, new token generated
                            prefManager.saveStringValue(PrefManager.SEARCH_URL, response.getResults().getSearch().getUrl());
                            prefManager.saveStringValue(PrefManager.SEARCH_API_KEY, response.getResults().getSearch().getApiKey());
                            prefManager.saveStringValue(UID, response.getResults().getAuthorization().getSdk());
                            prefManager.saveLongValue(PrefManager.SIGN_UP_TIME, Calendar.getInstance().getTimeInMillis());
                            if (response.getResults().getWeb() != null)
                                prefManager.saveStringValue(PrefManager.WEB_URL, response.getResults().getWeb().getUrl());

                            onResultListener.onResult(Constants.SignupResult.SUCCESS, "success");
                        }
                    } catch (Exception e) {
                        onResultListener.onResult(Constants.SignupResult.FAIL, e.toString());
                    }
                }
            }).start();

    }

    private void loadErrorUrl() {
        isPageLoaded = false;
        if(swipeRefreshLayout != null) swipeRefreshLayout.setEnabled(false);
        if(webview != null) webview.loadUrl("file:///android_asset/index.html");
    }

    @Override
    public void onCalendar(String date, String callback) {
        uiInterface.onCalendar(date, callback);
    }

    @Override
    public void onPageLoaded() {
        getActivity().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
            }
        });
    }

    @Override
    public void onScrollChanged(final int pos) {
    }

    @Override
    public void setRefreshEnabled(final boolean enabled) {
        swipeRefreshLayout.post(new Runnable() {
            @Override
            public void run() {
                swipeRefreshLayout.setEnabled(enabled);
            }
        });
    }

    @Override
    public void showDatePicker(DateRequest dateRequest) {
        DialogFragment newFragment = DatePickerFragment.newInstance(dateRequest);
        newFragment.show(getActivity().getSupportFragmentManager(), "datePicker");
    }

    @Override
    public void showTimePicker(TimeRequest timeRequest) {
        DialogFragment newFragment = TimePickerFragment.newInstance(timeRequest);
        newFragment.show(getActivity().getSupportFragmentManager(), "timePicker");
    }

    @Override
    public void onFinish(String path) {

        if(!TextUtils.isEmpty(path)) {
            Intent intent = getActivity().getIntent();
            intent.putExtra("path", path);
            getActivity().setResult(RESULT_OK, intent);
        }
        getActivity().finish();
    }

    @Override
    public void reload() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.reload();
                }
            });
        }
    }

    @Override
    public void retry() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.loadUrl(mFailingUrl);
                }
            });
        }
    }

}
