package com.tenqube.visual_third.ui;

import android.annotation.SuppressLint;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.app.DialogFragment;
import android.support.v4.app.FragmentManager;
import android.support.v4.content.ContextCompat;
import android.support.v4.widget.SwipeRefreshLayout;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.ProgressBar;

import com.tenqube.visual_third.R;
import com.tenqube.visual_third.VisualServiceImpl;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.api.VisualApi;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.js.DateRequest;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.model.js.TimeRequest;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.web.ErrorImpl;
import com.tenqube.visual_third.web.LogImpl;
import com.tenqube.visual_third.web.RepoImpl;
import com.tenqube.visual_third.web.SystemImpl;
import com.tenqube.visual_third.web.UiImpl;
import com.tenqube.visual_third.web.ViewContractor;
import com.tenqube.visual_third.web.VisualInterface;

import org.json.JSONException;
import org.json.JSONObject;

import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.ui.VisualActivityFragment.NOTIFICATION_REQUEST_CODE;
import static com.tenqube.visual_third.util.Utils.isNotiEnabled;
import static com.tenqube.visual_third.util.Utils.isSDKDisabled;
import static com.tenqube.visual_third.util.WebViewHelper.getBaseUrl;
import static com.tenqube.visual_third.util.WebViewHelper.settings;

public class VisualWebActivity extends AppCompatActivity implements TimePickerFragment.Callback, DatePickerFragment.Callback, ViewContractor {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();
    public static final String ARG_PATH = "PATH";
    public static final int POPUP_CODE = 1000;

    private PrefManager prefManager;
    private VisualApi api;

    private boolean isPageLoaded;
    private String mFailingUrl;

    public static void startActivity(Context context, String path) throws AuthException {
        if(TextUtils.isEmpty(PrefManager.getInstance(context).loadStringValue(PrefManager.UID, "")))
            throw new AuthException("Need to sign up");

        Intent intent = new Intent(context, VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        context.startActivity(intent);

    }

    private String path;
    private WebView webview;

    private VisualInterface.Repo repoInterface;
    private VisualInterface.UI uiInterface;
    private VisualInterface.System systemInterface;
    private VisualInterface.Error errorInterface;
    private VisualInterface.Log logInterface;

    private SwipeRefreshLayout swipeRefreshLayout;

    private ProgressBar progressBar;

    @SuppressLint({"JavascriptInterface", "AddJavascriptInterface", "SetJavaScriptEnabled"})
    @Override
    protected void onCreate(Bundle savedInstanceState) {


        requestWindowFeature(Window.FEATURE_NO_TITLE);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        getWindow().setFlags(WindowManager.LayoutParams.FLAG_BLUR_BEHIND,
                WindowManager.LayoutParams.FLAG_BLUR_BEHIND);

        setStatusColor();
        super.onCreate(savedInstanceState);
        try {
            if(isSDKDisabled(getApplicationContext())) {
                throw new Exception("SDK disabled");
            }

            initFabric(VisualWebActivity.this);
            AnswerManager.onKeyMetric(new LogRequest("VisualWebActivity"));

            api = VisualApi.getInstance(getApplicationContext());
            prefManager = PrefManager.getInstance(getApplicationContext());

            path = getIntent().getStringExtra(ARG_PATH);

            setContentView(R.layout.activity_visual_web);

            progressBar = (ProgressBar) findViewById(R.id.progress_bar);

            swipeRefreshLayout = (SwipeRefreshLayout) findViewById(R.id.swipe_refresh_layout);
            swipeRefreshLayout.setEnabled(false);
            swipeRefreshLayout.setColorSchemeResources(
                    R.color.colorPopupRed,
                    R.color.colorPopupRed,
                    R.color.colorPopupRed);

            swipeRefreshLayout.setOnRefreshListener(new SwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    swipeRefreshLayout.setRefreshing(false);
                    webview.reload();
                }
            });

            webview = (WebView) findViewById(R.id.visual_web_view);
            webview.setBackgroundColor(Color.TRANSPARENT);
            webview.getSettings().setTextZoom(100);//font size

            // javascriptInterface
            errorInterface = new ErrorImpl(VisualWebActivity.this, this, webview);
            logInterface = new LogImpl(VisualWebActivity.this, this, webview, errorInterface);
            repoInterface = new RepoImpl(VisualWebActivity.this, this, webview, errorInterface,
                    VisualRepository.getInstance(VisualWebActivity.this),
                    api,
                    new AnalysisServiceImpl(VisualWebActivity.this),
                    VisualAlarmManager.getInstance(VisualWebActivity.this));

            uiInterface = new UiImpl(VisualWebActivity.this, this, webview, errorInterface, new BottomDialog(VisualWebActivity.this));
            systemInterface = new SystemImpl(VisualWebActivity.this, this, webview, errorInterface);

            webview.addJavascriptInterface(errorInterface, "visualError");
            webview.addJavascriptInterface(repoInterface, "visualRepo");
            webview.addJavascriptInterface(uiInterface, "visualUI");
            webview.addJavascriptInterface(systemInterface, "visualSystem");
            webview.addJavascriptInterface(logInterface, "visualLog");

            // settings
            settings(webview);

            // error
            webview.setWebViewClient(new WebViewClient() {
                public void onReceivedError(WebView view, int errorCode, String description, String failingUrl) {
                    mFailingUrl = failingUrl;
                    loadErrorUrl();
                    progressBar.setVisibility(View.GONE);

                }

                @Override
                public void onPageStarted(WebView view, String url, Bitmap favicon) {
                    super.onPageStarted(view, url, favicon);

                    progressBar.setVisibility(View.VISIBLE);

                }

                @Override
                public void onPageCommitVisible(WebView view, String url) {
                    super.onPageCommitVisible(view, url);
                    progressBar.setVisibility(View.GONE);
                }
            });

            if(prefManager.loadStringValue(PrefManager.QUALIFIER, "dev").equals("dev"))
                webview.getSettings().setCacheMode(WebSettings.LOAD_NO_CACHE);

            loadUrl();

        } catch (Exception e) {
            finish();
        }
    }

    private void loadUrl() {
        if(webview != null) {
            isPageLoaded = false;
            if(TextUtils.isEmpty(path)) path = "";
            String url = getBaseUrl(prefManager) + path;
            webview.loadUrl(url);
        }
    }

    private void loadErrorUrl() {
        isPageLoaded = false;
        if(swipeRefreshLayout != null) swipeRefreshLayout.setEnabled(false);
        if(webview != null) webview.loadUrl("file:///android_asset/index.html");
    }

    @Override
    public void onCalendar(String date, String callback) {
        uiInterface.onCalendar(date, callback);
    }


    @Override
    public void onPageLoaded() {
        runOnUiThread(new Runnable() {
            @Override
            public void run() {
                isPageLoaded = true;
            }
        });
    }

    @Override
    public void setRefreshEnabled(final boolean enabled) {
        swipeRefreshLayout.post(new Runnable() {
            @Override
            public void run() {
                swipeRefreshLayout.setEnabled(enabled);
            }
        });
    }

    @Override
    public void showDatePicker(DateRequest dateRequest) {
        DialogFragment newFragment = DatePickerFragment.newInstance(dateRequest);
        newFragment.show(getSupportFragmentManager(), "datePicker");
    }

    @Override
    public void showTimePicker(TimeRequest timeRequest) {
        DialogFragment newFragment = TimePickerFragment.newInstance(timeRequest);
        newFragment.show(getSupportFragmentManager(), "timePicker");
    }

    @Override
    public void onFinish(String path) {

        Intent intent = getIntent();
        setResult(TextUtils.isEmpty(path) || "false".equals(path) ?
                RESULT_CANCELED
                :
                RESULT_OK, intent);
        finish();

    }

    @Override
    public void startPopup(String path) {
        Intent intent = new Intent(getApplicationContext(), VisualWebActivity.class);
        intent.putExtra(ARG_PATH, path);
        startActivityForResult(intent, POPUP_CODE);
    }

    @Override
    public void showNotiCatch() {
        NotiCatchDialogFragment notiCatchDialogFragment = NotiCatchDialogFragment.newInstance();

        if(!notiCatchDialogFragment.isAdded()) {
            FragmentManager fragmentManager = getSupportFragmentManager();
            notiCatchDialogFragment.show(fragmentManager, NotiCatchDialogFragment.TAG);
        }
    }

    @Override
    public void goNotiSettings() throws ActivityNotFoundException {
        Intent intent = new Intent("android.settings.ACTION_NOTIFICATION_LISTENER_SETTINGS");
        startActivityForResult(intent, NOTIFICATION_REQUEST_CODE);
    }

    @Override
    public void reload() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.reload();
                }
            });
        }
    }

    @Override
    public void retry() {
        if(webview != null)  {
            webview.post(new Runnable() {
                @Override
                public void run() {
                    webview.loadUrl(mFailingUrl);
                }
            });
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, final Intent data) {
        super.onActivityResult(requestCode, resultCode, data);

        switch (requestCode) {
            case NOTIFICATION_REQUEST_CODE:
                boolean isEnabled = isNotiEnabled(getApplicationContext());

                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("isNotiEnabled", isEnabled);
                    String json = jsonObject.toString();
                    if(systemInterface != null) systemInterface.onNotiCallback(json);

                } catch (JSONException e) {
                    e.printStackTrace();
                }
                break;

            case POPUP_CODE:
                setResult(resultCode);
                finish();

                break;
        }
    }

    @Override
    public void onBackPressed() {
        try {
            if(isPageLoaded && uiInterface != null) {
                uiInterface.onFinish();
            } else {
                super.onBackPressed();

            }
        } catch (Exception e) {
            super.onBackPressed();
        }
    }

    @Override
    protected void onResume() {
        super.onResume();
        overridePendingTransition(R.anim.slide_in_up, R.anim.stay);

    }

    @Override
    protected void onPause() {
        super.onPause();
        overridePendingTransition(R.anim.stay, R.anim.slide_out_down);
    }

    public void setStatusColor() {
        Window window = getWindow();

        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.M){
            window.getDecorView().setSystemUiVisibility(
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE
                            | View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR);
            window.setStatusBarColor(ContextCompat.getColor(getApplicationContext(), android.R.color.white));

        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP){
            window.getDecorView().setSystemUiVisibility(
                    View.SYSTEM_UI_FLAG_LAYOUT_STABLE);

            window.setStatusBarColor(ContextCompat.getColor(getApplicationContext(), android.R.color.white));

        }
    }
}
