package com.tek42.perforce.parse;

import java.io.Writer;

import com.tek42.perforce.PerforceException;

/**
 * Interface for parsing perforce output into a concrete object and also for saving the object back to perforce. 
 * <p>
 * The pattern for using this template to build an object is:
 * <ol>
 * <li>Retrieve commands to execute via {@link #getBuildCmd(String)}</li>
 * <li>Execute those commands and capture the output.</li>
 * <li>Pass the output to {@link #build(StringBuilder)}</li>
 * </ol>
 * <p>And conversely for saving:
 * <ol>
 * <li>Retrieve commands to execute via {@link #getSaveCmd()}</li>
 * <li>Execute those commands and retrieve a Writer for I/O</li>
 * <li>Pass the writer and the object to save to {@link #save(Object, Writer)}</li>
 * </ol>
 * 
 * @author Mike Wille
 */
public interface Builder<T> {
	/**
	 * This should return the command line tokens to execute for retrieving an object from Perforce. For example:
	 * String[] { "p4", "user", "-o", "mwille" }
	 * 
	 * @param id
	 *            The ID or Name of the object we are working on. In the case of a changelist, the changelist number. In
	 *            the case of a user, the username.
	 * @return A 1D string array of tokens to execute.
	 */
	public String[] getBuildCmd(String id);

	/**
	 * This should assemble an object from a string value. Format of the string is dependent on the object we are
	 * building and is unspecified.
	 * 
	 * @param sb
	 *            The StringBuilder containing the string value.
	 * @return The resulting object
	 * @throws PerforceException
	 *             If the format is invalid or other errors occur.
	 */
	public T build(StringBuilder sb) throws PerforceException;

	/**
	 * The converse of {@link #getBuildCmd(String)} this should return the command line tokens to execute for saving an
	 * object to Perforce. For example: String[] { "p4", "user", "-i", "mwille" }
	 * 
	 * @return A 1D string array of tokens to execute
	 */
	public String[] getSaveCmd();

	/**
	 * The converse of {@link #build(StringBuilder)} this should take an object and disassemble it for writing to the
	 * Perforce server. The specification of what is written to the Writer is dependant on the object being saved.
	 * 
	 * @param obj
	 *            The object to be saved
	 * @param writer
	 *            The Writer to write the string representation to
	 * @throws PerforceException
	 *             If the object is invalid or there is an issue with writing
	 */
	public void save(T obj, Writer writer) throws PerforceException;

}
