/**
 *
 *                                  Apache License
 *                            Version 2.0, January 2004
 *                         http://www.apache.org/licenses/
 *
 *    TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION
 *
 *    1. Definitions.
 *
 *       "License" shall mean the terms and conditions for use, reproduction,
 *       and distribution as defined by Sections 1 through 9 of this document.
 *
 *       "Licensor" shall mean the copyright owner or entity authorized by
 *       the copyright owner that is granting the License.
 *
 *       "Legal Entity" shall mean the union of the acting entity and all
 *       other entities that control, are controlled by, or are under common
 *       control with that entity. For the purposes of this definition,
 *       "control" means (i) the power, direct or indirect, to cause the
 *       direction or management of such entity, whether by contract or
 *       otherwise, or (ii) ownership of fifty percent (50%) or more of the
 *       outstanding shares, or (iii) beneficial ownership of such entity.
 *
 *       "You" (or "Your") shall mean an individual or Legal Entity
 *       exercising permissions granted by this License.
 *
 *       "Source" form shall mean the preferred form for making modifications,
 *       including but not limited to software source code, documentation
 *       source, and configuration files.
 *
 *       "Object" form shall mean any form resulting from mechanical
 *       transformation or translation of a Source form, including but
 *       not limited to compiled object code, generated documentation,
 *       and conversions to other media types.
 *
 *       "Work" shall mean the work of authorship, whether in Source or
 *       Object form, made available under the License, as indicated by a
 *       copyright notice that is included in or attached to the work
 *       (an example is provided in the Appendix below).
 *
 *       "Derivative Works" shall mean any work, whether in Source or Object
 *       form, that is based on (or derived from) the Work and for which the
 *       editorial revisions, annotations, elaborations, or other modifications
 *       represent, as a whole, an original work of authorship. For the purposes
 *       of this License, Derivative Works shall not include works that remain
 *       separable from, or merely link (or bind by name) to the interfaces of,
 *       the Work and Derivative Works thereof.
 *
 *       "Contribution" shall mean any work of authorship, including
 *       the original version of the Work and any modifications or additions
 *       to that Work or Derivative Works thereof, that is intentionally
 *       submitted to Licensor for inclusion in the Work by the copyright owner
 *       or by an individual or Legal Entity authorized to submit on behalf of
 *       the copyright owner. For the purposes of this definition, "submitted"
 *       means any form of electronic, verbal, or written communication sent
 *       to the Licensor or its representatives, including but not limited to
 *       communication on electronic mailing lists, source code control systems,
 *       and issue tracking systems that are managed by, or on behalf of, the
 *       Licensor for the purpose of discussing and improving the Work, but
 *       excluding communication that is conspicuously marked or otherwise
 *       designated in writing by the copyright owner as "Not a Contribution."
 *
 *       "Contributor" shall mean Licensor and any individual or Legal Entity
 *       on behalf of whom a Contribution has been received by Licensor and
 *       subsequently incorporated within the Work.
 *
 *    2. Grant of Copyright License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       copyright license to reproduce, prepare Derivative Works of,
 *       publicly display, publicly perform, sublicense, and distribute the
 *       Work and such Derivative Works in Source or Object form.
 *
 *    3. Grant of Patent License. Subject to the terms and conditions of
 *       this License, each Contributor hereby grants to You a perpetual,
 *       worldwide, non-exclusive, no-charge, royalty-free, irrevocable
 *       (except as stated in this section) patent license to make, have made,
 *       use, offer to sell, sell, import, and otherwise transfer the Work,
 *       where such license applies only to those patent claims licensable
 *       by such Contributor that are necessarily infringed by their
 *       Contribution(s) alone or by combination of their Contribution(s)
 *       with the Work to which such Contribution(s) was submitted. If You
 *       institute patent litigation against any entity (including a
 *       cross-claim or counterclaim in a lawsuit) alleging that the Work
 *       or a Contribution incorporated within the Work constitutes direct
 *       or contributory patent infringement, then any patent licenses
 *       granted to You under this License for that Work shall terminate
 *       as of the date such litigation is filed.
 *
 *    4. Redistribution. You may reproduce and distribute copies of the
 *       Work or Derivative Works thereof in any medium, with or without
 *       modifications, and in Source or Object form, provided that You
 *       meet the following conditions:
 *
 *       (a) You must give any other recipients of the Work or
 *           Derivative Works a copy of this License; and
 *
 *       (b) You must cause any modified files to carry prominent notices
 *           stating that You changed the files; and
 *
 *       (c) You must retain, in the Source form of any Derivative Works
 *           that You distribute, all copyright, patent, trademark, and
 *           attribution notices from the Source form of the Work,
 *           excluding those notices that do not pertain to any part of
 *           the Derivative Works; and
 *
 *       (d) If the Work includes a "NOTICE" text file as part of its
 *           distribution, then any Derivative Works that You distribute must
 *           include a readable copy of the attribution notices contained
 *           within such NOTICE file, excluding those notices that do not
 *           pertain to any part of the Derivative Works, in at least one
 *           of the following places: within a NOTICE text file distributed
 *           as part of the Derivative Works; within the Source form or
 *           documentation, if provided along with the Derivative Works; or,
 *           within a display generated by the Derivative Works, if and
 *           wherever such third-party notices normally appear. The contents
 *           of the NOTICE file are for informational purposes only and
 *           do not modify the License. You may add Your own attribution
 *           notices within Derivative Works that You distribute, alongside
 *           or as an addendum to the NOTICE text from the Work, provided
 *           that such additional attribution notices cannot be construed
 *           as modifying the License.
 *
 *       You may add Your own copyright statement to Your modifications and
 *       may provide additional or different license terms and conditions
 *       for use, reproduction, or distribution of Your modifications, or
 *       for any such Derivative Works as a whole, provided Your use,
 *       reproduction, and distribution of the Work otherwise complies with
 *       the conditions stated in this License.
 *
 *    5. Submission of Contributions. Unless You explicitly state otherwise,
 *       any Contribution intentionally submitted for inclusion in the Work
 *       by You to the Licensor shall be under the terms and conditions of
 *       this License, without any additional terms or conditions.
 *       Notwithstanding the above, nothing herein shall supersede or modify
 *       the terms of any separate license agreement you may have executed
 *       with Licensor regarding such Contributions.
 *
 *    6. Trademarks. This License does not grant permission to use the trade
 *       names, trademarks, service marks, or product names of the Licensor,
 *       except as required for reasonable and customary use in describing the
 *       origin of the Work and reproducing the content of the NOTICE file.
 *
 *    7. Disclaimer of Warranty. Unless required by applicable law or
 *       agreed to in writing, Licensor provides the Work (and each
 *       Contributor provides its Contributions) on an "AS IS" BASIS,
 *       WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 *       implied, including, without limitation, any warranties or conditions
 *       of TITLE, NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A
 *       PARTICULAR PURPOSE. You are solely responsible for determining the
 *       appropriateness of using or redistributing the Work and assume any
 *       risks associated with Your exercise of permissions under this License.
 *
 *    8. Limitation of Liability. In no event and under no legal theory,
 *       whether in tort (including negligence), contract, or otherwise,
 *       unless required by applicable law (such as deliberate and grossly
 *       negligent acts) or agreed to in writing, shall any Contributor be
 *       liable to You for damages, including any direct, indirect, special,
 *       incidental, or consequential damages of any character arising as a
 *       result of this License or out of the use or inability to use the
 *       Work (including but not limited to damages for loss of goodwill,
 *       work stoppage, computer failure or malfunction, or any and all
 *       other commercial damages or losses), even if such Contributor
 *       has been advised of the possibility of such damages.
 *
 *    9. Accepting Warranty or Additional Liability. While redistributing
 *       the Work or Derivative Works thereof, You may choose to offer,
 *       and charge a fee for, acceptance of support, warranty, indemnity,
 *       or other liability obligations and/or rights consistent with this
 *       License. However, in accepting such obligations, You may act only
 *       on Your own behalf and on Your sole responsibility, not on behalf
 *       of any other Contributor, and only if You agree to indemnify,
 *       defend, and hold each Contributor harmless for any liability
 *       incurred by, or claims asserted against, such Contributor by reason
 *       of your accepting any such warranty or additional liability.
 *
 *    END OF TERMS AND CONDITIONS
 *
 *    APPENDIX: How to apply the Apache License to your work.
 *
 *       To apply the Apache License to your work, attach the following
 *       boilerplate notice, with the fields enclosed by brackets "[]"
 *       replaced with your own identifying information. (Don't include
 *       the brackets!)  The text should be enclosed in the appropriate
 *       comment syntax for the file format. We also recommend that a
 *       file or class name and description of purpose be included on the
 *       same "printed page" as the copyright notice for easier
 *       identification within third-party archives.
 *
 *    Copyright 2016 Alibaba Group
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */
package com.taobao.weex.ui.component;

import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Build;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.widget.FrameLayout;
import android.widget.FrameLayout.LayoutParams;

import com.taobao.weex.WXEnvironment;
import com.taobao.weex.WXSDKInstance;
import com.taobao.weex.annotation.Component;
import com.taobao.weex.annotation.JSMethod;
import com.taobao.weex.common.Constants;
import com.taobao.weex.common.OnWXScrollListener;
import com.taobao.weex.common.WXThread;
import com.taobao.weex.dom.WXDomObject;
import com.taobao.weex.ui.ComponentCreator;
import com.taobao.weex.ui.component.helper.WXStickyHelper;
import com.taobao.weex.ui.view.IWXScroller;
import com.taobao.weex.ui.view.WXBaseRefreshLayout;
import com.taobao.weex.ui.view.WXHorizontalScrollView;
import com.taobao.weex.ui.view.WXScrollView;
import com.taobao.weex.ui.view.WXScrollView.WXScrollViewListener;
import com.taobao.weex.ui.view.refresh.wrapper.BaseBounceView;
import com.taobao.weex.ui.view.refresh.wrapper.BounceScrollerView;
import com.taobao.weex.utils.WXLogUtils;
import com.taobao.weex.utils.WXUtils;
import com.taobao.weex.utils.WXViewUtils;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

//import com.taobao.weex.ui.WXRecycleImageManager;

/**
 * Component for scroller. It also support features like
 * "appear", "disappear" and "sticky"
 */
@Component(lazyload = false)

public class WXScroller extends WXVContainer<ViewGroup> implements WXScrollViewListener,Scrollable {

  public static final String DIRECTION = "direction";
  protected int mOrientation = Constants.Orientation.VERTICAL;
  private List<WXComponent> mRefreshs=new ArrayList<>();
  private int mChildrenLayoutOffset = 0;//Use for offset children layout
  private boolean mForceLoadmoreNextTime = false;
  private int mOffsetAccuracy = 10;
  private Point mLastReport = new Point(-1, -1);

  public static class Creator implements ComponentCreator {
    public WXComponent createInstance(WXSDKInstance instance, WXDomObject node, WXVContainer parent) throws IllegalAccessException, InvocationTargetException, InstantiationException {
      return new WXScroller(instance,node,parent);
    }
  }
  /**
   * Map for storing appear information
   **/
  private Map<String,AppearanceHelper> mAppearanceComponents = new HashMap<>();

  /**
   * Map for storing component that is sticky.
   **/
  private Map<String, Map<String, WXComponent>> mStickyMap = new HashMap<>();
  private FrameLayout mRealView;

  private int mContentHeight = 0;

  private WXStickyHelper stickyHelper;
  private Handler handler=new Handler();

  private boolean isScrollable = true;

  @Deprecated
  public WXScroller(WXSDKInstance instance, WXDomObject dom, WXVContainer parent, String instanceId, boolean isLazy) {
    this(instance,dom,parent);
  }

  public WXScroller(WXSDKInstance instance, WXDomObject node,
                    WXVContainer parent) {
    super(instance, node, parent);
    stickyHelper = new WXStickyHelper(this);
  }

  /**
   * @return FrameLayout inner ScrollView
   */
  @Override
  public ViewGroup getRealView() {
    return mRealView;
  }


  @Override
  public void createViewImpl() {
    super.createViewImpl();
    for (int i = 0; i < mRefreshs.size(); i++) {
      WXComponent component = mRefreshs.get(i);
      component.createViewImpl();
      checkRefreshOrLoading(component);
    }
  }

  /**
   * @return ScrollView
   */
  public ViewGroup getInnerView() {
    if(getHostView() == null)
      return null;
    if (getHostView() instanceof BounceScrollerView) {
      return ((BounceScrollerView) getHostView()).getInnerView();
    } else {
      return getHostView();
    }
  }

  @Override
  public void addEvent(String type) {
    super.addEvent(type);
    if (Constants.Event.SCROLL.equals(type) && getInnerView() != null && getInnerView() instanceof WXScrollView) {
      ((WXScrollView) getInnerView()).addScrollViewListener(new WXScrollViewListener() {
        @Override
        public void onScrollChanged(WXScrollView scrollView, int x, int y, int oldx, int oldy) {
          if (shouldReport(x, y)) {
            Rect frame = scrollView.getContentFrame();

            Map<String, Object> event = new HashMap<>(2);
            Map<String, Object> contentSize = new HashMap<>(2);
            Map<String, Object> contentOffset = new HashMap<>(2);

            contentSize.put(Constants.Name.WIDTH, WXViewUtils.getWebPxByWidth(frame.width(), getInstance().getViewPortWidth()));
            contentSize.put(Constants.Name.HEIGHT, WXViewUtils.getWebPxByWidth(frame.height(), getInstance().getViewPortWidth()));

            contentOffset.put(Constants.Name.X, - WXViewUtils.getWebPxByWidth(x, getInstance().getViewPortWidth()));
            contentOffset.put(Constants.Name.Y, - WXViewUtils.getWebPxByWidth(y, getInstance().getViewPortWidth()));

            event.put(Constants.Name.CONTENT_SIZE, contentSize);
            event.put(Constants.Name.CONTENT_OFFSET, contentOffset);

            fireEvent(Constants.Event.SCROLL, event);
          }
        }

        @Override
        public void onScrollToBottom(WXScrollView scrollView, int x, int y) {
          //ignore
        }

        @Override
        public void onScrollStopped(WXScrollView scrollView, int x, int y) {
          //ignore
        }

        @Override
        public void onScroll(WXScrollView scrollView, int x, int y) {
          //ignore
        }
      });
    }
  }

  private boolean shouldReport(int x, int y) {
    if (mLastReport.x == -1 && mLastReport.y == -1) {
      mLastReport.x = x;
      mLastReport.y = y;
      return true;
    }

    if (mOrientation == Constants.Orientation.HORIZONTAL
            && Math.abs(x - mLastReport.x) >= mOffsetAccuracy) {
      mLastReport.x = x;
      mLastReport.y = y;
      return true;
    }

    if (mOrientation == Constants.Orientation.VERTICAL
            && Math.abs(y - mLastReport.y) >= mOffsetAccuracy) {
      mLastReport.x = x;
      mLastReport.y = y;
      return true;
    }

    return false;
  }

  /**
   * Intercept refresh view and loading view
   */
  @Override
  protected void addSubView(View child, int index) {
    if (child == null || getRealView() == null) {
      return;
    }

    if (child instanceof WXBaseRefreshLayout) {
      return;
    }

    int count = getRealView().getChildCount();
    index = index >= count ? -1 : index;
    if (index == -1) {
      getRealView().addView(child);
    } else {
      getRealView().addView(child, index);
    }
  }

  @Override
  protected int getChildrenLayoutTopOffset() {
    return mChildrenLayoutOffset;
  }

  /**
   * Intercept refresh view and loading view
   */
  @Override
  public void addChild(WXComponent child, int index) {
    mChildrenLayoutOffset += child.getLayoutTopOffsetForSibling();
    if (child instanceof WXBaseRefresh) {
      if (!checkRefreshOrLoading(child)) {
        mRefreshs.add(child);
      }
    }

    super.addChild(child,index);
  }

  /**
   * Setting refresh view and loading view
   * @param child the refresh_view or loading_view
   */

  private boolean checkRefreshOrLoading(final WXComponent child) {
    boolean result = false;
    if (child instanceof WXRefresh && getHostView() != null) {
      ((BaseBounceView) getHostView()).setOnRefreshListener((WXRefresh) child);
      Runnable runnable = WXThread.secure(new Runnable(){
        @Override
        public void run() {
          ((BaseBounceView) getHostView()).setHeaderView(child);
        }
      });
      handler.postDelayed(runnable,100);
    }

    if (child instanceof WXLoading && getHostView() !=null) {
      ((BaseBounceView) getHostView()).setOnLoadingListener((WXLoading)child);
      Runnable runnable= WXThread.secure(new Runnable(){
        @Override
        public void run() {
          ((BaseBounceView) getHostView()).setFooterView(child);
        }
      });
      handler.postDelayed(runnable, 100);
      result = true;
    }
    return result;
  }

  @Override
  public void remove(WXComponent child,boolean destory) {
    super.remove(child,destory);
    if(child instanceof WXLoading){
      ((BaseBounceView)getHostView()).removeFooterView(child);
    }else if(child instanceof WXRefresh){
      ((BaseBounceView)getHostView()).removeHeaderView(child);
    }
  }

  @Override
  public void destroy() {
    super.destroy();
    if (mAppearanceComponents != null) {
      mAppearanceComponents.clear();
    }
    if (mStickyMap != null) {
      mStickyMap.clear();
    }
    if (getInnerView() != null && getInnerView() instanceof IWXScroller) {
      ((IWXScroller) getInnerView()).destroy();
    }
  }

  @Override
  protected MeasureOutput measure(int width, int height) {
    MeasureOutput measureOutput = new MeasureOutput();
    if (this.mOrientation == Constants.Orientation.HORIZONTAL) {
      int screenW = WXViewUtils.getScreenWidth(WXEnvironment.sApplication);
      int weexW = WXViewUtils.getWeexWidth(getInstanceId());
      measureOutput.width = width > (weexW >= screenW ? screenW : weexW) ? FrameLayout.LayoutParams.MATCH_PARENT
                                                                         : width;
      measureOutput.height = height;
    } else {
      int screenH = WXViewUtils.getScreenHeight(WXEnvironment.sApplication);
      int weexH = WXViewUtils.getWeexHeight(getInstanceId());
      measureOutput.height = height > (weexH >= screenH ? screenH : weexH) ? FrameLayout.LayoutParams.MATCH_PARENT
                                                                           : height;
      measureOutput.width = width;
    }
    return measureOutput;
  }

  @Override
  protected ViewGroup initComponentHostView(@NonNull Context context) {
    String scroll;
    if (getDomObject() == null || getDomObject().getAttrs().isEmpty()) {
      scroll = "vertical";
    } else {
      scroll = getDomObject().getAttrs().getScrollDirection();
    }

    ViewGroup host;
    if(("horizontal").equals(scroll)){
      mOrientation = Constants.Orientation.HORIZONTAL;
      WXHorizontalScrollView scrollView = new WXHorizontalScrollView(context);
      mRealView = new FrameLayout(context);
      scrollView.setScrollViewListener(new WXHorizontalScrollView.ScrollViewListener() {
        @Override
        public void onScrollChanged(WXHorizontalScrollView scrollView, int x, int y, int oldx, int oldy) {
          procAppear(x,y,oldx,oldy);
        }
      });
      FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(
        LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
      scrollView.addView(mRealView, layoutParams);
      scrollView.setHorizontalScrollBarEnabled(false);

      host = scrollView;
    }else{
      mOrientation = Constants.Orientation.VERTICAL;
      BounceScrollerView scrollerView = new BounceScrollerView(context, mOrientation, this);
      mRealView = new FrameLayout(context);
      WXScrollView innerView = scrollerView.getInnerView();
      innerView.addScrollViewListener(this);
      FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(
        LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
      innerView.addView(mRealView, layoutParams);
      innerView.setVerticalScrollBarEnabled(true);
      innerView.addScrollViewListener(new WXScrollViewListener() {
        @Override
        public void onScrollChanged(WXScrollView scrollView, int x, int y, int oldx, int oldy) {

        }

        @Override
        public void onScrollToBottom(WXScrollView scrollView, int x, int y) {

        }

        @Override
        public void onScrollStopped(WXScrollView scrollView, int x, int y) {
          List<OnWXScrollListener> listeners = getInstance().getWXScrollListeners();
          if(listeners!=null && listeners.size()>0){
            for (OnWXScrollListener listener : listeners) {
              if (listener != null) {
                listener.onScrollStateChanged(scrollView,x,y,OnWXScrollListener.IDLE);
              }
            }
          }
        }

        @Override
        public void onScroll(WXScrollView scrollView, int x, int y) {
          List<OnWXScrollListener> listeners = getInstance().getWXScrollListeners();
          if(listeners!=null && listeners.size()>0){
            for (OnWXScrollListener listener : listeners) {
              if (listener != null) {
                listener.onScrolled(scrollView, x, y);
              }
            }
          }
        }
      });
      host = scrollerView;
    }

    host.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
      @TargetApi(Build.VERSION_CODES.JELLY_BEAN)
      @Override
      public void onGlobalLayout() {
        procAppear(0,0,0,0);
        View view;
        if( (view = getHostView()) == null){
          return;
        }
        if(Build.VERSION.SDK_INT >=  Build.VERSION_CODES.JELLY_BEAN) {
          view.getViewTreeObserver().removeOnGlobalLayoutListener(this);
        }else{
          view.getViewTreeObserver().removeGlobalOnLayoutListener(this);
        }
      }
    });
    return host;
  }

  @Override
  public int getScrollY() {
    return getInnerView() == null ? 0 : getInnerView().getScrollY();
  }

  @Override
  public int getScrollX() {
    return getInnerView() == null ? 0 : getInnerView().getScrollX();
  }

  @Override
  public int getOrientation() {
    return mOrientation;
  }

  public Map<String, Map<String, WXComponent>> getStickMap() {
    return mStickyMap;
  }

  @Override
  protected boolean setProperty(String key, Object param) {
    switch (key) {
      case Constants.Name.SHOW_SCROLLBAR:
        Boolean result = WXUtils.getBoolean(param,null);
        if (result != null)
          setShowScrollbar(result);
        return true;
      case Constants.Name.SCROLLABLE:
        boolean scrollable = WXUtils.getBoolean(param, true);
        setScrollable(scrollable);
        return true;
      case Constants.Name.OFFSET_ACCURACY:
        int accuracy = WXUtils.getInteger(param, 10);
        setOffsetAccuracy(accuracy);
        return true;
    }
    return super.setProperty(key, param);
  }

  @WXComponentProp(name = Constants.Name.SHOW_SCROLLBAR)
  public void setShowScrollbar(boolean show) {
    if(getInnerView()==null){
      return;
    }
    if (mOrientation == Constants.Orientation.VERTICAL) {
      getInnerView().setVerticalScrollBarEnabled(show);
    } else {
      getInnerView().setHorizontalScrollBarEnabled(show);
    }
  }

  @WXComponentProp(name = Constants.Name.SCROLLABLE)
  public void setScrollable(boolean scrollable) {
    this.isScrollable = scrollable;
    View hostView = getInnerView();
    if(hostView instanceof WXHorizontalScrollView) {
      ((WXHorizontalScrollView)hostView).setScrollable(scrollable);
    }else if(hostView instanceof WXScrollView) {
      ((WXScrollView)hostView).setScrollable(scrollable);
    }
  }

  @WXComponentProp(name = Constants.Name.OFFSET_ACCURACY)
  public void setOffsetAccuracy(int accuracy) {
    float realPx = WXViewUtils.getRealPxByWidth(accuracy, getInstance().getViewPortWidth());
    this.mOffsetAccuracy = (int) realPx;
  }

  @Override
  public boolean isScrollable() {
    return isScrollable;
  }


  // TODO Need constrain, each container can only have one sticky child
  @Override
  public void bindStickStyle(WXComponent component) {
    stickyHelper.bindStickStyle(component,mStickyMap);
  }

  @Override
  public void unbindStickStyle(WXComponent component) {
    stickyHelper.unbindStickStyle(component,mStickyMap);
  }

  /**
   * Bind appear event
   */
  @Override
  public void bindAppearEvent(WXComponent component) {
    setWatch(AppearanceHelper.APPEAR,component,true);
  }

  private void setWatch(int event,WXComponent component,boolean isWatch){
    AppearanceHelper item = mAppearanceComponents.get(component.getRef());
    if (item == null) {
      item = new AppearanceHelper(component);
      mAppearanceComponents.put(component.getRef(),item);
    }

    item.setWatchEvent(event,isWatch);

    procAppear(0,0,0,0);//check current components appearance status.
  }

  /**
   * Bind disappear event
   */
  @Override
  public void bindDisappearEvent(WXComponent component) {
    setWatch(AppearanceHelper.DISAPPEAR,component,true);
  }

  /**
   * Remove appear event
   */
  @Override
  public void unbindAppearEvent(WXComponent component) {
    setWatch(AppearanceHelper.APPEAR,component,false);
  }

  /**
   * Remove disappear event
   */
  @Override
  public void unbindDisappearEvent(WXComponent component) {
    setWatch(AppearanceHelper.DISAPPEAR,component,false);
  }

  @Override
  public void scrollTo(WXComponent component, Map<String, Object> options) {
    float offsetFloat = 0;
    boolean smooth = true;

    if (options != null) {
      String offset = options.get(Constants.Name.OFFSET) == null ? "0" : options.get(Constants.Name.OFFSET).toString();
      smooth = WXUtils.getBoolean(options.get(Constants.Name.ANIMATED), true);
      if (offset != null) {
        try {
          offsetFloat = WXViewUtils.getRealPxByWidth(Float.parseFloat(offset), getInstance().getViewPortWidth());
        }catch (Exception e ){
          WXLogUtils.e("Float parseFloat error :"+e.getMessage());
        }
      }
    }

    int viewYInScroller=component.getAbsoluteY() - getAbsoluteY();
    int viewXInScroller=component.getAbsoluteX() - getAbsoluteX();

    scrollBy(viewXInScroller - getScrollX() + (int) offsetFloat, viewYInScroller - getScrollY() + (int) offsetFloat, smooth);
  }

  /**
   * Scroll by specified distance. Horizontal scroll is not supported now.
   * @param x horizontal distance, not support
   * @param y vertical distance. Negative for scroll to top
   */
  public void scrollBy(final int x, final int y) {
    scrollBy(x, y, false);
  }

  public void scrollBy(final int x, final int y, final boolean smooth) {
    if (getInnerView() == null) {
      return;
    }

    getInnerView().postDelayed(new Runnable() {
      @Override
      public void run() {
        if (mOrientation == Constants.Orientation.VERTICAL) {
          if (smooth) {
            ((WXScrollView) getInnerView()).smoothScrollBy(0, y);
          } else {
            ((WXScrollView) getInnerView()).scrollBy(0, y);
          }
        } else {
          if (smooth) {
            ((WXHorizontalScrollView) getInnerView()).smoothScrollBy(x, 0);
          } else {
            ((WXHorizontalScrollView) getInnerView()).scrollBy(x, 0);
          }
        }
        getInnerView().invalidate();
      }
    }, 16);
  }

  @Override
  public void onScrollChanged(WXScrollView scrollView, int x, int y,
                              int oldx, int oldy) {
    procAppear( x, y, oldx, oldy);
  }

  /**
   * Process event like appear and disappear
   */
  private void procAppear(int x, int y, int oldx,
                          int oldy) {
    int moveY = y - oldy;
    int moveX = x - oldx;
    String direction = moveY > 0 ? Constants.Value.DIRECTION_UP :
            moveY < 0 ? Constants.Value.DIRECTION_DOWN : null;
    if (mOrientation == Constants.Orientation.HORIZONTAL && moveX != 0) {
      direction = moveX > 0 ? Constants.Value.DIRECTION_RIGHT : Constants.Value.DIRECTION_LEFT;
    }

    for (Entry<String, AppearanceHelper> item : mAppearanceComponents.entrySet()) {
      AppearanceHelper helper = item.getValue();

      if (!helper.isWatch()) {
        continue;
      }
      boolean visible = helper.isViewVisible();

      int result = helper.setAppearStatus(visible);
      if (result != AppearanceHelper.RESULT_NO_CHANGE) {
        helper.getAwareChild().notifyAppearStateChange(result == AppearanceHelper.RESULT_APPEAR ? Constants.Event.APPEAR : Constants.Event.DISAPPEAR, direction);
      }
    }
  }

  @Override
  public void onScrollToBottom(WXScrollView scrollView, int x, int y) {

  }

  @Override
  public void onScrollStopped(WXScrollView scrollView, int x, int y) {
  }

  @Override
  public void onScroll(WXScrollView scrollView, int x, int y) {

    this.onLoadMore(scrollView, x, y);
  }

  /**
   * Handle loadMore Event.when Scroller has bind loadMore Event and set the attr of loadMoreOffset
   * it will tell the JS to handle the event of onLoadMore;
   * @param scrollView  the WXScrollView
   * @param x the X direction
   * @param y the Y direction
   */
  protected void onLoadMore(WXScrollView scrollView, int x, int y) {
    try {
      String offset = getDomObject().getAttrs().getLoadMoreOffset();
      if (TextUtils.isEmpty(offset)) {
        return;
      }
      int offsetInt = (int)WXViewUtils.getRealPxByWidth(Float.parseFloat(offset), WXSDKInstance.getViewPortWidth());

      int contentH = scrollView.getChildAt(0).getHeight();
      int scrollerH = scrollView.getHeight();
      int offScreenY = contentH - y - scrollerH;
      if (offScreenY < offsetInt) {
        if (WXEnvironment.isApkDebugable()) {
          WXLogUtils.d("[WXScroller-onScroll] offScreenY :" + offScreenY);
        }
        if (mContentHeight != contentH || mForceLoadmoreNextTime) {
          fireEvent(Constants.Event.LOADMORE);
          mContentHeight = contentH;
          mForceLoadmoreNextTime = false;
        }
      }
    } catch (Exception e) {
      WXLogUtils.d("[WXScroller-onScroll] ", e);
    }

  }

  @JSMethod
  public void resetLoadmore() {
    mForceLoadmoreNextTime = true;
  }
}
