package com.surveymonkey.surveymonkeyandroidsdk

import android.annotation.SuppressLint
import android.app.ProgressDialog
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.graphics.Bitmap
import android.net.ConnectivityManager
import android.net.Uri
import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.lifecycleScope
import androidx.loader.content.Loader
import com.surveymonkey.surveymonkeyandroidsdk.databinding.FragmentSmfeedbackBinding
import com.surveymonkey.surveymonkeyandroidsdk.loaders.GetRespondentTaskLoader
import com.surveymonkey.surveymonkeyandroidsdk.loaders.GetRespondentTokenTaskLoader
import com.surveymonkey.surveymonkeyandroidsdk.manager.SurveyManager
import com.surveymonkey.surveymonkeyandroidsdk.repository.SurveyRepositoryImpl
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMConstants
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMError
import com.surveymonkey.surveymonkeyandroidsdk.utils.SMNetworkUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONException
import org.json.JSONObject
import java.net.MalformedURLException
import java.net.URL

internal class SMFeedbackFragmentNew : Fragment(), SMExceptionHandler {
    private var mWebView: WebView? = null
    private var mSPageHTML: String? = null
    private var mHasPreLoadedHTML = false
    private var mHasLoadedSPageWebView = false
    private lateinit var mURL: String
    private var mError: SMError? = null
    private var mTokenURL: String? = null
    private var mToken: String? = null
    private var mMasheryApiKey: String? = null
    private var mProgressDialog: ProgressDialog? = null
    private var connectivityMonitor: ConnectivityMonitor? = null
    private var getRespondentTokenTaskLoader: GetRespondentTokenTaskLoader? = null
    private var getRespondentTaskLoader: GetRespondentTaskLoader? = null

    lateinit var surveyManager: SurveyManager

    private lateinit var binding: FragmentSmfeedbackBinding

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        mHasPreLoadedHTML = false
        mHasLoadedSPageWebView = false
        mSPageHTML = null
        val bundle = this.arguments
        if (bundle != null) {
            mURL = bundle.getString(KEY_SM_SPAGE_URL,"")
            mHasPreLoadedHTML = bundle.getBoolean(KEY_SM_HAS_LOADED_SPAGE_HTML)
            if (mHasPreLoadedHTML) {
                mSPageHTML = bundle.getString(KEY_SM_SPAGE_HTML)
                loadSurveyPage()
            } else {
                surveyManager = SurveyManager(SurveyRepositoryImpl())
                lifecycleScope.launch(Dispatchers.IO){
                    val response = surveyManager.getDataFromServer(mURL)
                    withContext(Dispatchers.Main){
                        handleResponse(response)
                    }
                }
            }
        }
    }

    private fun handleResponse(data: JSONObject?){
        try {
            if (data != null) {
                val sdkData = data.getJSONObject(SURVEY_STATUS)
                mSPageHTML = data.getString(HTML)
                if (!sdkData.getBoolean(COLLECTOR_CLOSED)) {
                    loadSurveyPage()
                } else {
                    //Collector is closed, so create fragment with no html so we can provide an error
                    handleCollectorClosed()
                }
            } else {
                //Collector is closed, so start activity with no html so we can provide an error
                handleCollectorClosed()
            }
        } catch (e: JSONException) {
            e.printStackTrace()
        }
    }

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View {
        binding = FragmentSmfeedbackBinding.inflate(inflater, container, false)
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
       // connectivityMonitor = ConnectivityMonitor()
        if (!mHasLoadedSPageWebView && mSPageHTML != null) {
            loadSurveyPage()
        }
        if (activity != null) {
            requireActivity().registerReceiver(
                connectivityMonitor,
                IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION)
            )
            mProgressDialog =
                ProgressDialog.show(activity, null, getString(R.string.sm_loading_status))
        }
    }

    override fun onDestroy() {
        if (connectivityMonitor != null && activity != null) {
            requireActivity().unregisterReceiver(connectivityMonitor)
        }
        if (mProgressDialog != null && mProgressDialog!!.isShowing) {
            mProgressDialog!!.dismiss()
        }
        if (getRespondentTaskLoader != null) {
            getRespondentTaskLoader!!.cancelLoad()
        }
        if (getRespondentTokenTaskLoader != null) {
            getRespondentTokenTaskLoader!!.cancelLoad()
        }
        super.onDestroy()
    }

    private fun loadSurveyPage() {
        if (view != null) {
            mHasLoadedSPageWebView = true
            val mWebView = binding.smFeedbackWebview
            mWebView.settings.javaScriptEnabled = true
            mWebView.webViewClient = SMWebviewClient()
            mWebView.loadDataWithBaseURL(mURL, mSPageHTML!!, null, "UTF-8", null)
        }
    }

    private fun handleCollectorClosed() {
        val mError = SMError.sdkServerErrorFromCode(SMError.ErrorType.ERROR_CODE_COLLECTOR_CLOSED, null)
        Log.d(SMConstants.DEBUG_TAG, mError.getDescription())
        handleError(mError)
    }

    /*fun onCreateRespondentTokenTaskLoader(id: Int, args: Bundle?): GetRespondentTokenTaskLoader {
        getRespondentTokenTaskLoader = GetRespondentTokenTaskLoader(activity, mTokenURL, this)
        return getRespondentTokenTaskLoader!!
    }

    fun onCreateRespondentTaskLoader(id: Int, args: Bundle?): GetRespondentTaskLoader {
        getRespondentTaskLoader = GetRespondentTaskLoader(activity, mToken, mMasheryApiKey, this)
        return getRespondentTaskLoader!!
    }

    fun onGetRespondentTokenTaskLoadFinished(loader: Loader<JSONObject>?, data: JSONObject?) {
        if (data != null) {
            try {
                mToken = data.getString("respondent_token")
                mMasheryApiKey = data.getString("mashery_api_key")
                requireActivity().supportLoaderManager.restartLoader(
                    RESPONDENT_LOADER_KEY,
                    null,
                    object : LoaderManager.LoaderCallbacks<JSONObject> {
                        override fun onCreateLoader(id: Int, args: Bundle?): Loader<JSONObject> {
                            return onCreateRespondentTaskLoader(id, args)
                        }

                        override fun onLoadFinished(loader: Loader<JSONObject>, data: JSONObject) {
                            onGetRespondentTaskLoadFinished(loader, data)
                        }

                        override fun onLoaderReset(loader: Loader<JSONObject>) {}
                    })
            } catch (e: JSONException) {
                val mError = SMError.sdkServerErrorFromCode(SMError.ErrorType.ERROR_CODE_TOKEN, e)
                Log.d(SMConstants.DEBUG_TAG, mError.getDescription())
                handleError(mError)
            }
        }
        getRespondentTokenTaskLoader = null
    }*/

    fun onGetRespondentTaskLoadFinished(loader: Loader<JSONObject>?, data: JSONObject?) {
        if (data != null) {
            try {
                val result = data.getJSONObject("data")
                handleRespondent(result)
            } catch (e: JSONException) {
                val mError = SMError.sdkServerErrorFromCode(
                    SMError.ErrorType.ERROR_CODE_RETRIEVING_RESPONSE,
                    e
                )
                Log.d(SMConstants.DEBUG_TAG, mError.getDescription())
                handleError(mError)
            }
        }
        getRespondentTaskLoader = null
    }

    private fun handleRespondent(r: JSONObject) {
        try {
            if (activity != null) {
                (activity as SMFeedbackFragmentListener?)!!.onFetchRespondentSuccess(r)
            }
        } catch (cce: ClassCastException) {
            Log.d(SMConstants.DEBUG_TAG, "SMFeedbackFragmentListener has not been implemented")
            showEndOfSurveyOverlay()
        }
    }

    override fun handleError(e: SMError?) {
        try {
            if (activity != null) {
                (activity as SMFeedbackFragmentListener?)!!.onFetchRespondentFailure(e)
            }
        } catch (cce: ClassCastException) {
            Log.d(SMConstants.DEBUG_TAG, "SMFeedbackFragmentListener has not been implemented")
            if (e!!.getErrorCode() == SMError.ErrorType.ERROR_CODE_COLLECTOR_CLOSED.value) {
                showSurveyClosedOverlay()
            } else {
                showEndOfSurveyOverlay()
            }
        }
    }

    private fun showEndOfSurveyOverlay() {
        val v = view
        if (v != null) {
            v.findViewById<View>(R.id.sm_feedback_survey_ended).visibility =
                View.VISIBLE
            v.findViewById<View>(R.id.sm_feedback_webview).visibility = View.GONE
        }
    }

    private fun showSurveyClosedOverlay() {
        val v = view
        if (v != null) {
            v.findViewById<View>(R.id.sm_feedback_survey_closed).visibility =
                View.VISIBLE
            v.findViewById<View>(R.id.sm_feedback_webview).visibility = View.GONE
        }
    }

    /*private val token: Unit
        private get() {
            if (activity != null) {
                LoaderManager.getInstance(this).restartLoader(
                    RESPONDENT_TOKEN_LOADER_KEY,
                    null,
                    object : LoaderManager.LoaderCallbacks<JSONObject> {
                        override fun onCreateLoader(id: Int, args: Bundle?): Loader<JSONObject> {
                            return onCreateRespondentTokenTaskLoader(id, args)
                        }

                        override fun onLoadFinished(loader: Loader<JSONObject>, data: JSONObject) {
                            onGetRespondentTokenTaskLoadFinished(loader, data)
                        }

                        override fun onLoaderReset(loader: Loader<JSONObject>) {}
                    })
            } else {
                Log.d(SMConstants.DEBUG_TAG, "getActivity is null in SMFeedbackFragment.getToken()")
            }
        }*/

    private inner class SMWebviewClient : WebViewClient() {
        override fun onPageStarted(view: WebView, url: String, favicon: Bitmap) {
            var resourceURL: URL? = null
            mProgressDialog!!.show()
            try {
                resourceURL = URL(url)
            } catch (e: MalformedURLException) {
                e.printStackTrace()
            }
            if (resourceURL != null && resourceURL.path != null && resourceURL.path.startsWith("/r/embed/sdk_token")) {
                view.stopLoading()
                // Clear webview to prevent misc json response from appearing
                view.loadUrl("about:blank")
                mTokenURL = url
              //  token
            } else {
                super.onPageStarted(view, url, favicon)
            }
        }

        override fun onPageFinished(view: WebView, url: String) {
            if (activity != null && !requireActivity().isDestroyed) {
                mProgressDialog!!.dismiss()
            }
        }

        override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
            if (!url.contains("surveymonkey.com/r/")) {
                val browserIntent = Intent(Intent.ACTION_VIEW, Uri.parse(url))
                startActivity(browserIntent)
                return true
            }
            return super.shouldOverrideUrlLoading(view, url)
        }
    }

    class ConnectivityMonitor : BroadcastReceiver() {
        @SuppressLint("UseRequireInsteadOfGet")
        override fun onReceive(context: Context, intent: Intent) {
            if (SMNetworkUtils.getConnectivityType(context) != SMNetworkUtils.ConnectivityType.NO_CONNECTION.value) {
                (context as FragmentActivity).supportFragmentManager
                    .findFragmentByTag(TAG)!!.view!!.findViewById<View>(
                        R.id.sm_feedback_no_network
                    ).visibility = View.VISIBLE
                context.supportFragmentManager
                    .findFragmentByTag(TAG)!!.view!!.findViewById<View>(
                        R.id.sm_feedback_webview
                    ).visibility = View.GONE
            } else {
                (context as FragmentActivity).supportFragmentManager
                    .findFragmentByTag(TAG)!!.view!!.findViewById<View>(
                        R.id.sm_feedback_no_network
                    ).visibility = View.GONE
                context.supportFragmentManager
                    .findFragmentByTag(TAG)!!.view!!.findViewById<View>(
                        R.id.sm_feedback_webview
                    ).visibility = View.VISIBLE
            }
        }
    }

    companion object {
        @JvmStatic
        val TAG = SMFeedbackFragmentNew::class.java.simpleName
        private const val RESPONDENT_TOKEN_LOADER_KEY = 1
        private const val RESPONDENT_LOADER_KEY = 2
        const val KEY_SM_SPAGE_URL = "smSPageURL"
        const val KEY_SM_SPAGE_HTML = "smSPageHTML"
        const val KEY_SM_HAS_LOADED_SPAGE_HTML = "smHasLoadedSPageHTML"
        private const val SURVEY_STATUS = "survey_status"
        private const val HTML = "html"
        private const val COLLECTOR_CLOSED = "collector_closed"
        @JvmStatic
        fun newInstance(
            url: String?,
            spageHTML: String?,
            hasLoadedHTML: Boolean
        ): SMFeedbackFragmentNew {
            val fragment = SMFeedbackFragmentNew()
            val bundle = Bundle()
            bundle.putString(KEY_SM_SPAGE_URL, url)
            bundle.putString(KEY_SM_SPAGE_HTML, spageHTML)
            bundle.putBoolean(KEY_SM_HAS_LOADED_SPAGE_HTML, hasLoadedHTML)
            fragment.arguments = bundle
            return fragment
        }
    }
}