package com.stripe.android.stripe3ds2.transaction

import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelProvider
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.security.MessageTransformer
import com.stripe.android.stripe3ds2.security.MessageTransformerFactory
import com.stripe.android.stripe3ds2.transactions.ChallengeRequestData
import java.security.KeyPair
import java.security.cert.X509Certificate
import kotlin.coroutines.CoroutineContext

internal class TransactionViewModel(
    private val messageVersionRegistry: MessageVersionRegistry,
    private val sdkTransactionId: SdkTransactionId,
    private val sdkReferenceNumber: String,
    private val challengeParameters: ChallengeParameters,
    private val jwsValidator: JwsValidator,
    private val sdkKeyPair: KeyPair,
    private val messageTransformer: MessageTransformer,
    private val acsDataParser: AcsDataParser,
    private val challengeRequestResultRepository: ChallengeRequestResultRepository,
    private val errorRequestExecutorFactory: ErrorRequestExecutor.Factory,
    private val logger: Logger,
    private val errorReporter: ErrorReporter
) : ViewModel() {

    /**
     * Make the initial challenge request and return a [ChallengeRequestResult] representing the
     * result.
     */
    suspend fun startChallenge(): ChallengeRequestResult {
        logger.info("Make initial challenge request.")

        return runCatching {
            // will throw exception if acsSignedContent fails verification
            val (acsUrl, acsEphemPubKey) = getAcsData(
                requireNotNull(challengeParameters.acsSignedContent),
            )

            val creqData = createCreqData(challengeParameters)

            val errorRequestExecutor = errorRequestExecutorFactory.create(acsUrl, errorReporter)

            val creqExecutorConfig = ChallengeRequestExecutor.Config(
                messageTransformer,
                sdkReferenceNumber,
                creqData,
                acsUrl,
                ChallengeRequestExecutor.Config.Keys(
                    sdkKeyPair.private.encoded,
                    acsEphemPubKey.encoded
                )
            )

            val challengeRequestResult = challengeRequestResultRepository.get(
                creqExecutorConfig,
                creqData
            )

            // log errors
            when (challengeRequestResult) {
                is ChallengeRequestResult.ProtocolError -> {
                    errorRequestExecutor.executeAsync(challengeRequestResult.data)
                }
                is ChallengeRequestResult.Timeout -> {
                    errorRequestExecutor.executeAsync(challengeRequestResult.data)
                }
                else -> {
                    // no-op
                }
            }

            challengeRequestResult
        }.getOrElse {
            errorReporter.reportError(it)
            logger.error("Exception during challenge flow.", it)

            ChallengeRequestResult.RuntimeError(it)
        }
    }

    private fun getAcsData(acsSignedContent: String): AcsData {
        return acsDataParser.parse(
            jwsValidator.getPayload(acsSignedContent)
        )
    }

    private fun createCreqData(
        challengeParameters: ChallengeParameters
    ) = ChallengeRequestData(
        acsTransId = requireNotNull(challengeParameters.acsTransactionId),
        threeDsServerTransId = requireNotNull(challengeParameters.threeDsServerTransactionId),
        sdkTransId = sdkTransactionId,
        messageVersion = messageVersionRegistry.current
    )
}

internal fun interface ChallengeRequestResultRepository {
    suspend fun get(
        creqExecutorConfig: ChallengeRequestExecutor.Config,
        challengeRequestData: ChallengeRequestData
    ): ChallengeRequestResult
}

internal class DefaultChallengeRequestResultRepository(
    private val errorReporter: ErrorReporter,
    private val workContext: CoroutineContext
) : ChallengeRequestResultRepository {
    override suspend fun get(
        creqExecutorConfig: ChallengeRequestExecutor.Config,
        challengeRequestData: ChallengeRequestData
    ): ChallengeRequestResult {
        return StripeChallengeRequestExecutor.Factory(creqExecutorConfig)
            .create(errorReporter, workContext)
            .execute(challengeRequestData)
    }
}

internal class TransactionViewModelFactory(
    private val sdkTransactionId: SdkTransactionId,
    private val sdkReferenceNumber: String,
    private val isLiveMode: Boolean,
    private val challengeParameters: ChallengeParameters,
    private val rootCerts: List<X509Certificate>,
    private val sdkKeyPair: KeyPair,
    private val logger: Logger,
    private val errorReporter: ErrorReporter,
    private val workContext: CoroutineContext
) : ViewModelProvider.Factory {

    @Suppress("UNCHECKED_CAST")
    override fun <T : ViewModel?> create(modelClass: Class<T>): T {
        return TransactionViewModel(
            MessageVersionRegistry(),
            sdkTransactionId,
            sdkReferenceNumber,
            challengeParameters,
            DefaultJwsValidator(isLiveMode, rootCerts, errorReporter),
            sdkKeyPair,
            MessageTransformerFactory(errorReporter).create(isLiveMode),
            DefaultAcsDataParser(errorReporter),
            DefaultChallengeRequestResultRepository(errorReporter, workContext),
            StripeErrorRequestExecutor.Factory(workContext),
            logger = logger,
            errorReporter = errorReporter
        ) as T
    }
}
