package com.stripe.android.stripe3ds2.views

import android.app.Dialog
import android.content.Intent
import android.os.Bundle
import android.view.WindowManager
import android.webkit.WebView
import android.widget.CheckBox
import androidx.activity.viewModels
import androidx.annotation.IdRes
import androidx.appcompat.app.AppCompatActivity
import androidx.core.os.bundleOf
import androidx.fragment.app.commit
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.stripe.android.stripe3ds2.databinding.StripeChallengeActivityBinding
import com.stripe.android.stripe3ds2.databinding.StripeChallengeFragmentBinding
import com.stripe.android.stripe3ds2.init.ui.UiCustomization
import com.stripe.android.stripe3ds2.observability.DefaultErrorReporter
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.observability.Stripe3ds2ErrorReporterConfig
import com.stripe.android.stripe3ds2.transaction.ChallengeAction
import com.stripe.android.stripe3ds2.transaction.ChallengeActionHandler
import com.stripe.android.stripe3ds2.transaction.ChallengeStatusReceiver
import com.stripe.android.stripe3ds2.transaction.ChallengeStatusReceiverProvider
import com.stripe.android.stripe3ds2.transaction.ErrorRequestExecutor
import com.stripe.android.stripe3ds2.transaction.ImmediateTimeoutTransactionTimer
import com.stripe.android.stripe3ds2.transaction.NoopChallengeStatusReceiver
import com.stripe.android.stripe3ds2.transaction.SdkTransactionId
import com.stripe.android.stripe3ds2.transaction.TransactionTimer
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData
import com.ults.listeners.BaseSdkChallenge
import com.ults.listeners.ChallengeType
import com.ults.listeners.SdkChallengeInterface
import com.ults.listeners.challenges.MultiSelectChallenge
import com.ults.listeners.challenges.SingleSelectChallenge
import com.ults.listeners.challenges.TextChallenge
import com.ults.listeners.challenges.WebChallenge
import kotlinx.coroutines.Dispatchers

class ChallengeActivity :
    AppCompatActivity(),
    TextChallenge,
    SingleSelectChallenge,
    MultiSelectChallenge,
    WebChallenge,
    SdkChallengeInterface {

    private val sdkTransactionId: SdkTransactionId by lazy {
        viewArgs.sdkTransactionId
    }

    private val transactionTimer: TransactionTimer by lazy {
        // immediately time-out if `ChallengeStatusReceiver` is unavailable
        challengeStatusReceiverResult.fold(
            onSuccess = {
                TransactionTimer.Default(
                    challengeStatusReceiver,
                    viewArgs.timeoutMins,
                    errorRequestExecutor,
                    viewArgs.creqData,
                    Dispatchers.IO
                )
            },
            onFailure = {
                ImmediateTimeoutTransactionTimer()
            }
        )
    }

    private val challengeStatusReceiverResult: Result<ChallengeStatusReceiver> by lazy {
        runCatching {
            ChallengeStatusReceiverProvider.Default[sdkTransactionId]
        }.onFailure {
            errorReporter.reportError(it)
        }
    }

    private val challengeStatusReceiver: ChallengeStatusReceiver by lazy {
        challengeStatusReceiverResult.getOrDefault(NoopChallengeStatusReceiver())
    }

    private val errorReporter: ErrorReporter by lazy {
        DefaultErrorReporter(
            applicationContext,
            Stripe3ds2ErrorReporterConfig(sdkTransactionId)
        )
    }

    internal val fragment: ChallengeFragment by lazy {
        (supportFragmentManager.findFragmentById(fragmentContainerId) as ChallengeFragment)
    }
    internal val fragmentViewBinding: StripeChallengeFragmentBinding by lazy {
        fragment.viewBinding
    }

    internal val viewBinding: StripeChallengeActivityBinding by lazy {
        StripeChallengeActivityBinding.inflate(layoutInflater)
    }

    private val informationZoneView: InformationZoneView by lazy {
        fragmentViewBinding.caInformationZone
    }

    private val challengeActionHandler: ChallengeActionHandler by lazy {
        ChallengeActionHandler.Default(
            viewArgs.creqData,
            errorReporter,
            viewArgs.creqExecutorFactory,
            viewArgs.creqExecutorConfig,
            Dispatchers.IO
        )
    }

    private val errorRequestExecutor: ErrorRequestExecutor by lazy {
        viewArgs.errorExecutorFactory.create(
            viewArgs.creqExecutorConfig.acsUrl,
            errorReporter
        )
    }

    internal val viewModel: ChallengeActivityViewModel by viewModels {
        ChallengeActivityViewModel.Factory(
            application,
            challengeActionHandler,
            transactionTimer,
            errorReporter,
            Dispatchers.IO
        )
    }

    private val viewArgs: ChallengeViewArgs by lazy {
        ChallengeViewArgs.create(intent.extras)
    }

    private val keyboardController: KeyboardController by lazy {
        KeyboardController(this)
    }

    private val progressDialogFactory: ChallengeSubmitDialogFactory by lazy {
        ChallengeSubmitDialogFactory(this, viewArgs.uiCustomization)
    }

    private var progressDialog: Dialog? = null

    private val fragmentContainerId: Int
        @IdRes
        get() = viewBinding.fragmentContainer.id

    override fun onCreate(savedInstanceState: Bundle?) {
        supportFragmentManager.fragmentFactory = ChallengeFragmentFactory(
            sdkTransactionId = sdkTransactionId,
            uiCustomization = viewArgs.uiCustomization,
            transactionTimer = transactionTimer,
            challengeStatusReceiver = challengeStatusReceiver,
            errorRequestExecutor = errorRequestExecutor,
            errorReporter = errorReporter,
            challengeActionHandler = challengeActionHandler
        )

        super.onCreate(savedInstanceState)

        window.setFlags(
            WindowManager.LayoutParams.FLAG_SECURE,
            WindowManager.LayoutParams.FLAG_SECURE
        )

        setContentView(viewBinding.root)

        viewModel.submitClicked.observe(this) { challengeAction ->
            if (!isFinishing) {
                dismissKeyboard()

                progressDialog = progressDialogFactory.create().also {
                    it.show()
                }

                viewModel.submit(challengeAction)
            }
        }

        viewModel.shouldFinish.observe(this) {
            if (!isFinishing) {
                finish()
            }
        }

        configureHeaderZone()

        viewModel.nextScreen.observe(this) { cres ->
            dismissDialog()

            if (cres != null) {
                startFragment(cres)
            }
        }

        if (savedInstanceState == null) {
            viewModel.onNextScreen(viewArgs.cresData)
        }
    }

    private fun startFragment(
        cres: ChallengeResponseData
    ) {
        supportFragmentManager.commit {
            setCustomAnimations(
                AnimationConstants.SLIDE_IN,
                AnimationConstants.SLIDE_OUT,
                AnimationConstants.SLIDE_IN,
                AnimationConstants.SLIDE_OUT
            )

            replace(
                fragmentContainerId,
                ChallengeFragment::class.java,
                bundleOf(ChallengeFragment.ARG_CRES to cres)
            )
        }
    }

    override fun onLowMemory() {
        super.onLowMemory()
        viewModel.onMemoryEvent()
    }

    override fun onTrimMemory(level: Int) {
        super.onTrimMemory(level)
        viewModel.onMemoryEvent()
    }

    private fun dismissKeyboard() {
        keyboardController.hide()
    }

    override fun onPause() {
        super.onPause()
        viewModel.shouldRefreshUi = true
        dismissKeyboard()
    }

    override fun onResume() {
        super.onResume()

        if (viewModel.shouldRefreshUi) {
            viewModel.onRefreshUi()
        } else {
            LocalBroadcastManager.getInstance(this)
                .sendBroadcast(Intent().setAction(SdkChallengeInterface.UL_HANDLE_CHALLENGE_ACTION))
        }

        viewModel.getTimeout().observe(this) { isTimeout ->
            if (isTimeout == true) {
                viewModel.onFinish()
            }
        }
    }

    private fun configureHeaderZone() {
        val headerZoneCustomizer = HeaderZoneCustomizer(this)
        val cancelButton = headerZoneCustomizer.customize(
            viewArgs.uiCustomization.toolbarCustomization,
            viewArgs.uiCustomization.getButtonCustomization(UiCustomization.ButtonType.CANCEL)
        )
        cancelButton?.setOnClickListener {
            cancelButton.isClickable = false
            viewModel.submit(ChallengeAction.Cancel)
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        dismissDialog()
    }

    private fun dismissDialog() {
        progressDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        progressDialog = null
    }

    override fun onBackPressed() {
        // do not call super.onBackPressed() - handle tap on system back button as a tap on the
        // cancel button
        viewModel.submit(ChallengeAction.Cancel)
    }

    override fun typeTextChallengeValue(s: String) {
        viewModel.updateChallengeText(s)
    }

    override fun clickSubmitButton() {
        fragment.clickSubmitButton()
    }

    override fun clickCancelButton() {
        viewModel.submit(ChallengeAction.Cancel)
    }

    override fun getChallengeType(): ChallengeType? {
        return viewArgs.cresData.uiType?.challengeType
    }

    override fun expandTextsBeforeScreenshot() {
        informationZoneView.expandViews()
    }

    override fun selectObject(i: Int) {
        fragment.selectChallengeOption(i)
    }

    override fun getCheckboxesOrdered(): Array<CheckBox?>? = fragment.getCheckboxesOrdered()

    override fun getWebView(): WebView? = fragment.getWebView()

    override fun getCurrentChallenge(): BaseSdkChallenge {
        return this
    }
}
