package com.stripe.android.stripe3ds2.service

import android.content.Context
import androidx.annotation.VisibleForTesting
import com.stripe.android.stripe3ds2.exceptions.InvalidInputException
import com.stripe.android.stripe3ds2.exceptions.SDKAlreadyInitializedException
import com.stripe.android.stripe3ds2.exceptions.SDKNotInitializedException
import com.stripe.android.stripe3ds2.exceptions.SDKRuntimeException
import com.stripe.android.stripe3ds2.init.DefaultAppInfoRepository
import com.stripe.android.stripe3ds2.init.DeviceDataFactoryImpl
import com.stripe.android.stripe3ds2.init.DeviceParamNotAvailableFactoryImpl
import com.stripe.android.stripe3ds2.init.HardwareIdSupplier
import com.stripe.android.stripe3ds2.init.SecurityChecker
import com.stripe.android.stripe3ds2.init.Warning
import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.init.ui.UiCustomization
import com.stripe.android.stripe3ds2.observability.DefaultErrorReporter
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.security.EphemeralKeyPairGenerator
import com.stripe.android.stripe3ds2.security.PublicKeyFactory
import com.stripe.android.stripe3ds2.security.StripeEphemeralKeyPairGenerator
import com.stripe.android.stripe3ds2.transaction.ChallengeStatusReceiverProvider
import com.stripe.android.stripe3ds2.transaction.DefaultAuthenticationRequestParametersFactory
import com.stripe.android.stripe3ds2.transaction.Logger
import com.stripe.android.stripe3ds2.transaction.MessageVersionRegistry
import com.stripe.android.stripe3ds2.transaction.SdkTransactionId
import com.stripe.android.stripe3ds2.transaction.Transaction
import com.stripe.android.stripe3ds2.transaction.TransactionFactory
import com.stripe.android.stripe3ds2.transaction.TransactionTimerProvider
import com.stripe.android.stripe3ds2.utils.ImageCache
import com.stripe.android.stripe3ds2.utils.ParcelUtils
import com.stripe.android.stripe3ds2.views.ProgressViewFactory
import java.security.PublicKey
import java.security.cert.X509Certificate
import java.util.concurrent.atomic.AtomicBoolean

class StripeThreeDs2ServiceImpl @VisibleForTesting internal constructor(
    private val isInitialized: AtomicBoolean,
    private val securityChecker: SecurityChecker,
    private val publicKeyFactory: PublicKeyFactory,
    private val messageVersionRegistry: MessageVersionRegistry,
    private val imageCache: ImageCache,
    private val challengeStatusReceiverProvider: ChallengeStatusReceiverProvider,
    private val transactionTimerProvider: TransactionTimerProvider,
    private val errorReporter: ErrorReporter,
    private val transactionFactory: TransactionFactory
) : StripeThreeDs2Service {

    @VisibleForTesting
    internal var uiCustomization: StripeUiCustomization? = null

    override val sdkVersion: String
        @Throws(SDKNotInitializedException::class, SDKRuntimeException::class)
        get() {
            requireInitialization()
            return SDK_VERSION
        }

    override val warnings: List<Warning>
        get() = securityChecker.warnings

    @JvmOverloads
    constructor(
        context: Context,
        enableLogging: Boolean = false
    ) : this(
        context,
        STRIPE_SDK_REFERENCE_NUMBER,
        enableLogging
    )

    constructor(
        context: Context,
        sdkReferenceNumber: String,
        enableLogging: Boolean = false
    ) : this(
        context,
        ImageCache.Default,
        ChallengeStatusReceiverProvider.Default,
        TransactionTimerProvider.Default,
        sdkReferenceNumber,

        logger = if (enableLogging) {
            Logger.real()
        } else {
            Logger.noop()
        }
    )

    private constructor(
        context: Context,
        imageCache: ImageCache,
        challengeStatusReceiverProvider: ChallengeStatusReceiverProvider,
        transactionTimerProvider: TransactionTimerProvider,
        sdkReferenceNumber: String,
        logger: Logger
    ) : this(
        context,
        imageCache,
        challengeStatusReceiverProvider,
        transactionTimerProvider,
        sdkReferenceNumber,
        logger,

        // TODO(mshafrir): add config
        DefaultErrorReporter(
            context = context.applicationContext,
            logger = logger
        )
    )

    private constructor(
        context: Context,
        imageCache: ImageCache,
        challengeStatusReceiverProvider: ChallengeStatusReceiverProvider,
        transactionTimerProvider: TransactionTimerProvider,
        sdkReferenceNumber: String,
        logger: Logger,
        errorReporter: ErrorReporter
    ) : this(
        context,
        imageCache,
        challengeStatusReceiverProvider,
        transactionTimerProvider,
        sdkReferenceNumber,
        logger,
        errorReporter,
        StripeEphemeralKeyPairGenerator(errorReporter),
        HardwareIdSupplier(context),
        SecurityChecker.Default(),
        MessageVersionRegistry()
    )

    private constructor(
        context: Context,
        imageCache: ImageCache,
        challengeStatusReceiverProvider: ChallengeStatusReceiverProvider,
        transactionTimerProvider: TransactionTimerProvider,
        sdkReferenceNumber: String,
        logger: Logger,
        errorReporter: ErrorReporter,
        ephemeralKeyPairGenerator: EphemeralKeyPairGenerator,
        hardwareIdSupplier: HardwareIdSupplier,
        securityChecker: SecurityChecker,
        messageVersionRegistry: MessageVersionRegistry
    ) : this(
        isInitialized = AtomicBoolean(false),
        securityChecker = securityChecker,
        publicKeyFactory = PublicKeyFactory(context, errorReporter),
        messageVersionRegistry = messageVersionRegistry,
        imageCache = imageCache,
        challengeStatusReceiverProvider = challengeStatusReceiverProvider,
        transactionTimerProvider = transactionTimerProvider,
        errorReporter = errorReporter,
        transactionFactory = TransactionFactory.Default(
            DefaultAuthenticationRequestParametersFactory(
                DeviceDataFactoryImpl(
                    context = context.applicationContext,
                    hardwareIdSupplier = hardwareIdSupplier
                ),
                DeviceParamNotAvailableFactoryImpl(
                    hardwareIdSupplier
                ),
                securityChecker,
                ephemeralKeyPairGenerator,
                DefaultAppInfoRepository(context),
                messageVersionRegistry,
                sdkReferenceNumber,
                errorReporter
            ),
            ephemeralKeyPairGenerator,
            messageVersionRegistry,
            sdkReferenceNumber,
            errorReporter,
            logger
        )
    )

    @Throws(
        InvalidInputException::class, SDKAlreadyInitializedException::class,
        SDKRuntimeException::class
    )
    override fun initialize(uiCustomization: UiCustomization?) {
        if (!isInitialized.compareAndSet(false, true)) {
            throw SDKAlreadyInitializedException()
        }

        this.uiCustomization = when (uiCustomization) {
            is StripeUiCustomization -> copyUiCustomization(uiCustomization)
            null -> null
            else -> throw InvalidInputException(
                RuntimeException("UiCustomization must be an instance of StripeUiCustomization")
            )
        }
    }

    override fun createTransaction(
        directoryServerID: String,
        messageVersion: String?
    ): Transaction {
        return createTransaction(
            directoryServerID,
            messageVersion,
            true,
            "visa"
        )
    }

    @Throws(InvalidInputException::class, SDKNotInitializedException::class, SDKRuntimeException::class)
    override fun createTransaction(
        directoryServerID: String,
        messageVersion: String?,
        isLiveMode: Boolean,
        directoryServerName: String
    ): Transaction {
        val publicKey = publicKeyFactory.create(directoryServerID)
        return createTransaction(
            directoryServerID = directoryServerID,
            messageVersion = messageVersion,
            isLiveMode = isLiveMode,
            directoryServerName = directoryServerName,
            rootCerts = emptyList(),
            dsPublicKey = publicKey,
            keyId = null,
            sdkTransactionId = SdkTransactionId.create()
        )
    }

    @Throws(
        InvalidInputException::class, SDKNotInitializedException::class,
        SDKRuntimeException::class
    )
    override fun createTransaction(
        directoryServerID: String,
        messageVersion: String?,
        isLiveMode: Boolean,
        directoryServerName: String,
        rootCerts: List<X509Certificate>,
        dsPublicKey: PublicKey,
        keyId: String?
    ): Transaction {
        return createTransaction(
            directoryServerID,
            messageVersion,
            isLiveMode,
            directoryServerName,
            rootCerts,
            dsPublicKey,
            keyId,
            sdkTransactionId = SdkTransactionId.create()
        )
    }

    private fun createTransaction(
        directoryServerID: String,
        messageVersion: String?,
        isLiveMode: Boolean,
        directoryServerName: String,
        rootCerts: List<X509Certificate>,
        dsPublicKey: PublicKey,
        keyId: String?,
        sdkTransactionId: SdkTransactionId
    ): Transaction {
        requireInitialization()
        if (!messageVersionRegistry.isSupported(messageVersion)) {
            throw InvalidInputException("Message version is unsupported: ${messageVersion.orEmpty()}")
        }

        return transactionFactory.create(
            directoryServerID,
            rootCerts,
            dsPublicKey,
            keyId,
            sdkTransactionId,
            uiCustomization,
            isLiveMode,
            ProgressViewFactory.Brand.lookup(
                directoryServerName,
                errorReporter
            )
        )
    }

    @Throws(SDKNotInitializedException::class)
    override fun cleanup() {
        requireInitialization()
        imageCache.clear()
        challengeStatusReceiverProvider.clear()
        transactionTimerProvider.clear()
    }

    private fun requireInitialization() {
        if (!isInitialized.get()) {
            throw SDKNotInitializedException()
        }
    }

    private fun copyUiCustomization(uiCustomization: StripeUiCustomization): StripeUiCustomization {
        return ParcelUtils.copy(uiCustomization, StripeUiCustomization.CREATOR)
    }

    private companion object {
        private const val STRIPE_SDK_REFERENCE_NUMBER = "3DS_LOA_SDK_STIN_020100_00142"
        private const val SDK_VERSION = "1.0.0"
    }
}
