package com.stripe.android.stripe3ds2.views

import android.content.Context
import android.content.Intent
import android.os.Bundle
import android.view.WindowManager
import android.view.inputmethod.InputMethodManager
import android.webkit.WebView
import android.widget.CheckBox
import androidx.activity.viewModels
import androidx.annotation.VisibleForTesting
import androidx.appcompat.app.AppCompatActivity
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.stripe.android.stripe3ds2.databinding.StripeChallengeActivityBinding
import com.stripe.android.stripe3ds2.observability.DefaultErrorReporter
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.observability.Stripe3ds2ErrorReporterConfig
import com.stripe.android.stripe3ds2.transaction.ChallengeActionHandler
import com.stripe.android.stripe3ds2.transaction.ChallengeStatusReceiver
import com.stripe.android.stripe3ds2.transaction.ChallengeStatusReceiverProvider
import com.stripe.android.stripe3ds2.transaction.ImmediateTimeoutTransactionTimer
import com.stripe.android.stripe3ds2.transaction.NoopChallengeStatusReceiver
import com.stripe.android.stripe3ds2.transaction.TransactionTimer
import com.stripe.android.stripe3ds2.transaction.TransactionTimerProvider
import com.ults.listeners.BaseSdkChallenge
import com.ults.listeners.ChallengeType
import com.ults.listeners.SdkChallengeInterface
import com.ults.listeners.challenges.MultiSelectChallenge
import com.ults.listeners.challenges.SingleSelectChallenge
import com.ults.listeners.challenges.TextChallenge
import com.ults.listeners.challenges.WebChallenge
import kotlinx.coroutines.Dispatchers

/**
 * See [ChallengePresenter] for business logic
 */
class ChallengeActivity :
    AppCompatActivity(),
    TextChallenge,
    SingleSelectChallenge,
    MultiSelectChallenge,
    WebChallenge,
    SdkChallengeInterface {

    @VisibleForTesting
    internal var refreshUi = false

    private val transactionTimer: TransactionTimer by lazy {
        // immediately time-out if `ChallengeStatusReceiver` or `TransactionTimer` are unavailable
        challengeStatusReceiver.fold(
            onSuccess = {
                runCatching {
                    TransactionTimerProvider.Default[viewArgs.creqData.sdkTransId]
                }.onFailure {
                    errorReporter.reportError(it)
                }.getOrDefault(ImmediateTimeoutTransactionTimer())
            },
            onFailure = {
                ImmediateTimeoutTransactionTimer()
            }
        )
    }

    private val challengeStatusReceiver: Result<ChallengeStatusReceiver> by lazy {
        runCatching {
            ChallengeStatusReceiverProvider.Default[viewArgs.creqData.sdkTransId]
        }.onFailure {
            errorReporter.reportError(it)
        }
    }

    private val errorReporter: ErrorReporter by lazy {
        DefaultErrorReporter(
            applicationContext,
            Stripe3ds2ErrorReporterConfig(viewArgs.sdkTransactionId)
        )
    }

    private val presenter: ChallengePresenter by lazy {
        ChallengePresenter(
            this,
            viewArgs,
            challengeStatusReceiver.getOrDefault(NoopChallengeStatusReceiver()),
            errorReporter,
            viewArgs.errorExecutorFactory,
            viewModel
        )
    }

    internal val viewBinding: StripeChallengeActivityBinding by lazy {
        StripeChallengeActivityBinding.inflate(layoutInflater)
    }

    private val viewModel: ChallengeActivityViewModel by viewModels {
        ChallengeActivityViewModel.Factory(
            application,
            ChallengeActionHandler.Default(
                viewArgs.creqData,
                errorReporter,
                viewArgs.creqExecutorFactory,
                viewArgs.creqExecutorConfig,
                Dispatchers.IO
            ),
            transactionTimer,
            errorReporter,
            Dispatchers.IO
        )
    }

    private val viewArgs: ChallengeViewArgs by lazy {
        ChallengeViewArgs.create(intent.extras)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        window.setFlags(
            WindowManager.LayoutParams.FLAG_SECURE,
            WindowManager.LayoutParams.FLAG_SECURE
        )

        setContentView(viewBinding.root)

        refreshUi = savedInstanceState?.getBoolean(STATE_REFRESH_UI, false) ?: false

        presenter.start()
    }

    override fun onSaveInstanceState(outState: Bundle) {
        super.onSaveInstanceState(outState)
        outState.putBoolean(STATE_REFRESH_UI, true)
    }

    override fun onLowMemory() {
        presenter.clearImageCache()
        super.onLowMemory()
    }

    override fun onTrimMemory(level: Int) {
        presenter.clearImageCache()
        super.onTrimMemory(level)
    }

    internal fun dismissKeyboard() {
        val inputMethodManager = getSystemService(Context.INPUT_METHOD_SERVICE) as? InputMethodManager
        if (inputMethodManager?.isAcceptingText == true) {
            inputMethodManager.hideSoftInputFromWindow(this.currentFocus?.windowToken, 0)
        }
    }

    override fun onPause() {
        refreshUi = true
        dismissKeyboard()
        super.onPause()
    }

    override fun onResume() {
        super.onResume()
        if (refreshUi) {
            presenter.refreshUi()
        } else {
            LocalBroadcastManager.getInstance(this)
                .sendBroadcast(Intent().setAction(SdkChallengeInterface.UL_HANDLE_CHALLENGE_ACTION))
        }

        presenter.onResume()
    }

    override fun onDestroy() {
        presenter.onDestroy()
        super.onDestroy()
    }

    override fun onBackPressed() {
        // do not call super.onBackPressed() - handle tap on system back button as a tap on the
        // cancel button
        presenter.cancelChallengeRequest()
    }

    override fun typeTextChallengeValue(s: String) {
        presenter.updateChallengeText(s)
    }

    override fun clickSubmitButton() {
        presenter.onSubmitClicked()
    }

    override fun clickCancelButton() {
        presenter.onCancelClicked()
    }

    override fun getChallengeType(): ChallengeType? {
        return presenter.challengeType
    }

    override fun expandTextsBeforeScreenshot() {
        presenter.expandInformationZoneViews()
    }

    override fun selectObject(i: Int) {
        presenter.selectChallengeOption(i)
    }

    override fun getCheckboxesOrdered(): Array<CheckBox?>? {
        return presenter.challengeEntryCheckBoxes?.toTypedArray()
    }

    override fun getWebView(): WebView? {
        return presenter.challengeWebView
    }

    override fun getCurrentChallenge(): BaseSdkChallenge {
        return this
    }

    internal companion object {
        @VisibleForTesting
        internal val STATE_REFRESH_UI = "refresh_ui"
    }
}
