/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AbortMultipartUploadRequest;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CompleteMultipartUploadRequest;
import com.amazonaws.services.s3.model.CopyObjectRequest;
import com.amazonaws.services.s3.model.InitiateMultipartUploadRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.PartETag;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.UploadPartRequest;
import com.amazonaws.services.s3.model.UploadPartResult;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.NoSuchFileException;
import java.security.AccessController;
import java.util.ArrayList;
import java.util.Map;
import org.apache.lucene.util.SetOnce;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.BlobStoreException;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.common.collect.MapBuilder;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.repositories.s3.S3BlobStore;
import org.elasticsearch.repositories.s3.S3Repository;
import org.elasticsearch.repositories.s3.SocketAccess;

class S3BlobContainer
extends AbstractBlobContainer {
    private final S3BlobStore blobStore;
    private final String keyPath;

    S3BlobContainer(BlobPath path, S3BlobStore blobStore) {
        super(path);
        this.blobStore = blobStore;
        this.keyPath = path.buildAsString();
    }

    public boolean blobExists(String blobName) {
        try {
            return SocketAccess.doPrivileged(() -> this.blobStore.client().doesObjectExist(this.blobStore.bucket(), this.buildKey(blobName)));
        }
        catch (Exception e) {
            throw new BlobStoreException("Failed to check if blob [" + blobName + "] exists", (Throwable)e);
        }
    }

    public InputStream readBlob(String blobName) throws IOException {
        try {
            S3Object s3Object = SocketAccess.doPrivileged(() -> this.blobStore.client().getObject(this.blobStore.bucket(), this.buildKey(blobName)));
            return s3Object.getObjectContent();
        }
        catch (AmazonClientException e) {
            if (e instanceof AmazonS3Exception && 404 == ((AmazonS3Exception)e).getStatusCode()) {
                throw new NoSuchFileException("Blob object [" + blobName + "] not found: " + e.getMessage());
            }
            throw e;
        }
    }

    public void writeBlob(String blobName, InputStream inputStream, long blobSize) throws IOException {
        if (this.blobExists(blobName)) {
            throw new FileAlreadyExistsException("Blob [" + blobName + "] already exists, cannot overwrite");
        }
        SocketAccess.doPrivilegedIOException(() -> {
            if (blobSize <= this.blobStore.bufferSizeInBytes()) {
                this.executeSingleUpload(this.blobStore, this.buildKey(blobName), inputStream, blobSize);
            } else {
                this.executeMultipartUpload(this.blobStore, this.buildKey(blobName), inputStream, blobSize);
            }
            return null;
        });
    }

    public void deleteBlob(String blobName) throws IOException {
        if (!this.blobExists(blobName)) {
            throw new NoSuchFileException("Blob [" + blobName + "] does not exist");
        }
        try {
            SocketAccess.doPrivilegedVoid(() -> this.blobStore.client().deleteObject(this.blobStore.bucket(), this.buildKey(blobName)));
        }
        catch (AmazonClientException e) {
            throw new IOException("Exception when deleting blob [" + blobName + "]", e);
        }
    }

    public Map<String, BlobMetaData> listBlobsByPrefix(@Nullable String blobNamePrefix) throws IOException {
        return AccessController.doPrivileged(() -> {
            MapBuilder blobsBuilder = MapBuilder.newMapBuilder();
            AmazonS3 client = this.blobStore.client();
            SocketAccess.doPrivilegedVoid(() -> {
                ObjectListing prevListing = null;
                while (true) {
                    ObjectListing list = prevListing != null ? client.listNextBatchOfObjects(prevListing) : (blobNamePrefix != null ? client.listObjects(this.blobStore.bucket(), this.buildKey(blobNamePrefix)) : client.listObjects(this.blobStore.bucket(), this.keyPath));
                    for (S3ObjectSummary summary : list.getObjectSummaries()) {
                        String name = summary.getKey().substring(this.keyPath.length());
                        blobsBuilder.put((Object)name, (Object)new PlainBlobMetaData(name, summary.getSize()));
                    }
                    if (!list.isTruncated()) break;
                    prevListing = list;
                }
            });
            return blobsBuilder.immutableMap();
        });
    }

    public void move(String sourceBlobName, String targetBlobName) throws IOException {
        try {
            CopyObjectRequest request = new CopyObjectRequest(this.blobStore.bucket(), this.buildKey(sourceBlobName), this.blobStore.bucket(), this.buildKey(targetBlobName));
            if (this.blobStore.serverSideEncryption()) {
                ObjectMetadata objectMetadata = new ObjectMetadata();
                objectMetadata.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
                request.setNewObjectMetadata(objectMetadata);
            }
            SocketAccess.doPrivilegedVoid(() -> {
                this.blobStore.client().copyObject(request);
                this.blobStore.client().deleteObject(this.blobStore.bucket(), this.buildKey(sourceBlobName));
            });
        }
        catch (AmazonS3Exception e) {
            throw new IOException(e);
        }
    }

    public Map<String, BlobMetaData> listBlobs() throws IOException {
        return this.listBlobsByPrefix(null);
    }

    private String buildKey(String blobName) {
        return this.keyPath + blobName;
    }

    void executeSingleUpload(S3BlobStore blobStore, String blobName, InputStream input, long blobSize) throws IOException {
        if (blobSize > S3Repository.MAX_FILE_SIZE.getBytes()) {
            throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than " + S3Repository.MAX_FILE_SIZE);
        }
        if (blobSize > blobStore.bufferSizeInBytes()) {
            throw new IllegalArgumentException("Upload request size [" + blobSize + "] can't be larger than buffer size");
        }
        try {
            ObjectMetadata md = new ObjectMetadata();
            md.setContentLength(blobSize);
            if (blobStore.serverSideEncryption()) {
                md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
            }
            PutObjectRequest putRequest = new PutObjectRequest(blobStore.bucket(), blobName, input, md);
            putRequest.setStorageClass(blobStore.getStorageClass());
            putRequest.setCannedAcl(blobStore.getCannedACL());
            blobStore.client().putObject(putRequest);
        }
        catch (AmazonClientException e) {
            throw new IOException("Unable to upload object [" + blobName + "] using a single upload", e);
        }
    }

    void executeMultipartUpload(S3BlobStore blobStore, String blobName, InputStream input, long blobSize) throws IOException {
        if (blobSize > S3Repository.MAX_FILE_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be larger than " + S3Repository.MAX_FILE_SIZE_USING_MULTIPART);
        }
        if (blobSize < S3Repository.MIN_PART_SIZE_USING_MULTIPART.getBytes()) {
            throw new IllegalArgumentException("Multipart upload request size [" + blobSize + "] can't be smaller than " + S3Repository.MIN_PART_SIZE_USING_MULTIPART);
        }
        long partSize = blobStore.bufferSizeInBytes();
        Tuple<Long, Long> multiparts = S3BlobContainer.numberOfMultiparts(blobSize, partSize);
        if ((Long)multiparts.v1() > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Too many multipart upload requests, maybe try a larger buffer size?");
        }
        int nbParts = ((Long)multiparts.v1()).intValue();
        long lastPartSize = (Long)multiparts.v2();
        assert (blobSize == (long)(nbParts - 1) * partSize + lastPartSize) : "blobSize does not match multipart sizes";
        SetOnce uploadId = new SetOnce();
        String bucketName = blobStore.bucket();
        boolean success = false;
        try {
            InitiateMultipartUploadRequest initRequest = new InitiateMultipartUploadRequest(bucketName, blobName);
            initRequest.setStorageClass(blobStore.getStorageClass());
            initRequest.setCannedACL(blobStore.getCannedACL());
            if (blobStore.serverSideEncryption()) {
                ObjectMetadata md = new ObjectMetadata();
                md.setSSEAlgorithm(ObjectMetadata.AES_256_SERVER_SIDE_ENCRYPTION);
                initRequest.setObjectMetadata(md);
            }
            uploadId.set((Object)blobStore.client().initiateMultipartUpload(initRequest).getUploadId());
            if (Strings.isEmpty((CharSequence)((CharSequence)uploadId.get()))) {
                throw new IOException("Failed to initialize multipart upload " + blobName);
            }
            ArrayList<PartETag> parts = new ArrayList<PartETag>();
            long bytesCount = 0L;
            for (int i = 1; i <= nbParts; ++i) {
                UploadPartRequest uploadRequest = new UploadPartRequest();
                uploadRequest.setBucketName(bucketName);
                uploadRequest.setKey(blobName);
                uploadRequest.setUploadId((String)uploadId.get());
                uploadRequest.setPartNumber(i);
                uploadRequest.setInputStream(input);
                if (i < nbParts) {
                    uploadRequest.setPartSize(partSize);
                    uploadRequest.setLastPart(false);
                } else {
                    uploadRequest.setPartSize(lastPartSize);
                    uploadRequest.setLastPart(true);
                }
                bytesCount += uploadRequest.getPartSize();
                UploadPartResult uploadResponse = blobStore.client().uploadPart(uploadRequest);
                parts.add(uploadResponse.getPartETag());
            }
            if (bytesCount != blobSize) {
                throw new IOException("Failed to execute multipart upload for [" + blobName + "], expected " + blobSize + "bytes sent but got " + bytesCount);
            }
            CompleteMultipartUploadRequest complRequest = new CompleteMultipartUploadRequest(bucketName, blobName, (String)uploadId.get(), parts);
            blobStore.client().completeMultipartUpload(complRequest);
            success = true;
        }
        catch (AmazonClientException e) {
            throw new IOException("Unable to upload object [" + blobName + "] using multipart upload", e);
        }
        finally {
            if (!success && Strings.hasLength((String)((String)uploadId.get()))) {
                AbortMultipartUploadRequest abortRequest = new AbortMultipartUploadRequest(bucketName, blobName, (String)uploadId.get());
                blobStore.client().abortMultipartUpload(abortRequest);
            }
        }
    }

    static Tuple<Long, Long> numberOfMultiparts(long totalSize, long partSize) {
        if (partSize <= 0L) {
            throw new IllegalArgumentException("Part size must be greater than zero");
        }
        if (totalSize == 0L || totalSize <= partSize) {
            return Tuple.tuple((Object)1L, (Object)totalSize);
        }
        long parts = totalSize / partSize;
        long remaining = totalSize % partSize;
        if (remaining == 0L) {
            return Tuple.tuple((Object)parts, (Object)partSize);
        }
        return Tuple.tuple((Object)(parts + 1L), (Object)remaining);
    }
}

