package com.segway.robot.sdk.perception.sensor;


import android.content.Context;

import com.segway.robot.algo.Pose2D;
import com.segway.robot.algo.tf.AlgoTfData;
import com.segway.robot.algo.tf.AlgoTfRequest;
import com.segway.robot.sdk.base.bind.ServiceBinder;

import java.util.LinkedList;
import java.util.List;


/**
 * The manager is used to get the robot's sensor status, include ultrasonic distance infrared distance
 * and tf data.
 */
public class Sensor extends ISensor implements ServiceBinder {
    private static final String TAG = "Sensor";
    private static Sensor mSensor;


    @Deprecated
    public static final int G1_INFRARED = 0;
    @Deprecated
    public static final int G1_ULTRASONIC = 1;
    @Deprecated
    public static final int GX_INFRARED = 2;
    @Deprecated
    public static final int GX_ULTRASONIC = 3;
    @Deprecated
    public static final int GX_BUMPER = 4;

    /**
     * Use this field to get robot encoder,
     * value type: int
     * value[0]:left encoder
     * value[1]:right encoder
     */
    public static final int ENCODER = 0 << 16;

    /**
     * To get robot wheel speed
     * value type: int
     * value[0]:left wheel speed
     * value[1]:right wheel speed
     */
    public static final int WHEEL_SPEED = 1 << 16;

    /**
     * To get robot base imu
     * value type: float
     * value[0]:pitch angle
     * value[1]:roll angle
     * value[2]:yaw angle
     */
    public static final int BASE_IMU = 2 << 16;

    /**
     * To get robot body imu
     * value type: float
     * value[0]:pitch angle
     * value[1]:roll angle
     * value[2]:yaw angle
     */
    public static final int BODY_IMU = 3 << 16;

    /**
     * To get robot bumper state, only for GX. Now gx have 4 bumper device.
     * value type: int
     */
    public static final int BUMPER = 4 << 16;

    /**
     * To get robot infrared A, now A have 4 infrared.
     * value type: int
     */
    public static final int INFRARED_A = 5 << 16;

    /**
     * To get robot infrared B, now B have 4 infrared.
     * value type: int
     */
    public static final int INFRARED_B = 6 << 16;

    /**
     * To get robot infrared C, now C have 4 infrared.
     * value type: int
     */
    public static final int INFRARED_C = 7 << 16;

    /**
     * To get robot body infrared, have 2 infrared.
     * value type: int
     * value[0]:left infrared
     * value[1]:right infrared
     */
    public static final int INFRARED_BODY = 8 << 16;

    /**
     * To get robot body ultrasonic, have 1 ultrasonic.
     * value type: int
     * value[0]:body ultrasonic
     */
    public static final int ULTRASONIC_BODY = 9 << 16;

    /**
     * Get robot pose, {@link com.segway.robot.algo.Pose2D}
     * value[0]:x
     * value[1]:y
     * value[2]:theta
     * value[3]:linear speed
     * value[4]:angular speed
     */
    public static final int POSE_2D = 10 << 16;

    /**
     * Get robot head world pose.
     * value type: float
     * value[0]:world pitch
     * value[1]:world roll
     * value[2]:world yaw
     */
    public static final int HEAD_WORLD_IMU = 11 << 16;

    public static final int PRESSURE = 12 << 16;

    /**
     * Get robot head joint pitch.
     * value type: float
     * value[0]:joint pitch
     */
    public static final int HEAD_JOINT_PITCH = 13 << 16;

    /**
     * Get robot head joint roll.
     * value type: float
     * value[0]:joint roll
     */
    public static final int HEAD_JOINT_ROLL = 14 << 16;

    /**
     * Get robot head joint yaw.
     * value type: float
     * value[0]:joint yaw
     */
    public static final int HEAD_JOINT_YAW = 15 << 16;

    /**
     * To get robot infrared E, now A have 4 infrared.
     * value type: int
     */
    public static final int INFRARED_E = 16 << 16;

    /**
     * To get robot infrared D, now A have 4 infrared.
     * value type: int
     */
    public static final int INFRARED_D = 17 << 16;

    /**
     * Get motor current of robot site
     * value type: int
     * value[0]: motor lift value
     * value[1]: motor right value
     */
    public static final int MOTOR_L_R = 18 << 16;

    /**
     * Get robot charging status
     * value type: int
     * value[0]: 1 or 0; 1:charging 0:Stop charging
     */
    public static final int CHARGE_STATUS = 19 << 16;

    /**
     * 获取底盘陀螺仪数据
     * value type: int
     * value[0]: x
     * value[1]: y
     * value[2]: z
     */
    public static final int CHASSIS_IMU_GYRO = 16;

    /**
     * 获取底盘加速器数据
     * value type: int
     * value[0]: x
     * value[1]: y
     * value[2]: z
     */
    public static final int CHASSIS_IMU_ACC = 17;

    /**
     * 获取激光雷达数据
     * value type: int  数据以int[]保存。数据为数据头 + 数据体。
     * 数据头占2个int
     * 数据体格式为：{数据类型(1 int),数据大小N(1 int),真实数据(N int)}；
     * value[0]: 目前始终为0，保留字段。
     * value[1]: 数据体的总长度
     * value[...]: 数据体，依次排列。
     *
     * 数据体类型type说明：
     * type = 1 速度 RPM(每分钟转数)
     * type = 2 采样的分辨率 度/1000
     * type = 3 距离 mm
     * type = 4 强度 0-255 值越小越若
     *
     */
    public static final int LIDAR_DATA = 18;

    // robot frame name for get tf frame data
    public static final String WORLD_ODOM_ORIGIN = "world_odom_frame";
    public static final String WORLD_EVIO_ORIGIN = "world_evio_frame";
    public static final String BASE_ODOM_FRAME = "base_center_ground_frame";
    public static final String BASE_POSE_FRAME = "base_center_wheel_axis_frame";
    public static final String NECK_POSE_FRAME = "neck_center_body_internal_frame";
    public static final String HEAD_POSE_Y_FRAME = "neck_center_body_yaw_frame";
    public static final String HEAD_POSE_P_FRAME = "head_center_neck_internal_frame";
    public static final String RS_DEPTH_FRAME = "rsdepth_center_neck_fix_frame";
    public static final String RS_FE_FRAME = "rsfisheye_center_neck_fix_frame";
    public static final String RS_COLOR_FRAME = "rscolor_center_neck_fix_frame";
    public static final String HEAD_POSE_P_R_FRAME = "head_center_neck_pitch_frame";
    public static final String PLATFORM_CAM_FRAME = "platform_center_head_fix_frame";  //+head_pose_roll

    SensorManager mSensorManager;

    private Sensor() {
        mSensorManager = new SensorManager();
    }

    /**
     * Gets an instance of the RobotInfraredManager.
     *
     * @return the RobotInfraredManager instance.
     */
    public static synchronized Sensor getInstance() {
        if (mSensor == null) {
            mSensor = new Sensor();
        }
        return mSensor;
    }

    /**
     * Get robot ultrasonic distance info.
     *
     * @return {@link UltrasonicData}
     */
    public UltrasonicData getUltrasonicDistance() {
        return mSensorManager.getUltrasonicDistance();
    }

    /**
     * Get robot infrared distance info.
     *
     * @return {@link InfraredData}
     */
    public InfraredData getInfraredDistance() {
        return mSensorManager.getInfraredDistance();
    }

    /**
     * Get robot info, include Head (pitch yaw roll...)
     * Base (pitch yaw roll) {@link InfraredData} and {@link UltrasonicData}.
     * use {@link Sensor#getRobotAllSensors()} replace, this method will delete.
     *
     * @return {@link RobotTotalInfo}
     */
    @Deprecated
    public RobotTotalInfo getRobotTotalInfo() {
        return mSensorManager.getRobotTotalInfo();
    }

    /**
     * Get robot Tf (transforms information) data from sourceFrame to targetFrame (based on sourceFrame).
     *
     * @param sourceFrame source {@link Sensor#HEAD_POSE_P_FRAME ...}
     * @param targetFrame target {@link Sensor#HEAD_POSE_P_FRAME ...}
     * @return {@link AlgoTfData}
     */
    public AlgoTfData getTfData(String targetFrame, String sourceFrame, long lookupTimestamp, int timeTrhMs) {
        return mSensorManager.getTfData(targetFrame, sourceFrame, lookupTimestamp, timeTrhMs);
    }

    public AlgoTfData getTfData(AlgoTfRequest req) {
        return mSensorManager.getTfData(req);
    }

    public List<AlgoTfData> getMassiveTfData(List<AlgoTfRequest> requestList) {
        return mSensorManager.getMassiveTfData(requestList);
    }

    public boolean setCustomizationTF(AlgoTfData tfData) {
        return mSensorManager.setCustomizationTF(tfData);
    }

    /**
     * Get robot info, include Head (pitch yaw roll...)
     * Base (pitch yaw roll) {@link InfraredData} and {@link UltrasonicData}.
     *
     * @return {@link RobotAllSensors}
     */
    @Override
    public RobotAllSensors getRobotAllSensors() {
        return mSensorManager.getRobotAllSensors();
    }

    /**
     * get Sensor data {@link SensorData}
     *
     * @param type
     * @return
     */
    @Override
    @Deprecated
    public SensorData[] getSensorData(int type) {
        return mSensorManager.getSensorData(type);
    }

    @Override
    public List<SensorData> querySensorData(List queryList) {
        return mSensorManager.querySensorData(queryList);
    }

    @Override
    public Pose2D sensorDataToPose2D(SensorData sensorData) {
        return mSensorManager.sensorDataToPose2D(sensorData);
    }

    @Override
    public int resetFrameData(String frameName) {
        return mSensorManager.resetFrameData(frameName);
    }

    /**
     * Connect to the robot base service. This method must be called before using the {@link Sensor}
     * APIs.
     *
     * @param context  any Android context.
     * @param listener the bind state listener {@link BindStateListener}.
     * @return true if bind service successfully, otherwise false.
     */
    @Override
    public boolean bindService(Context context, BindStateListener listener) {
        return mSensorManager.bindService(context, listener);
    }

    /**
     * Disconnect from the robot {@link Sensor} service.
     */
    @Override
    public void unbindService() {
        mSensorManager.unbindService();
    }

    /**
     * Get the binding state.
     *
     * @return the binding state.
     */
    @Override
    public boolean isBind() {
        return mSensorManager.isBind();
    }

    @Override
    public BindStateListener getBindStateListener() {
        return mSensorManager.getBindStateListener();
    }


}
