package com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.offline

import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.ShapeDefaults
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalClipboardManager
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import com.paystack.android.ui.R
import com.paystack.android.ui.components.buttons.PrimaryButton
import com.paystack.android.ui.components.buttons.SecondaryButton
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun MpesaOfflineUi(
    amount: String,
    paybillNumber: String,
    accountNumber: String,
    showPaybillNumberCopyIcon: Boolean,
    showAccountNumberCopyIcon: Boolean,
    onShowPayWithLipaSteps: () -> Unit,
    onPaymentCompleted: () -> Unit,
    onSwitchToOnline: () -> Unit,
    onPaybillNumberCopied: () -> Unit,
    onAccountNumberCopied: () -> Unit,
    modifier: Modifier = Modifier
) {
    val clipboardManager = LocalClipboardManager.current

    Column(
        modifier = modifier,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Text(
            text = stringResource(R.string.pstk_mpesa_offline_instruction),
            style = PaystackTheme.typography.body.body16M,
            textAlign = TextAlign.Center,
            modifier = Modifier
                .fillMaxWidth()
                .padding(horizontal = PaystackTheme.dimensions.spacing.triple)
        )

        Spacer(Modifier.height(PaystackTheme.dimensions.spacing.single))

        Text(
            text = stringResource(R.string.pstk_mpesa_offline_with_lipa_instruction),
            style = PaystackTheme.typography.body.body14M,
            textAlign = TextAlign.Center,
            color = PaystackTheme.colors.feedbackColors.information02,
            modifier = Modifier
                .fillMaxWidth()
                .padding(horizontal = PaystackTheme.dimensions.spacing.triple)
                .clickable(
                    interactionSource = MutableInteractionSource(),
                    indication = null,
                    onClick = { onShowPayWithLipaSteps() }
                )
        )

        Spacer(Modifier.height(PaystackTheme.dimensions.spacing.triple))

        MpesaPaybillDetails(
            amount = amount,
            paybillNumber = paybillNumber,
            accountNumber = accountNumber,
            showPaybillNumberCopyIcon = showPaybillNumberCopyIcon,
            showAccountNumberCopyIcon = showAccountNumberCopyIcon,
            onPaybillNumberCopied = { value ->
                clipboardManager.setText(AnnotatedString(value))
                onPaybillNumberCopied()
            },
            onAccountNumberCopied = { value ->
                clipboardManager.setText(AnnotatedString(value))
                onAccountNumberCopied()
            }
        )

        Spacer(Modifier.height(PaystackTheme.dimensions.spacing.triple))

        PrimaryButton(
            buttonText = stringResource(R.string.pstk_mpesa_offline_completed_payment_button_text),
            onClick = onPaymentCompleted,
            isEnabled = true
        )

        Spacer(Modifier.height(PaystackTheme.dimensions.spacing.single))

        SecondaryButton(
            text = stringResource(id = R.string.pstk_switch_to_mpesa_express),
            onClick = onSwitchToOnline,
            modifier = Modifier.fillMaxWidth(),
            showBorder = false,
            contentColor = PaystackTheme.colors.primaryColors.navy01
        )
    }
}

@Composable
private fun MpesaPaybillDetails(
    amount: String,
    paybillNumber: String,
    accountNumber: String,
    showPaybillNumberCopyIcon: Boolean,
    showAccountNumberCopyIcon: Boolean,
    onPaybillNumberCopied: (String) -> Unit,
    onAccountNumberCopied: (String) -> Unit
) {
    Surface(
        color = PaystackTheme.colors.primaryColors.navy06,
        shape = ShapeDefaults.Large,
        modifier = Modifier.fillMaxWidth()
    ) {
        Column(
            modifier = Modifier.padding(
                vertical = PaystackTheme.dimensions.spacing.triple,
                horizontal = PaystackTheme.dimensions.spacing.double
            )
        ) {
            Row(verticalAlignment = Alignment.CenterVertically) {
                Column {
                    Text(
                        text = stringResource(id = R.string.pstk_mpesa_offline_amount_label),
                        style = PaystackTheme.typography.body.body12R,
                        color = PaystackTheme.colors.primaryColors.navy02
                    )

                    Spacer(Modifier.height(PaystackTheme.dimensions.spacing.half))

                    Text(
                        text = amount,
                        style = PaystackTheme.typography.body.body14M,
                        color = PaystackTheme.colors.primaryColors.navy01
                    )
                }
            }

            Spacer(Modifier.height(PaystackTheme.dimensions.spacing.double))

            Row(verticalAlignment = Alignment.CenterVertically) {
                Column {
                    Text(
                        text = stringResource(id = R.string.pstk_paybill_business_number),
                        style = PaystackTheme.typography.body.body12R,
                        color = PaystackTheme.colors.primaryColors.navy02
                    )

                    Spacer(Modifier.height(PaystackTheme.dimensions.spacing.half))

                    Text(
                        text = paybillNumber,
                        style = PaystackTheme.typography.body.body14M,
                        color = PaystackTheme.colors.primaryColors.navy01
                    )
                }

                Column(
                    modifier = Modifier.weight(1f),
                    horizontalAlignment = Alignment.End
                ) {
                    if (!showPaybillNumberCopyIcon) {
                        CopyToClipboardConfirmMessage()
                    } else {
                        IconButton(onClick = { onPaybillNumberCopied(paybillNumber) }) {
                            Icon(
                                painter = painterResource(id = R.drawable.pstk_ic_copy),
                                contentDescription = stringResource(
                                    id = R.string.pstk_icon_copy
                                )
                            )
                        }
                    }
                }
            }

            Spacer(Modifier.height(PaystackTheme.dimensions.spacing.double))

            Row(verticalAlignment = Alignment.CenterVertically) {
                Column {
                    Text(
                        text = stringResource(id = R.string.pstk_paybill_account_number),
                        style = PaystackTheme.typography.body.body12R,
                        color = PaystackTheme.colors.primaryColors.navy02
                    )

                    Spacer(Modifier.height(PaystackTheme.dimensions.spacing.half))

                    Text(
                        text = accountNumber,
                        style = PaystackTheme.typography.body.body14M,
                        color = PaystackTheme.colors.primaryColors.navy01
                    )
                }

                Column(
                    modifier = Modifier.weight(1f),
                    horizontalAlignment = Alignment.End
                ) {
                    if (!showAccountNumberCopyIcon) {
                        CopyToClipboardConfirmMessage()
                    } else {
                        IconButton(onClick = { onAccountNumberCopied(accountNumber) }) {
                            Icon(
                                painter = painterResource(id = R.drawable.pstk_ic_copy),
                                contentDescription = stringResource(
                                    id = R.string.pstk_icon_copy
                                )
                            )
                        }
                    }
                }
            }
        }
    }
}

@Composable
private fun CopyToClipboardConfirmMessage() {
    Text(
        text = stringResource(id = R.string.pstk_clipboard_copy_message),
        style = PaystackTheme.typography.body.body12M,
        color = PaystackTheme.colors.primaryColors.navy02
    )
}

@Preview
@Composable
private fun MpesaOfflineUiPreview() {
    Surface {
        Column(
            modifier = Modifier
                .fillMaxSize()
                .padding(PaystackTheme.dimensions.spacing.double)
        ) {
            MpesaOfflineUi(
                amount = "KES 3",
                paybillNumber = "880100",
                accountNumber = "4666360017",
                showPaybillNumberCopyIcon = false,
                showAccountNumberCopyIcon = false,
                {}, {}, {}, {}, {}
            )
        }
    }
}
