package com.paystack.android.ui.paymentchannels.mobilemoney.mpesa

import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelStore
import androidx.lifecycle.ViewModelStoreOwner
import androidx.lifecycle.viewModelScope
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.PaystackError
import com.paystack.android.core.api.models.TransactionEvent
import com.paystack.android.core.api.models.TransactionStatus
import com.paystack.android.core.logging.Logger
import com.paystack.android.ui.R
import com.paystack.android.ui.data.transaction.TransactionRepository
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.models.MobileMoneyCharge
import com.paystack.android.ui.models.MobileMoneyCharge.Action
import com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.numberform.MpesaNumberFormState
import com.paystack.android.ui.utilities.CountdownTimer
import com.paystack.android.ui.utilities.CurrencyFormatter
import com.paystack.android.ui.utilities.StringProvider
import com.paystack.android.ui.utilities.isFatal
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class MpesaViewModel(
    private val accessCodeData: AccessCodeData,
    private val providerKey: String,
    private val transactionRepository: TransactionRepository,
    private val onPaymentComplete: (Charge) -> Unit,
    private val onError: (Throwable) -> Unit,
    private val stringProvider: StringProvider,
    private val countdownDispatcher: CoroutineDispatcher = Dispatchers.Default
) : ViewModel() {
    private val _mpesaPaymentsState =
        MutableStateFlow<MpesaPaymentState>(MpesaPaymentState.EnterNumber)
    val mpesaPaymentsState: StateFlow<MpesaPaymentState>
        get() = _mpesaPaymentsState

    /*
     * Provide a ViewModelStoreOwner for the different states of the Mpesa Payment flow.
     * This is needed for the individual screens to have their own ViewModelStore for keeping state
     * between configuration changes.
     */
    private val mpesaFlowViewModelStores = mutableMapOf<String, ViewModelStore>()
    val mpesaFlowViewModelStoreOwner: ViewModelStoreOwner = object : ViewModelStoreOwner {
        override val viewModelStore: ViewModelStore
            get() {
                val key = _mpesaPaymentsState.value.javaClass.canonicalName ?: run {
                    Logger.error("Unable to get name for ${_mpesaPaymentsState.value.javaClass}")
                    return ViewModelStore()
                }

                return mpesaFlowViewModelStores[key] ?: run {
                    val store = ViewModelStore()
                    mpesaFlowViewModelStores[key] = store
                    store
                }
            }
    }

    private val _numberFormUiState = MutableStateFlow(
        MpesaNumberFormState(
            providerKey = providerKey,
            providers = accessCodeData.channelOptions.mobileMoney.orEmpty(),
            amount = accessCodeData.amount,
            currency = accessCodeData.currency
        )
    )
    val numberFormUiState = _numberFormUiState.asStateFlow()

    val amount: String
        get() = CurrencyFormatter.format(accessCodeData.amount, accessCodeData.currency)

    private val paybillNumber = MutableStateFlow("")
    private val accountNumber = MutableStateFlow("")
    private val isOfflineState = MutableStateFlow(false)
    private val offlineRequeryCounter = MutableStateFlow(OFFLINE_REQUERY_INIT_COUNT)

    private val _showPaybillNumberCopyIcon = MutableStateFlow(true)
    val showPaybillNumberCopyIcon = _showPaybillNumberCopyIcon.asStateFlow()

    private val _showAccountNumberCopyIcon = MutableStateFlow(true)
    val showAccountNumberCopyIcon = _showAccountNumberCopyIcon.asStateFlow()

    fun onPhoneNumberChanged(phoneNumber: String) {
        _numberFormUiState.update { it.copy(phoneNumber = phoneNumber) }
    }

    fun validatePhoneNumber() {
        if (_numberFormUiState.value.isPhoneNumberValid) {
            val formattedPhoneNumber = formatPhoneNumber()
            if (!isFormattedPhoneNumberValid(formattedPhoneNumber)) {
                _numberFormUiState.update { it.copy(formattedPhoneNumber = null) }
            } else _numberFormUiState.update { it.copy(formattedPhoneNumber = formattedPhoneNumber) }
        } else _numberFormUiState.update { it.copy(formattedPhoneNumber = null) }
    }

    /**
     * Formats an inputted and validated KE phone number to start with dial code of "+254"
     * and followed by other 9-digits.
     */
    private fun formatPhoneNumber(): String {
        val prefixLen =
            _numberFormUiState.value.phoneNumber.length - KE_STANDARD_PHONE_NUMBER_SUFFIX_LENGTH
        val standardFormatPrefix = _numberFormUiState.value.phoneNumber.take(prefixLen)
        return if (standardFormatPrefix != KE_DIALING_CODE) {
            _numberFormUiState.value.phoneNumber
                .replaceRange(
                    range = IntRange(start = 0, endInclusive = standardFormatPrefix.length - 1),
                    replacement = KE_DIALING_CODE
                )
        } else _numberFormUiState.value.phoneNumber
    }

    /**
     * The standard KE phone number length with dial code is 13.
     */
    private fun isFormattedPhoneNumberValid(formattedPhoneNumber: String): Boolean =
        formattedPhoneNumber.length == KE_STANDARD_PHONE_NUMBER_FULL_LENGTH

    private fun resetNumberFormUiState() =
        _numberFormUiState.update { it.copy(phoneNumber = "", isProcessing = false) }

    private fun chargeMoney(phoneNumber: String?, providerKey: String) {
        viewModelScope.launch {
            val result = transactionRepository.chargeMobileMoney(
                transactionId = accessCodeData.id,
                provider = providerKey,
                phone = phoneNumber
            )
            processResult(result)
        }
    }

    fun payOnline() {
        _numberFormUiState.update { it.copy(isProcessing = true) }
        validatePhoneNumber()
        isOfflineState.value = false
        chargeMoney(_numberFormUiState.value.formattedPhoneNumber, providerKey)
    }

    fun payOffline() {
        _mpesaPaymentsState.value = MpesaPaymentState.Loading(
            stringProvider.getString(R.string.pstk_mpesa_offline_switch_text)
        )
        isOfflineState.value = true
        chargeMoney(null, KEY_MPESA_OFFLINE)
    }

    private fun processResult(result: Result<MobileMoneyCharge>) {
        result.onSuccess(this::handleSuccessResult)
            .onFailure(this::handleFailure)
    }

    private fun handleSuccessResult(chargeResult: MobileMoneyCharge) {
        val action = chargeResult.action ?: return
        when (action) {
            Action.EnterNumber -> _mpesaPaymentsState.value = MpesaPaymentState.EnterNumber
            is Action.ShowInstruction -> {
                val requeryDelayMs = action.requeryDelayMs
                _mpesaPaymentsState.value = MpesaPaymentState.InProgress(
                    instruction = action.instruction,
                    phoneNumber = chargeResult.phone.orEmpty(),
                    timeLeftMs = requeryDelayMs,
                    requeryDelayMs = requeryDelayMs
                )

                val requeryTimerJob = startRequeryDelayTimer(requeryDelayMs)
                observeTransactionEvent(chargeResult.channelName) { transactionEvent ->
                    if (transactionEvent.status == TransactionStatus.Success) {
                        requeryTimerJob.cancel()
                        checkPaymentStatus()
                    } else {
                        requeryTimerJob.cancel()
                        handleTransactionEventFailure(transactionEvent.message)
                    }
                }
            }

            is Action.ShowOfflineDetails -> {
                paybillNumber.value = action.paybillNumber.orEmpty()
                accountNumber.value = action.accountNumber.orEmpty()

                _mpesaPaymentsState.value = MpesaPaymentState.ShowOfflineDetails(
                    amount = amount,
                    paybillNumber = paybillNumber.value,
                    accountNumber = accountNumber.value
                )

                observeTransactionEvent(chargeResult.channelName) { transactionEvent ->
                    if (transactionEvent.status == TransactionStatus.Success) checkPaymentStatus()
                    else handleTransactionEventFailure(transactionEvent.message)
                }
            }
        }
    }

    private fun handleFailure(error: Throwable) {
        Logger.error(error, error.message.orEmpty())

        if (error is PaystackError && !error.isFatal) {
            _mpesaPaymentsState.update {
                if (isOfflineState.value) MpesaPaymentState.OfflineError(message = error.message)
                else MpesaPaymentState.Error(message = error.message)
            }
            return
        }

        val message = stringProvider.getString(R.string.pstk_generic_error_msg)
        _mpesaPaymentsState.update {
            if (isOfflineState.value) MpesaPaymentState.OfflineError(message)
            else MpesaPaymentState.Error(message)
        }
        onError(error)
    }

    private fun startRequeryDelayTimer(delayTimeMs: Int): Job {
        return viewModelScope.launch(countdownDispatcher) {
            CountdownTimer().start(delayTimeMs).collectLatest { remainingTimeMs ->
                val state = _mpesaPaymentsState.value
                if (state is MpesaPaymentState.InProgress) {
                    _mpesaPaymentsState.value = state.copy(timeLeftMs = remainingTimeMs)

                    if (remainingTimeMs == 0) {
                        checkPaymentStatus()
                    }
                }
            }
        }
    }

    private fun checkPaymentStatus() {
        viewModelScope.launch {
            _mpesaPaymentsState.value = MpesaPaymentState.VerifyingPayment
            transactionRepository.checkPendingCharge(accessCodeData.accessCode)
                .onFailure(this@MpesaViewModel::handleFailure)
                .onSuccess { charge ->
                    when (charge.status) {
                        TransactionStatus.Success -> onPaymentComplete(charge)
                        TransactionStatus.Requery -> startOfflineRequeryDelay()
                        TransactionStatus.Failed -> {
                            val errorMessage = charge.message
                                ?: stringProvider.getString(R.string.pstk_mpesa_payment_generic_error_message)
                            _mpesaPaymentsState.update {
                                if (isOfflineState.value) MpesaPaymentState.OfflineError(
                                    errorMessage
                                )
                                else MpesaPaymentState.Error(errorMessage)
                            }
                        }

                        else -> {
                            val errorMessage =
                                stringProvider.getString(R.string.pstk_mpesa_payment_generic_error_message)
                            _mpesaPaymentsState.update {
                                if (isOfflineState.value) MpesaPaymentState.OfflineError(
                                    errorMessage
                                )
                                else MpesaPaymentState.Error(errorMessage)
                            }
                        }
                    }
                }
        }
    }

    private fun observeTransactionEvent(channelName: String, onEvent: (TransactionEvent) -> Unit) {
        viewModelScope.launch {
            val result = transactionRepository.awaitTransactionEvent(channelName)
            result.onFailure(this@MpesaViewModel::handleFailure)
                .onSuccess(onEvent)
        }
    }

    private fun handleTransactionEventFailure(message: String?) {
        val errorMessage =
            message ?: stringProvider.getString(R.string.pstk_mpesa_payment_generic_error_message)
        if (isOfflineState.value) _mpesaPaymentsState.value =
            MpesaPaymentState.OfflineError(errorMessage)
        else _mpesaPaymentsState.value = MpesaPaymentState.Error(errorMessage)
    }

    private fun startOfflineRequeryDelay() {
        viewModelScope.launch {
            delay(OFFLINE_REQUERY_DELAY_MS)
            val state = _mpesaPaymentsState.value
            if (state is MpesaPaymentState.VerifyingPayment) {
                if (offlineRequeryCounter.value == OFFLINE_REQUERY_MAX_COUNT) {
                    _mpesaPaymentsState.value = MpesaPaymentState.OfflineError(
                        message = stringProvider.getString(R.string.pstk_mpesa_offline_requery_error_message),
                        isStatusRequery = true
                    )

                    // Reset the requery counter value
                    offlineRequeryCounter.value = OFFLINE_REQUERY_INIT_COUNT
                } else {
                    offlineRequeryCounter.value += OFFLINE_REQUERY_COUNT_INCREMENTER
                    checkPaymentStatus()
                }
            }
        }
    }

    fun showPayWithLipaSteps() {
        _mpesaPaymentsState.value = MpesaPaymentState.ShowPayWithLipaSteps
    }

    fun showOfflineDetails() {
        _mpesaPaymentsState.value = MpesaPaymentState.ShowOfflineDetails(
            amount = amount,
            paybillNumber = paybillNumber.value,
            accountNumber = accountNumber.value
        )
    }

    fun checkOfflinePaymentStatus() {
        checkPaymentStatus()
    }

    fun retrySameNumber() {
        _mpesaPaymentsState.value = MpesaPaymentState.Loading()
        isOfflineState.value = false
        chargeMoney(_numberFormUiState.value.formattedPhoneNumber, providerKey)
    }

    fun switchToOnline() {
        resetNumberFormUiState()
        isOfflineState.value = false
        _mpesaPaymentsState.value = MpesaPaymentState.EnterNumber
    }

    fun startCopyPaybillNumberCountdownTimer() {
        _showPaybillNumberCopyIcon.value = false
        viewModelScope.launch(countdownDispatcher) {
            CountdownTimer().start(TEXT_COPY_DURATION).collectLatest { remainingTimeMs ->
                val state = _mpesaPaymentsState.value
                if (state is MpesaPaymentState.ShowOfflineDetails && remainingTimeMs == 0) {
                    _showPaybillNumberCopyIcon.value = true
                }
            }
        }
    }

    fun startCopyAccountNumberCountdownTimer() {
        _showAccountNumberCopyIcon.value = false
        viewModelScope.launch(countdownDispatcher) {
            CountdownTimer().start(TEXT_COPY_DURATION).collectLatest { remainingTimeMs ->
                val state = _mpesaPaymentsState.value
                if (state is MpesaPaymentState.ShowOfflineDetails && remainingTimeMs == 0) {
                    _showAccountNumberCopyIcon.value = true
                }
            }
        }
    }

    fun retryPayOffline() {
        _mpesaPaymentsState.value = MpesaPaymentState.Loading(
            stringProvider.getString(R.string.pstk_mpesa_offline_retry_payment_text)
        )
        isOfflineState.value = true
        chargeMoney(null, KEY_MPESA_OFFLINE)
    }

    companion object {
        internal const val KEY_MPESA_OFFLINE = "MPESA_OFF"
        private const val KE_DIALING_CODE = "+254"
        private const val KE_STANDARD_PHONE_NUMBER_SUFFIX_LENGTH = 9
        private const val KE_STANDARD_PHONE_NUMBER_FULL_LENGTH = 13
        private const val TEXT_COPY_DURATION = 5000
        private const val OFFLINE_REQUERY_DELAY_MS = 5000L
        private const val OFFLINE_REQUERY_MAX_COUNT = 12
        private const val OFFLINE_REQUERY_INIT_COUNT = 0
        private const val OFFLINE_REQUERY_COUNT_INCREMENTER = 1
    }
}
