package com.paystack.android.ui.components.views.inputs.validators

import androidx.compose.ui.text.AnnotatedString
import androidx.compose.ui.text.input.OffsetMapping
import androidx.compose.ui.text.input.TransformedText

/**
 * Singleton class that formats the card expiry in the MM/YY format for the Compose UI
 */
internal object CardExpirationDateFormatter {

    /**
     * The exposed function that takes in a regular string, possibly given from a StateFlow
     * and emits the transformed date in the MM/YY format.
     *
     * This can be used for non compose validators or other non compose sources.
     *
     * @param dateText eg. 1226
     * @return formatted date text such as 12/26
     */
    fun formatEnteredCardDate(dateText: String): String {
        return dateText.chunked(2).joinToString("/")
    }

    /**
     * The exposed function that takes in a Compose annotated string emitted by the Visual
     * transformer and emits the transformed date
     *
     * @param dateText the entered date text, usually the first two characters before the
     * formatter kicks in to apply the separator "/"
     *
     * @return the formatted month date "MM" with the date separator "MM/YY"
     */
    fun formatEnteredCardDate(dateText: AnnotatedString): TransformedText {
        return makeCardExpiryTextFilter(dateText = dateText)
    }

    private fun makeCardExpiryTextFilter(dateText: AnnotatedString): TransformedText {
        val trimmed = if (dateText.text.length > 4) {
            dateText.text.substring(0..3)
        } else {
            dateText.text
        }

        var output = ""

        trimmed.indices.forEach { index ->
            output += dateText[index]
            if (index == 1) output += "/"
        }

        val dateTextOffsetMapping = object : OffsetMapping {

            override fun originalToTransformed(offset: Int): Int {
                if (offset <= 1) return offset
                if (offset <= 4) return offset + 1
                return 5
            }

            override fun transformedToOriginal(offset: Int): Int {
                if (offset <= 1) return offset
                if (offset <= 4) return offset - 1
                return 4
            }
        }

        return TransformedText(AnnotatedString(output), dateTextOffsetMapping)
    }
}
