package com.paystack.android.ui.components.views.inputs

import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.ArrowDropDown
import androidx.compose.material3.Divider
import androidx.compose.material3.Icon
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.draw.rotate
import androidx.compose.ui.window.Dialog
import com.paystack.android.ui.components.views.inputs.textfield.StackTextField
import com.paystack.android.ui.theme.shapes
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
fun PrimaryDropDownMenu(
    menuItems: List<String>,
    onItemSelected: (index: Int) -> Unit,
    modifier: Modifier = Modifier,
    selectedItemIndex: Int?,
    label: String? = null
) {
    var expanded by remember { mutableStateOf(false) }
    val iconDirection by animateFloatAsState(if (expanded) 180f else 0f)

    Box {
        StackTextField(
            modifier = modifier,
            value = selectedItemIndex?.let { menuItems.getOrNull(it) }.orEmpty(),
            label = label?.let { { Text(text = it) } },
            readOnly = true,
            trailingIcon = {
                Icon(
                    imageVector = Icons.Filled.ArrowDropDown,
                    contentDescription = null,
                    modifier = Modifier.rotate(iconDirection)
                )
            },
            onValueChange = {},
        )

        // Click event receiver for the text field
        Box(
            modifier = Modifier
                .matchParentSize()
                .clip(PaystackTheme.shapes.textField)
                .clickable { expanded = true },
        )
    }

    if (expanded) {
        ItemListDialog(
            selectedItemIndex = selectedItemIndex,
            label = label,
            menuItems = menuItems,
            onItemSelected = onItemSelected,
            onDismissDialog = { expanded = false }
        )
    }
}

@Composable
private fun ItemListDialog(
    selectedItemIndex: Int?,
    label: String?,
    menuItems: List<String>,
    onItemSelected: (index: Int) -> Unit,
    onDismissDialog: () -> Unit
) {
    Dialog(onDismissRequest = onDismissDialog) {
        PaystackTheme {
            Surface(shape = PaystackTheme.shapes.modal) {
                val listState = rememberLazyListState()

                if (selectedItemIndex != null && selectedItemIndex > 1) {
                    LaunchedEffect("ScrollState") {
                        listState.scrollToItem(index = selectedItemIndex)
                    }
                }

                LazyColumn(modifier = Modifier.fillMaxWidth(), state = listState) {
                    if (label != null) {
                        item {
                            DropDownMenuItem(
                                text = label,
                                onClick = {}
                            )
                        }
                    }
                    itemsIndexed(menuItems) { index, item ->
                        DropDownMenuItem(
                            text = item,
                            onClick = {
                                onItemSelected(index)
                                onDismissDialog()
                            }
                        )

                        if (index < menuItems.lastIndex) {
                            Divider(
                                modifier =
                                Modifier.padding(
                                    horizontal = PaystackTheme.dimensions.spacing.double
                                )
                            )
                        }
                    }
                }
            }
        }
    }
}
