package com.paystack.android.ui

import android.os.Bundle
import androidx.lifecycle.AbstractSavedStateViewModelFactory
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import androidx.savedstate.SavedStateRegistryOwner
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.logging.Logger
import com.paystack.android.ui.data.transaction.TransactionRepository
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.paymentsheet.PaymentSheetResultContract
import com.paystack.android.ui.utilities.StringProvider
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class PaymentContainerViewModel(
    private val transactionRepository: TransactionRepository,
    private val stringProvider: StringProvider,
    savedStateHandle: SavedStateHandle,
) : ViewModel() {

    private val _paymentContainerState =
        MutableStateFlow<PaymentContainerState>(PaymentContainerState.Initialization)
    val paymentContainerState: StateFlow<PaymentContainerState>
        get() = _paymentContainerState

    private val _transactionAccessData = MutableStateFlow<AccessCodeData?>(null)
    val transactionAccessData: StateFlow<AccessCodeData?>
        get() = _transactionAccessData

    init {
        val accessCode: String? = savedStateHandle[PaymentSheetResultContract.EXTRA_ACCESS_CODE]
        if (accessCode == null) {
            _paymentContainerState.update {
                val error = IllegalStateException("No transaction access code provided")
                PaymentContainerState.Error(
                    stringProvider.getString(R.string.pstk_could_not_start_transaction),
                    error
                )
            }
        } else {
            loadTransaction(accessCode)
        }
    }

    private fun loadTransaction(accessCode: String) {
        viewModelScope.launch {
            transactionRepository.getByAccessCode(accessCode)
                .onSuccess {
                    _transactionAccessData.value = it
                    _paymentContainerState.value = PaymentContainerState.PaymentProcess(it)
                }
                .onFailure { error ->
                    Logger.error(error, error.message.orEmpty())
                    _paymentContainerState.value = PaymentContainerState.Error(error)
                }
        }
    }

    fun onTransactionComplete(charge: Charge) {
        _paymentContainerState.value = PaymentContainerState.Success(
            charge = charge,
            merchantName = transactionAccessData.value?.merchantName.orEmpty()
        )
    }

    fun onError(error: Throwable) {
        _paymentContainerState.value = PaymentContainerState.Error(error)
    }

    class Factory(
        private val transactionRepository: TransactionRepository,
        private val stringProvider: StringProvider,
        savedStateRegistryOwner: SavedStateRegistryOwner,
        defaultArgs: Bundle? = null
    ) : AbstractSavedStateViewModelFactory(savedStateRegistryOwner, defaultArgs) {
        @Suppress("UNCHECKED_CAST")
        override fun <T : ViewModel> create(
            key: String,
            modelClass: Class<T>,
            handle: SavedStateHandle
        ): T {
            if (modelClass.isAssignableFrom(PaymentContainerViewModel::class.java)) {
                return PaymentContainerViewModel(
                    transactionRepository,
                    stringProvider,
                    handle
                ) as T
            }
            throw IllegalArgumentException("Unknown ViewModel class")
        }
    }
}
