package com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.inprogress

import androidx.compose.animation.core.LinearEasing
import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.animation.core.tween
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import com.paystack.android.ui.R
import com.paystack.android.ui.components.views.text.CountdownText
import com.paystack.android.ui.theme.countdownProgressTrackColor
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun MpesaProgressUi(
    phoneNumber: String,
    instruction: String,
    timeLeftMs: Int,
    timeAllowedMs: Int,
    modifier: Modifier = Modifier
) {
    Column(
        modifier = modifier.padding(PaystackTheme.dimensions.spacing.double),
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Image(painterResource(R.drawable.pstk_ic_otp), contentDescription = null)
        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))
        Text(
            text = phoneNumber,
            style = PaystackTheme.typography.body.body16M,
            textAlign = TextAlign.Center,
            modifier = Modifier.fillMaxWidth()
        )

        Spacer(Modifier.height(PaystackTheme.dimensions.spacing.single))
        Text(
            text = instruction,
            style = PaystackTheme.typography.body.body14R,
            textAlign = TextAlign.Center,
            modifier = Modifier
                .fillMaxWidth()
                .padding(horizontal = PaystackTheme.dimensions.spacing.triple)
        )
        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.quad))

        OtpCountdownTimer(timeAllowedMs, timeLeftMs)
    }
}

@Composable
private fun OtpCountdownTimer(timeAllowedMs: Int, timeLeftMs: Int) {
    val progress by animateFloatAsState(
        targetValue = timeLeftMs.toFloat() / timeAllowedMs.toFloat(),
        animationSpec = tween(
            durationMillis = CountdownProgressAnimDurationMs,
            easing = LinearEasing
        ),
        label = "CountdownProgressAnim"
    )

    Box(contentAlignment = Alignment.Center) {
        CircularProgressIndicator(
            trackColor = PaystackTheme.colors.countdownProgressTrackColor,
            color = PaystackTheme.colors.secondaryColors.stackGreen,
            progress = progress,
            modifier = Modifier.matchParentSize(),
        )

        Icon(
            modifier = Modifier.wrapContentSize(),
            painter = painterResource(R.drawable.pstk_countdown_otp_bubble_34dp),
            contentDescription = null,
            tint = Color.Unspecified
        )
    }

    Spacer(Modifier.height(PaystackTheme.dimensions.spacing.double))
    CompositionLocalProvider(LocalTextStyle provides PaystackTheme.typography.body.body14R) {
        CountdownText(timeLeftMs.toLong(), stringResource(R.string.pstk_mpesa_countdown_prefix))
    }
}

@Preview
@Composable
private fun MpesaProgressUiPreview() {
    PaystackTheme {
        Surface(color = Color.White) {
            Column(Modifier.fillMaxSize()) {
                MpesaProgressUi(
                    phoneNumber = "0710000000",
                    instruction = "Please enter your pin on your phone to complete this payment",
                    timeLeftMs = 35000,
                    timeAllowedMs = 60000,
                )
            }
        }
    }
}

private const val CountdownProgressAnimDurationMs = 1100
