package com.paystack.android.ui.paymentsheet

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.compose.foundation.layout.WindowInsets
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.navigationBars
import androidx.compose.foundation.layout.windowInsetsPadding
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.core.view.WindowCompat
import com.paystack.android.ui.PaymentContainerUi
import com.paystack.android.ui.PaymentContainerViewModel
import com.paystack.android.ui.R
import com.paystack.android.ui.components.views.dialog.PaymentDialog
import com.paystack.android.ui.data.dataComponent
import com.paystack.android.ui.models.PaymentCompletionDetails
import com.paystack.android.ui.none
import com.paystack.android.ui.theme.PaymentSheetShape
import com.paystack.android.ui.theme.PaystackTheme
import com.paystack.android.ui.utilities.utilitiesComponent

internal class PaymentSheetActivity : ComponentActivity() {

    private val paymentContainerViewModel: PaymentContainerViewModel by viewModels {
        PaymentContainerViewModel.Factory(
            transactionRepository = dataComponent().transactionRepository,
            stringProvider = utilitiesComponent(this@PaymentSheetActivity).stringProvider,
            savedStateRegistryOwner = this,
            defaultArgs = intent.extras
        )
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        overridePendingTransition(android.R.anim.fade_in, android.R.anim.fade_out)

        WindowCompat.setDecorFitsSystemWindows(window, false)

        setContent {
            PaystackTheme(false) {
                PaymentSheetUI { result ->
                    setResult(
                        Activity.RESULT_OK,
                        Intent().putExtras(PaymentSheetResultContract.Result(result).toBundle())
                    )
                    finish()
                }
            }
        }
    }

    @Composable
    @OptIn(ExperimentalMaterial3Api::class)
    private fun PaymentSheetUI(onPaymentComplete: (PaymentSheetResult) -> Unit) {
        // ModalBottomSheet does not pass window insets down to its children.
        // This captures the required insets to be used in PaymentsUi composable.
        val navigationBarInsets = WindowInsets.navigationBars

        val modalBottomSheetState =
            rememberModalBottomSheetState(skipPartiallyExpanded = true) { false }
        var openDialog by remember { mutableStateOf(false) }

        ModalBottomSheet(
            modifier = Modifier.fillMaxHeight(0.85f),
            onDismissRequest = { openDialog = true },
            dragHandle = null,
            shape = PaymentSheetShape,
            windowInsets = WindowInsets.none,
            sheetState = modalBottomSheetState
        ) {
            PaymentContainerUi(
                viewModel = paymentContainerViewModel,
                modifier = Modifier.windowInsetsPadding(navigationBarInsets),
                onPaymentComplete = { charge ->
                    val paymentInfo = PaymentCompletionDetails(charge.reference)
                    onPaymentComplete(PaymentSheetResult.Completed(paymentInfo))
                },
                onPaymentCancelled = { openDialog = true },
                onError = { error, reference ->
                    onPaymentComplete(PaymentSheetResult.Failed(error, reference))
                }
            )
        }

        if (openDialog) {
            PaymentDialog(
                dialogText = stringResource(id = R.string.pstk_payment_cancel_title),
                dismissText = stringResource(id = R.string.pstk_payment_cancel_no),
                confirmText = stringResource(id = R.string.pstk_payment_cancel_abort),
                onDismissOptionSelected = { openDialog = false },
                onPositiveOptionSelected = {
                    onPaymentComplete(PaymentSheetResult.Cancelled)
                    openDialog = false
                })
        }
    }

    override fun finish() {
        super.finish()
        overridePendingTransition(android.R.anim.fade_in, android.R.anim.fade_out)
    }
}
