package com.paystack.android.ui.paymentchannels.mobilemoney.mpesa.numberform

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android.core.api.models.MobileMoneyChannelOption
import com.paystack.android.ui.R
import com.paystack.android.ui.components.buttons.PrimaryButton
import com.paystack.android.ui.components.views.inputs.textfield.StackTextField
import com.paystack.android.ui.models.MobileMoneyCharge
import com.paystack.android.ui.paymentchannels.mobilemoney.MobileMoneyViewModelProviderFactory
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun MpesaNumberFormUi(
    viewModelsProviderFactory: MobileMoneyViewModelProviderFactory,
    onResult: (Result<MobileMoneyCharge>) -> Unit,
    modifier: Modifier = Modifier,
    viewModel: MpesaNumberFormViewModel = viewModel(factory = viewModelsProviderFactory)
) {
    val state by viewModel.uiState.collectAsState()
    val result by viewModel.result.collectAsState()

    LaunchedEffect(result) {
        result?.let { onResult(it) }
    }

    MpesaNumberForm(
        state = state,
        onPhoneNumberChanged = viewModel::onPhoneNumberChanged,
        onSubmitButtonClick = viewModel::submitForm,
        modifier = modifier
    )
}

@OptIn(ExperimentalComposeUiApi::class)
@Composable
internal fun MpesaNumberForm(
    state: MpesaNumberFormState,
    onPhoneNumberChanged: (String) -> Unit,
    onSubmitButtonClick: () -> Unit,
    modifier: Modifier = Modifier,
) {
    val keyboardController = LocalSoftwareKeyboardController.current
    val focusRequester = remember { FocusRequester() }

    LaunchedEffect(Unit) {
        focusRequester.requestFocus()
    }

    Column(
        modifier = modifier,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        Text(
            text = stringResource(R.string.pstk_mpesa_phone_number_instruction),
            style = PaystackTheme.typography.body.body16M,
            textAlign = TextAlign.Center,
            modifier = Modifier
                .fillMaxWidth()
                .padding(horizontal = PaystackTheme.dimensions.spacing.triple)
        )

        Spacer(Modifier.height(PaystackTheme.dimensions.spacing.triple))

        StackTextField(
            value = state.phoneNumber,
            onValueChange = onPhoneNumberChanged,
            keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Phone),
            placeholder = { Text(stringResource(R.string.pstk_phone_number_placeholder_ke)) },
            trailingIcon = {
                Box(
                    modifier = Modifier.padding(end = PaystackTheme.dimensions.spacing.single),
                    contentAlignment = Alignment.Center
                ) {
                    Icon(
                        painter = painterResource(R.drawable.pstk_ic_flag_ke_24dp),
                        contentDescription = stringResource(R.string.pstk_kenyan_flag_description),
                        tint = Color.Unspecified
                    )
                }
            },
            modifier = Modifier
                .fillMaxWidth()
                .focusRequester(focusRequester)
        )

        Spacer(Modifier.height(PaystackTheme.dimensions.spacing.double))

        PrimaryButton(
            buttonText = stringResource(
                R.string.pstk_component_pay_button_pay_amount,
                state.amountText
            ),
            onClick = {
                keyboardController?.hide()
                onSubmitButtonClick()
            },
            isEnabled = state.isPhoneNumberValid,
            isLoading = state.isProcessing
        )
    }
}

@Preview
@Composable
private fun MpesaNumberFormPreview() {
    val mobileMoneyProviders = listOf(
        MobileMoneyChannelOption(
            key = "mpesa",
            value = "M-Pesa",
            isNew = false,
            phoneNumberRegex = ""
        )
    )
    PaystackTheme {
        Column(modifier = Modifier.padding(16.dp)) {
            MpesaNumberForm(
                state = MpesaNumberFormState(
                    phoneNumber = "0712345678",
                    amount = 90000,
                    currency = "KES",
                    providers = mobileMoneyProviders,
                    providerKey = "mpesa"
                ),
                onPhoneNumberChanged = {},
                onSubmitButtonClick = {}
            )
        }
    }
}
