package com.paystack.android_sdk.ui.paymentchannels.card.cardform.views

import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.ExperimentalAnimationApi
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.res.stringResource
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android_sdk.ui.R
import com.paystack.android_sdk.ui.components.buttons.PrimaryButton
import com.paystack.android_sdk.ui.components.buttons.StackTextButton
import com.paystack.android_sdk.ui.components.views.inputs.CVVInputField
import com.paystack.android_sdk.ui.components.views.inputs.CardExpiryInputField
import com.paystack.android_sdk.ui.components.views.inputs.CardNumberInputField
import com.paystack.android_sdk.ui.components.views.text.PaymentTitleText
import com.paystack.android_sdk.ui.models.Charge
import com.paystack.android_sdk.ui.paymentchannels.card.CardViewModelsProviderFactory
import com.paystack.android_sdk.ui.paymentchannels.card.cardform.TestCardList
import com.paystack.android_sdk.ui.paymentchannels.card.cardform.models.CardFormData
import com.paystack.android_sdk.ui.utilities.CurrencyFormatter
import com.paystack.androiddesignsystem.PaystackTheme

@OptIn(ExperimentalAnimationApi::class)
@Composable
internal fun CardDetails(
    viewModelsProviderFactory: CardViewModelsProviderFactory,
    modifier: Modifier = Modifier,
    viewModel: CardDetailsViewModel = viewModel(factory = viewModelsProviderFactory),
    onResult: (Result<Charge>) -> Unit,
) {
    val state by viewModel.state.collectAsState()
    val result by viewModel.result.collectAsState()

    LaunchedEffect(result) {
        result?.let { onResult(it) }
    }

    Column(
        modifier = modifier,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        CardDetailInstruction(state.showTestCards)

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.triple))

        AnimatedContent(state.showTestCards, label = ContentAnimLabel) { showTestCards ->
            if (showTestCards) {
                TestCardList(
                    testCards = state.testCards,
                    selectedIndex = state.selectedTestCardIndex,
                    onCardSelected = viewModel::setSelectedTestCard
                )
            } else {
                CardForm(
                    cardFormData = state.cardData,
                    onCardNumberChanged = viewModel::setCardNumber,
                    onCardExpiryChanged = viewModel::setCardExpiry,
                    onCvvChanged = viewModel::setCardCVV,
                )
            }
        }

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        PrimaryButton(
            buttonText = stringResource(
                id = R.string.pstk_component_pay_button_pay_amount,
                CurrencyFormatter.format(state.amount, state.currency)
            ),
            onClick = viewModel::submitCardDetails,
            modifier = Modifier,
            isEnabled = state.payButtonEnabled,
            isLoading = state.isSubmitting
        )

        if (state.isTestTransaction) {
            Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))
            TestCardsToggle(state.showTestCards, viewModel::toggleShowTestCards)
        }
    }
}

@Composable
@OptIn(ExperimentalAnimationApi::class)
private fun TestCardsToggle(
    showTestCards: Boolean,
    onClick: () -> Unit,
    modifier: Modifier = Modifier
) {
    AnimatedContent(targetState = showTestCards, label = TestCardsToggleAnimLabel) {
        StackTextButton(
            text = stringResource(
                if (it) {
                    R.string.pstk_use_another_card
                } else {
                    R.string.pstk_use_a_test_card
                }
            ),
            onClick = onClick,
            modifier = modifier
        )
    }
}

@Composable
@OptIn(ExperimentalAnimationApi::class)
private fun CardDetailInstruction(showTestCards: Boolean) {
    AnimatedContent(
        targetState = showTestCards,
        label = TitleAnimLabel
    ) {
        PaymentTitleText(
            titleText = stringResource(
                id = if (it) {
                    R.string.pstk_card_payment_test_card_instruction
                } else {
                    R.string.pstk_card_payment_enter_details_title
                }
            ),
            modifier = Modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double)
        )
    }
}

@Composable
private fun CardForm(
    cardFormData: CardFormData,
    onCardNumberChanged: (String) -> Unit,
    onCardExpiryChanged: (String) -> Unit,
    onCvvChanged: (String) -> Unit,
    modifier: Modifier = Modifier,
) {
    val focusRequester = remember { FocusRequester() }

    LaunchedEffect(Unit) {
        focusRequester.requestFocus()
    }

    Column(
        modifier = modifier,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        CardNumberInputField(
            cardText = cardFormData.cardNumber,
            onCardTextChanged = onCardNumberChanged,
            maxInputLength = cardFormData.cardType.maxPanLength,
            isValidCard = cardFormData.isCardNumberValid,
            modifier = Modifier.focusRequester(focusRequester)
        )

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

        Row(
            modifier = Modifier.fillMaxWidth(),
            horizontalArrangement = Arrangement.spacedBy(PaystackTheme.dimensions.spacing.double)
        ) {
            CardExpiryInputField(
                modifier = Modifier.weight(1f),
                expiryInputValue = cardFormData.cardExpiry,
                isCardExpiryValid = cardFormData.isExpiryDateValid,
                onDateTextChange = onCardExpiryChanged
            )

            CVVInputField(
                modifier = Modifier.weight(1f),
                inputCVVText = cardFormData.cardCVV,
                maxInputLength = cardFormData.cardType.maxCvvLength,
                onTextChange = onCvvChanged
            )
        }
    }
}

private const val TestCardsToggleAnimLabel = "TestCardsToggleAnim"
private const val TitleAnimLabel = "CardScreenTitleAnim"
private const val ContentAnimLabel = "CardScreenContentAnim"
