package com.paystack.android_sdk.ui.paymentchannels.card.auth.redirect

import android.annotation.SuppressLint
import android.graphics.Bitmap
import android.webkit.WebChromeClient
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.Crossfade
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.viewinterop.AndroidView
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android_sdk.core.api.models.TransactionStatus
import com.paystack.android_sdk.core.logging.Logger
import com.paystack.android_sdk.ui.R
import com.paystack.android_sdk.ui.components.buttons.PrimaryButton
import com.paystack.android_sdk.ui.components.buttons.StackTextButton
import com.paystack.android_sdk.ui.components.progressbars.LoadingIndicator
import com.paystack.android_sdk.ui.components.views.text.PaymentTitleText
import com.paystack.android_sdk.ui.paymentchannels.card.CardViewModelsProviderFactory
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun RedirectAuth(
    authUrl: String,
    viewModelsProviderFactory: CardViewModelsProviderFactory,
    onResult: (Result<TransactionStatus>) -> Unit,
    onCancelTransaction: () -> Unit,
    viewModel: RedirectAuthViewModel = viewModel(factory = viewModelsProviderFactory),
    modifier: Modifier = Modifier,
) {
    val state by viewModel.state.collectAsState()
    LaunchedEffect(state.result) {
        state.result?.let {
            onResult(it)
        }
    }

    Crossfade(targetState = state.launchWebUrl, label = "WebUrlAuthCrossfade") { showWebView ->
        if (showWebView) {
            WebAuth(
                url = authUrl,
                onFinish = { /* On finish */ },
                modifier = modifier.fillMaxSize()
            )
        } else {
            Column(
                modifier = modifier,
                horizontalAlignment = Alignment.CenterHorizontally
            ) {
                Image(
                    painter = painterResource(id = R.drawable.pstk_ic_payment_3ds),
                    contentDescription = stringResource(
                        id = R.string.pstk_card_payment_3ds_content_description
                    )
                )

                PaymentTitleText(
                    titleText = stringResource(id = R.string.pstk_card_payment_3ds_auth_start_title),
                    modifier = Modifier.padding(top = PaystackTheme.dimensions.spacing.triple)
                )

                Spacer(modifier = Modifier.padding(PaystackTheme.dimensions.spacing.double))

                PrimaryButton(
                    buttonText = stringResource(id = R.string.pstk_card_payment_3ds_btn_authenticate),
                    onClick = viewModel::startWebAuth,
                    isEnabled = true,
                )

                StackTextButton(
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(PaystackTheme.dimensions.spacing.double),
                    text = stringResource(id = R.string.pstk_card_payment_cancel_transaction),
                    onClick = onCancelTransaction
                )
            }
        }
    }
}

@Composable
fun WebAuth(
    url: String,
    onFinish: () -> Unit,
    modifier: Modifier = Modifier
) {
    val (isLoading, setIsLoading) = remember { mutableStateOf(true) }
    val context = LocalContext.current

    Box(contentAlignment = Alignment.TopCenter) {
        AndroidView(
            modifier = modifier,
            factory = {

                WebView(context).apply {
                    @SuppressLint("SetJavaScriptEnabled")
                    settings.javaScriptEnabled = true
                    settings.javaScriptCanOpenWindowsAutomatically = true

                    webChromeClient = object : WebChromeClient() {
                        override fun onCloseWindow(window: WebView?) {
                            Logger.debug("Web Client Window closed")
                            onFinish()
                        }
                    }

                    webViewClient = object : WebViewClient() {
                        override fun onPageStarted(view: WebView?, url: String?, favicon: Bitmap?) {
                            super.onPageStarted(view, url, favicon)
                            Logger.debug("onPageStarted: $url")
                            setIsLoading(true)
                        }

                        override fun onPageFinished(view: WebView?, url: String?) {
                            super.onPageFinished(view, url)
                            Logger.debug("onPageFinished: $url")
                            setIsLoading(false)
                        }
                    }

                    loadUrl(url)
                }
            }
        )

        AnimatedVisibility(isLoading) {
            Box(
                contentAlignment = Alignment.Center,
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(vertical = PaystackTheme.dimensions.spacing.quad)
            ) {
                LoadingIndicator(
                    color = PaystackTheme.colors.neutralColors.gray01,
                    modifier = Modifier.size(PaystackTheme.dimensions.spacing.triple)
                )
            }
        }
    }
}
