package com.paystack.android_sdk.ui.paymentchannels.card

import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelStore
import androidx.lifecycle.ViewModelStoreOwner
import androidx.lifecycle.viewModelScope
import com.paystack.android_sdk.core.api.models.AccessCodeData
import com.paystack.android_sdk.core.api.models.PaystackError
import com.paystack.android_sdk.core.api.models.TransactionStatus
import com.paystack.android_sdk.core.api.models.TransactionStatus.Failed
import com.paystack.android_sdk.core.api.models.TransactionStatus.OpenUrl
import com.paystack.android_sdk.core.api.models.TransactionStatus.Pending
import com.paystack.android_sdk.core.api.models.TransactionStatus.SendAddress
import com.paystack.android_sdk.core.api.models.TransactionStatus.SendBirthday
import com.paystack.android_sdk.core.api.models.TransactionStatus.SendPhone
import com.paystack.android_sdk.core.api.models.TransactionStatus.SendPin
import com.paystack.android_sdk.core.api.models.TransactionStatus.SentOtp
import com.paystack.android_sdk.core.api.models.TransactionStatus.Success
import com.paystack.android_sdk.core.api.models.TransactionStatus.Timeout
import com.paystack.android_sdk.core.logging.Logger
import com.paystack.android_sdk.ui.R
import com.paystack.android_sdk.ui.data.transaction.TransactionRepository
import com.paystack.android_sdk.ui.models.Charge
import com.paystack.android_sdk.ui.utilities.StringProvider
import com.paystack.android_sdk.ui.utilities.isFatal
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class CardPaymentViewModel(
    private val transactionRepository: TransactionRepository,
    private val transactionAccessData: AccessCodeData,
    private val stringProvider: StringProvider,
    private val onPaymentComplete: (Charge) -> Unit,
    private val onError: (Throwable) -> Unit
) : ViewModel() {
    private val _cardPaymentsState: MutableStateFlow<CardPaymentsState> =
        MutableStateFlow(CardPaymentsState.CardDetails)
    val cardPaymentsState: StateFlow<CardPaymentsState>
        get() = _cardPaymentsState

    private val paymentFlowVmStores = mutableMapOf<String, ViewModelStore>()
    val cardFlowViewModelStoreOwner: ViewModelStoreOwner = object : ViewModelStoreOwner {
        override val viewModelStore: ViewModelStore
            get() {
                val key = cardPaymentsState.value.javaClass.canonicalName ?: run {
                    Logger.error(
                        "Unable to get canonical name for ${cardPaymentsState.value.javaClass}",
                        null
                    )
                    return ViewModelStore()
                }

                return paymentFlowVmStores[key] ?: run {
                    val store = ViewModelStore()
                    paymentFlowVmStores[key] = store
                    store
                }
            }
    }

    init {
        // Clean up unused ViewModelStores
        cardPaymentsState.onEach {
            val currentViewModelStoreKey = it.javaClass.canonicalName ?: return@onEach
            val obsoleteKeys =
                paymentFlowVmStores.keys.filter { key -> key != currentViewModelStoreKey }
            paymentFlowVmStores.keys.removeAll(obsoleteKeys.toSet())
        }.launchIn(viewModelScope)
    }

    // LOGIC FOR MOVING BETWEEN CARD PAYMENTS STATES LIVES HERE

    fun onPinAuth() {
        _cardPaymentsState.update { CardPaymentsState.CardPin }
    }

    fun onDateOfBirthAuth() {
        _cardPaymentsState.update { CardPaymentsState.DateOfBirthAuthentication }
    }

    fun restartCardPayment() {
        _cardPaymentsState.update { CardPaymentsState.CardDetails }
    }

    fun processCardChargeResponse(result: Result<Charge>) {
        result.onSuccess { transaction -> parseTransaction(transaction) }
            .onFailure { handleFailure(it) }
    }

    /**
     * Temporarily used to  trigger check pending for 3DS transactions.
     * The UI module has to be refactored to use its own charge model before the full response can be used.
     */
    fun process3dsResponse(result: Result<TransactionStatus>) {
        result.onFailure {
            Logger.error(it.message.orEmpty(), it)
        }
        handlePending()
    }

    private fun parseTransaction(charge: Charge) {
        when (charge.status) {
            Success -> onPaymentComplete(charge)
            Failed -> {
                val message =
                    charge.message ?: stringProvider.getString(R.string.pstk_generic_error_msg)
                _cardPaymentsState.update {
                    CardPaymentsState.Error(message = message, isRecoverable = true)
                }
            }

            SentOtp -> _cardPaymentsState.update { CardPaymentsState.OtpAuth(charge.displayText) }
            SendBirthday -> _cardPaymentsState.update { CardPaymentsState.DateOfBirthAuthentication }
            SendPin -> _cardPaymentsState.update { CardPaymentsState.CardPin }
            SendAddress -> handleAddressAuth(charge)
            SendPhone -> _cardPaymentsState.update { CardPaymentsState.PhoneNumberAuth }
            OpenUrl -> handleRedirect(charge)
            Pending -> handlePending()
            Timeout -> handleTimeout(charge)
        }
    }

    private fun handleAddressAuth(charge: Charge) {
        _cardPaymentsState.update {
            CardPaymentsState.AddressAuthentication(charge.countryCode.orEmpty())
        }
    }

    private fun handleRedirect(charge: Charge) {
        val errorMessage = stringProvider.getString(R.string.pstk_generic_error_msg)
        val authUrl = charge.authUrl
        if (authUrl == null) {
            _cardPaymentsState.update { CardPaymentsState.Error(errorMessage) }
            onError(PaystackError(IllegalStateException("No auth url found")))
            return
        }

        _cardPaymentsState.update { CardPaymentsState.RedirectAuth(authUrl) }
    }

    private fun handleTimeout(charge: Charge) {
        val message =
            charge.displayText ?: stringProvider.getString(R.string.pstk_transaction_timed_out)
        _cardPaymentsState.update {
            CardPaymentsState.Error(message = message, isRecoverable = false)
        }
        onError(PaystackError(message))
    }

    private fun handleFailure(error: Throwable) {
        Logger.error(error.message.orEmpty(), error)

        if (error is PaystackError && !error.isFatal) {
            _cardPaymentsState.update {
                CardPaymentsState.Error(message = error.message, isRecoverable = true)
            }
            return
        }

        val message = stringProvider.getString(R.string.pstk_generic_error_msg)
        _cardPaymentsState.update { CardPaymentsState.Error(message) }
        onError(error)
    }

    private fun handlePending() {
        _cardPaymentsState.update { CardPaymentsState.Pending }
        checkPendingCharge()
    }

    private fun checkPendingCharge() {
        viewModelScope.launch {
            delay(CHECK_PENDING_DELAY_MILLIS)
            val result = transactionRepository.checkPendingCharge(transactionAccessData.accessCode)
            processCardChargeResponse(result)
        }
    }

    companion object {
        private const val CHECK_PENDING_DELAY_MILLIS = 5000L
    }
}
