package com.paystack.android_sdk.ui.data.transaction

import com.paystack.android_sdk.core.api.await
import com.paystack.android_sdk.core.api.models.AccessCodeData
import com.paystack.android_sdk.core.api.models.AddressState
import com.paystack.android_sdk.core.api.models.CardParams
import com.paystack.android_sdk.core.api.models.RedirectAuthResponse
import com.paystack.android_sdk.core.api.models.TransactionStatus
import com.paystack.android_sdk.core.api.transactions.PaystackTransactionService
import com.paystack.android_sdk.core.api.transactions.RedirectAuthResultListener
import com.paystack.android_sdk.ui.models.Charge
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.coroutines.suspendCoroutine

internal class TransactionRepositoryImpl(
    private val paystackTransactionService: PaystackTransactionService
) : TransactionRepository {
    override suspend fun getByAccessCode(accessCode: String): Result<AccessCodeData> {
        return runCatching {
            paystackTransactionService.verifyAccessCode(accessCode).await()
        }
    }

    override suspend fun chargeCard(
        accessCode: String,
        cardParams: CardParams,
        encryptionKey: String,
    ): Result<Charge> {
        return runCatching {
            val response =
                paystackTransactionService.chargeCard(accessCode, cardParams, encryptionKey).await()
            Charge.from(response)
        }
    }

    override suspend fun checkPendingCharge(accessCode: String): Result<Charge> {
        return runCatching {
            val response = paystackTransactionService.checkPendingCharge(accessCode).await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithPin(
        accessCode: String,
        pin: String,
        encryptionKey: String
    ): Result<Charge> {
        return runCatching {
            val response = paystackTransactionService
                .authenticateWithPin(accessCode, pin, encryptionKey).await()
            Charge.from(response)
        }
    }

    override suspend fun getAddressStateList(countryCode: String): Result<List<AddressState>> {
        return runCatching {
            paystackTransactionService.getAddressStateList(countryCode).await()
        }
    }

    override suspend fun authenticateWithAddress(
        accessCode: String,
        address: String,
        city: String,
        state: String,
        zipCode: String,
    ): Result<Charge> {
        return runCatching {
            val response = paystackTransactionService.authenticateWithAddress(
                accessCode = accessCode,
                address = address,
                city = city,
                state = state,
                zipCode = zipCode
            ).await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithOtp(accessCode: String, otp: String): Result<Charge> {
        return runCatching {
            val response = paystackTransactionService.authenticateWithOtp(accessCode, otp).await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithPhone(
        accessCode: String,
        phoneNumber: String
    ): Result<Charge> {
        return runCatching {
            val response =
                paystackTransactionService.authenticateWithPhone(accessCode, phoneNumber).await()
            Charge.from(response)
        }
    }

    override suspend fun authenticateWithBirthday(
        accessCode: String,
        day: String,
        month: String,
        year: String
    ): Result<Charge> {
        return runCatching {
            val response =
                paystackTransactionService.authenticateWithBirthday(accessCode, day, month, year)
                    .await()
            Charge.from(response)
        }
    }

    override suspend fun await3dsResult(transactionId: String): Result<TransactionStatus> {
        return suspendCoroutine { continuation ->
            val callback = object : RedirectAuthResultListener {
                override fun onResult(response: RedirectAuthResponse) {
                    continuation.resume(Result.success(response.status))
                }

                override fun onError(exception: Throwable) {
                    continuation.resumeWithException(exception)
                }
            }
            paystackTransactionService.listenFor3dsResult(transactionId, callback)
        }
    }
}
