package com.paystack.android_sdk.ui.paymentchannels.card.dateofbirth.views

import androidx.compose.animation.Crossfade
import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android_sdk.ui.R
import com.paystack.android_sdk.ui.components.buttons.PrimaryButton
import com.paystack.android_sdk.ui.components.buttons.StackTextButton
import com.paystack.android_sdk.ui.components.progressbars.LoadingIndicator
import com.paystack.android_sdk.ui.components.views.inputs.PrimaryDropDownMenu
import com.paystack.android_sdk.ui.components.views.inputs.PrimaryTextInputField
import com.paystack.android_sdk.ui.components.views.text.PaymentTitleText
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun DateOfBirthView(
    modifier: Modifier = Modifier,
    viewModel: DateOfBirthViewModel = viewModel(),
    onComplete: () -> Unit,
    onCancel: () -> Unit
) {

    val monthsList by viewModel.monthList.collectAsState()
    val selectedMonthIndex by viewModel.selectedMonthIndex.collectAsState()
    val birthDay by viewModel.dayText.collectAsState()
    val birthYear by viewModel.yearText.collectAsState()
    val isAuthorizationEnabled by viewModel.isAuthorizationButtonEnabled.collectAsState()
    val isSubmitting by viewModel.isSubmitting.collectAsState()
    val result by viewModel.result.collectAsState()

    LaunchedEffect(result) {
        if (result) {
            onComplete()
        }
    }

    Column(modifier = modifier, horizontalAlignment = Alignment.CenterHorizontally) {

        PaymentTitleText(
            titleText = stringResource(id = R.string.date_of_birth_title),
            modifier = Modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double)
        )

        Crossfade(targetState = isSubmitting, label = "DateOfBirthAnim") { showLoadingIndicator ->
            Column(
                horizontalAlignment = Alignment.CenterHorizontally,
                modifier = Modifier.fillMaxWidth()
            ) {
                if (showLoadingIndicator) {
                    Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.triple))

                    LoadingIndicator(
                        modifier = Modifier.size(PaystackTheme.dimensions.spacing.quad),
                        color = PaystackTheme.colors.primaryColors.navy04
                    )
                } else {
                    Image(
                        painter = painterResource(id = R.drawable.dob_auth_logo),
                        contentDescription = stringResource(id = R.string.date_of_birth_auth_logo),
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(PaystackTheme.dimensions.spacing.double),
                        alignment = Alignment.Center,
                    )

                    PrimaryDropDownMenu(
                        modifier = Modifier.fillMaxWidth(),
                        menuItems = monthsList,
                        onItemSelected = { index -> viewModel.setSelectedMonthIndex(index) },
                        label = stringResource(id = R.string.date_of_birth_month_label),
                        selectedItemIndex = selectedMonthIndex
                    )

                    Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

                    Row(
                        modifier = Modifier.fillMaxWidth(),
                        horizontalArrangement = Arrangement
                            .spacedBy(PaystackTheme.dimensions.spacing.double)
                    ) {
                        PrimaryTextInputField(
                            modifier = Modifier.weight(1f),
                            text = birthDay,
                            onTextChanged = viewModel::setDay,
                            label = stringResource(id = R.string.date_of_birth_day_label),
                            placeholder = stringResource(id = R.string.date_of_birth_day_placeholder),
                            keyboardType = KeyboardType.Number,
                            maxLength = DAY_CHARACTER_LIMIT
                        )

                        PrimaryTextInputField(
                            modifier = Modifier.weight(1f),
                            text = birthYear,
                            onTextChanged = viewModel::setBirthYear,
                            label = stringResource(id = R.string.date_of_birth_year_label),
                            placeholder = stringResource(id = R.string.date_of_birth_year_placeholder),
                            keyboardType = KeyboardType.Number,
                            maxLength = YEAR_CHARACTER_LIMIT
                        )
                    }

                    Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))

                    PrimaryButton(
                        buttonText = stringResource(id = R.string.card_payment_authorize_payment),
                        isEnabled = isAuthorizationEnabled,
                        onClick = viewModel::submitData
                    )

                    StackTextButton(
                        modifier = Modifier
                            .fillMaxWidth()
                            .padding(PaystackTheme.dimensions.spacing.double),
                        text = stringResource(id = R.string.card_payment_cancel_transaction),
                        onClick = onCancel
                    )
                }
            }
        }
    }
}

private const val DAY_CHARACTER_LIMIT = 2
private const val YEAR_CHARACTER_LIMIT = 4
