package com.paystack.android_sdk.ui

import android.os.Bundle
import androidx.lifecycle.AbstractSavedStateViewModelFactory
import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import androidx.savedstate.SavedStateRegistryOwner
import com.paystack.android_sdk.core.api.models.AccessCodeData
import com.paystack.android_sdk.core.logging.Logger
import com.paystack.android_sdk.ui.data.transaction.TransactionRepository
import com.paystack.android_sdk.ui.models.Charge
import com.paystack.android_sdk.ui.paymentsheet.PaymentSheetResultContract
import com.paystack.android_sdk.ui.utilities.StringProvider
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch

internal class PaymentsViewModel(
    private val transactionRepository: TransactionRepository,
    private val stringProvider: StringProvider,
    savedStateHandle: SavedStateHandle,
) : ViewModel() {

    private val _paymentUiState = MutableStateFlow<PaymentUiState>(PaymentUiState.Initialization)
    val paymentUiState: StateFlow<PaymentUiState>
        get() = _paymentUiState

    private val _transactionAccessData = MutableStateFlow<AccessCodeData?>(null)
    val transactionAccessData: StateFlow<AccessCodeData?>
        get() = _transactionAccessData

    init {
        val accessCode: String? = savedStateHandle[PaymentSheetResultContract.EXTRA_ACCESS_CODE]
        if (accessCode == null) {
            _paymentUiState.update {
                val error = IllegalStateException("No transaction access code provided")
                PaymentUiState.Error(
                    stringProvider.getString(R.string.could_not_start_transaction),
                    error
                )
            }
        } else {
            loadTransaction(accessCode)
        }
    }

    private fun loadTransaction(accessCode: String) {
        viewModelScope.launch {
            transactionRepository.getByAccessCode(accessCode)
                .onSuccess {
                    _transactionAccessData.value = it
                    startPaymentProcess(it)
                }
                .onFailure { error ->
                    Logger.error(error.message.orEmpty(), error)
                    _paymentUiState.value = PaymentUiState.Error(error)
                }
        }
    }

    private fun startPaymentProcess(accessCodeData: AccessCodeData) {
        _paymentUiState.value = PaymentUiState.PaymentProcess(
            selectedPaymentChannel = "card",
            accessCodeData = accessCodeData,
        )
    }

    fun onTransactionComplete(charge: Charge) {
        _paymentUiState.value = PaymentUiState.Success(
            charge = charge,
            merchantName = transactionAccessData.value?.merchantName.orEmpty()
        )
    }

    class Factory(
        private val transactionRepository: TransactionRepository,
        private val stringProvider: StringProvider,
        savedStateRegistryOwner: SavedStateRegistryOwner,
        defaultArgs: Bundle? = null
    ) : AbstractSavedStateViewModelFactory(savedStateRegistryOwner, defaultArgs) {
        override fun <T : ViewModel> create(
            key: String,
            modelClass: Class<T>,
            handle: SavedStateHandle
        ): T {
            if (modelClass.isAssignableFrom(PaymentsViewModel::class.java)) {
                return PaymentsViewModel(transactionRepository, stringProvider, handle) as T
            }
            throw IllegalArgumentException("Unknown ViewModel class")
        }
    }
}
