package com.paystack.android.ui

import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.SideEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.lifecycle.viewmodel.compose.viewModel
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.PaymentChannel
import com.paystack.android.core.logging.Logger
import com.paystack.android.ui.components.PaymentErrorUi
import com.paystack.android.ui.components.WarningBadge
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.paymentchannels.PaymentChannelUi
import com.paystack.android.ui.paymentchannels.card.CardPaymentUi
import com.paystack.android.ui.paymentchannels.mobilemoney.MobileMoneyScreen
import com.paystack.android.ui.theme.PaystackTheme

@Composable
internal fun PaymentProcessUi(
    viewModelFactory: PaymentProcessViewModel.Factory,
    onComplete: (Charge) -> Unit,
    onError: (Throwable) -> Unit,
) {
    val viewModel: PaymentProcessViewModel = viewModel(factory = viewModelFactory)
    val accessCodeData = viewModel.accessCodeData

    val viewState by viewModel.state.collectAsState()
    val menuItems by viewModel.menuItems.collectAsState()

    if (viewState is PaymentProcessState.Error) {
        SideEffect {
            onError((viewState as PaymentProcessState.Error).error)
        }
    }

    Column(horizontalAlignment = Alignment.CenterHorizontally) {
        if (accessCodeData.isTestTransaction) {
            WarningBadge(text = stringResource(R.string.pstk_test))
            Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))
        }
        when (val state = viewState) {
            is PaymentProcessState.SelectPaymentChannel -> PaymentChannelUi(
                channels = menuItems,
                onChannelClick = {
                    viewModel.onPaymentChannelSelected(
                        it.paymentChannel,
                        it.providerKey
                    )
                }
            )

            is PaymentProcessState.PaymentChannelProcess -> ChannelUiContainer(
                accessCodeData = accessCodeData,
                selectedPaymentChannel = state.selectedPaymentChannel,
                providerKey = state.providerKey,
                onComplete = onComplete,
                onError = onError,
            )

            else -> Logger.debug("Error state: Handled by parent composable")
        }
    }
}

@Composable
private fun ChannelUiContainer(
    accessCodeData: AccessCodeData,
    selectedPaymentChannel: PaymentChannel,
    providerKey: String,
    onComplete: (Charge) -> Unit,
    onError: (Throwable) -> Unit
) {
    when (selectedPaymentChannel) {
        PaymentChannel.CARD -> CardPaymentUi(
            modifier = Modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double),
            onComplete = onComplete,
            onError = onError,
            transactionAccessData = accessCodeData
        )

        PaymentChannel.MOBILE_MONEY -> MobileMoneyScreen(
            accessCodeData = accessCodeData,
            providerKey = providerKey,
            onComplete = onComplete,
            onError = onError,
            modifier = Modifier.fillMaxWidth()
        )

        else -> PaymentErrorUi(
            message = stringResource(R.string.pstk_unsupported_payment_channel),
            modifier = Modifier.fillMaxWidth()
        )
    }
}
