package com.paystack.android.ui

import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.ExperimentalAnimationApi
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.IconButtonDefaults
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import com.paystack.android.ui.components.LoadingScreen
import com.paystack.android.ui.components.PaymentErrorUi
import com.paystack.android.ui.components.success.PaymentSuccessful
import com.paystack.android.ui.components.views.text.SecuredByPaystackText
import com.paystack.android.ui.models.Charge
import com.paystack.android.ui.theme.PaystackTheme
import kotlinx.coroutines.delay

@OptIn(ExperimentalAnimationApi::class)
@Composable
internal fun PaymentContainerUi(
    viewModel: PaymentContainerViewModel,
    modifier: Modifier = Modifier,
    onPaymentComplete: (Charge) -> Unit,
    onPaymentCancelled: () -> Unit,
    onError: (error: Throwable, reference: String?) -> Unit
) {
    val context = LocalContext.current
    val paymentUiState by viewModel.paymentContainerState.collectAsState()
    val transactionAccessData by viewModel.transactionAccessData.collectAsState()

    if (paymentUiState.isTerminal) {
        LaunchedEffect("autoClose") {
            delay(AutoCloseTimeOutMs)
            when (val state = paymentUiState) {
                is PaymentContainerState.Success -> onPaymentComplete(state.charge)
                is PaymentContainerState.Error -> onError(
                    state.error,
                    transactionAccessData?.reference
                )
                else -> { /* Do nothing */
                }
            }
        }
    }

    Column(modifier = modifier) {
        Row(
            modifier = Modifier.padding(
                top = PaystackTheme.dimensions.spacing.half,
                start = PaystackTheme.dimensions.spacing.half,
                end = PaystackTheme.dimensions.spacing.half,
            )
        ) {
            IconButton(
                onClick = onPaymentCancelled,
                colors = IconButtonDefaults.iconButtonColors(
                    contentColor = PaystackTheme.colors.neutralColors.gray01
                )
            ) {
                Icon(
                    painter = painterResource(R.drawable.pstk_ic_close),
                    contentDescription = stringResource(R.string.pstk_close_icon)
                )
            }
        }

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.double))
        Column(
            modifier = Modifier
                .weight(1f)
                .padding(bottom = PaystackTheme.dimensions.spacing.double),
            verticalArrangement = Arrangement.SpaceBetween,
            horizontalAlignment = Alignment.CenterHorizontally,
        ) {
            AnimatedContent(
                targetState = paymentUiState,
                label = "MainContentCrossfade"
            ) { paymentState ->
                when (paymentState) {
                    is PaymentContainerState.Initialization -> LoadingScreen()

                    is PaymentContainerState.PaymentProcess -> {
                        val accessCodeData = paymentState.accessCodeData
                        val vmFactory = PaymentProcessViewModel.Factory(context, accessCodeData)

                        PaymentProcessUi(
                            viewModelFactory = vmFactory,
                            onComplete = viewModel::onTransactionComplete,
                            onError = viewModel::onError
                        )
                    }

                    is PaymentContainerState.Success -> PaymentSuccessful(
                        amount = paymentState.charge.amount,
                        currencyCode = paymentState.charge.currency,
                        merchantName = paymentState.merchantName,
                    )

                    is PaymentContainerState.Error -> PaymentErrorUi(
                        message = paymentState.displayText,
                        modifier = Modifier.fillMaxWidth()
                    )
                }
            }

            SecuredByPaystackText()
        }
    }
}

private const val AutoCloseTimeOutMs = 2000L
