package com.paystack.android_sdk.core.api

import com.paystack.android_sdk.core.api.core.exceptions.NoResponseBodyError
import com.paystack.android_sdk.core.api.extensions.ApiResponse
import com.paystack.android_sdk.core.api.models.ErrorResponse
import com.paystack.android_sdk.core.api.models.PaystackError
import com.paystack.android_sdk.core.api.models.asPaystackError
import kotlinx.serialization.SerializationException
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json
import okhttp3.ResponseBody
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response

internal class RetrofitApiRequest<T> constructor(
    private val call: Call<ApiResponse<T>>
) : ApiRequest<T> {
    private val json = Json { ignoreUnknownKeys = true }

    override fun execute(callback: ApiRequestCallback<T>) {
        call.enqueue(object : Callback<ApiResponse<T>> {
            override fun onResponse(
                call: Call<ApiResponse<T>>,
                response: Response<ApiResponse<T>>
            ) {
                if (response.isSuccessful) {
                    handleSuccessfulResponse(response.body(), callback)
                } else {
                    handleErrorResponse(response.errorBody(), callback)
                }
            }

            override fun onFailure(call: Call<ApiResponse<T>>, exception: Throwable) {
                callback.onError(exception.asPaystackError())
            }
        })
    }

    private fun handleSuccessfulResponse(
        apiResponse: ApiResponse<T>?,
        callback: ApiRequestCallback<T>
    ) {
        val data = apiResponse?.data
        if (data != null) {
            callback.onSuccess(data)
        } else {
            callback.onError(NoResponseBodyError().asPaystackError())
        }
    }

    private fun handleErrorResponse(
        errorBody: ResponseBody?,
        callback: ApiRequestCallback<T>
    ) {
        val errorStr = errorBody?.string() ?: run {
            callback.onError(PaystackError("An error occurred"))
            return
        }
        val error = try {
            val errorResponse = json.decodeFromString<ErrorResponse>(errorStr)
            PaystackError(errorResponse.message)
        } catch (e: SerializationException) {
            e.asPaystackError()
        }
        callback.onError(error)
    }
}

internal fun <T> Call<ApiResponse<T>>.asApiRequest(): ApiRequest<T> {
    return RetrofitApiRequest(this)
}
