package com.paystack.android.core.api.services.transactions

import com.paystack.android.core.api.ApiComponent
import com.paystack.android.core.api.ApiRequest
import com.paystack.android.core.api.asApiRequest
import com.paystack.android.core.api.models.AccessCodeData
import com.paystack.android.core.api.models.ChargeResponse
import com.paystack.android.core.api.models.PaymentChannel
import com.paystack.android.core.api.models.TransactionEvent
import com.paystack.android.core.events.EventManager
import com.paystack.android.core.events.EventsComponent
import com.paystack.android.core.events.Subscription
import com.paystack.android.core.events.SubscriptionParams
import kotlinx.serialization.decodeFromString
import kotlinx.serialization.json.Json

internal class PaystackTransactionServiceImpl(
    private val transactionApi: TransactionApi,
    private val eventManager: EventManager,
) : PaystackTransactionService {
    private val json = Json { ignoreUnknownKeys = true }

    constructor(apiComponent: ApiComponent, eventsComponent: EventsComponent) : this(
        apiComponent.transactionApi,
        eventsComponent.eventManager
    )

    override fun verifyAccessCode(accessCode: String): ApiRequest<AccessCodeData> {
        return transactionApi.verifyAccessCode(accessCode).asApiRequest()
    }

    override fun checkPendingCharge(accessCode: String): ApiRequest<ChargeResponse> {
        return transactionApi.checkPendingCharge(accessCode).asApiRequest()
    }

    override fun getSupportedPaymentChannels(): List<PaymentChannel> {
        return listOf(
            PaymentChannel.CARD,
            PaymentChannel.MOBILE_MONEY
        )
    }

    override fun listenForTransactionEvent(
        channnel: String,
        callback: TransactionEventListener
    ) {
        var subscription: Subscription? = null
        subscription = eventManager.subscribe(
            params = SubscriptionParams(
                channelName = channnel,
                eventName = "response",
            ),
            onEvent = { result ->
                runCatching { json.decodeFromString<TransactionEvent>(result) }
                    .onSuccess { transaction ->
                        callback.onResult(transaction)
                        subscription?.cancel()
                    }
                    .onFailure {
                        callback.onError(it)
                        subscription?.cancel()
                    }
            }
        )
    }
}
