/*
 * Decompiled with CFR 0.152.
 */
package com.orbitz.consul.cache;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.orbitz.consul.ConsulException;
import com.orbitz.consul.async.ConsulResponseCallback;
import com.orbitz.consul.cache.CacheDescriptor;
import com.orbitz.consul.config.CacheConfig;
import com.orbitz.consul.model.ConsulResponse;
import com.orbitz.consul.monitoring.ClientEventHandler;
import com.orbitz.consul.option.ImmutableQueryOptions;
import com.orbitz.consul.option.QueryOptions;
import java.math.BigInteger;
import java.time.Duration;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsulCache<K, V>
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(ConsulCache.class);
    private final AtomicReference<BigInteger> latestIndex = new AtomicReference<Object>(null);
    private final AtomicLong lastContact = new AtomicLong();
    private final AtomicBoolean isKnownLeader = new AtomicBoolean();
    private final AtomicReference<ImmutableMap<K, V>> lastResponse = new AtomicReference<Object>(null);
    private final AtomicReference<State> state = new AtomicReference<State>(State.latent);
    private final CountDownLatch initLatch = new CountDownLatch(1);
    private final ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setNameFormat("consulCacheScheduledCallback-%d").setDaemon(true).build());
    private final CopyOnWriteArrayList<Listener<K, V>> listeners = new CopyOnWriteArrayList();
    private final ReentrantLock listenersStartingLock = new ReentrantLock();
    private final Stopwatch stopWatch = Stopwatch.createUnstarted();
    private final Function<V, K> keyConversion;
    private final CallbackConsumer<V> callBackConsumer;
    private final ConsulResponseCallback<List<V>> responseCallback;
    private final ClientEventHandler eventHandler;
    private final CacheDescriptor cacheDescriptor;

    protected ConsulCache(Function<V, K> keyConversion, CallbackConsumer<V> callbackConsumer, final CacheConfig cacheConfig, final ClientEventHandler eventHandler, final CacheDescriptor cacheDescriptor) {
        if (keyConversion == null) {
            Validate.notNull(keyConversion, (String)"keyConversion must not be null", (Object[])new Object[0]);
        }
        if (callbackConsumer == null) {
            Validate.notNull(callbackConsumer, (String)"callbackConsumer must not be null", (Object[])new Object[0]);
        }
        if (cacheConfig == null) {
            Validate.notNull((Object)cacheConfig, (String)"cacheConfig must not be null", (Object[])new Object[0]);
        }
        if (eventHandler == null) {
            Validate.notNull((Object)eventHandler, (String)"eventHandler must not be null", (Object[])new Object[0]);
        }
        this.keyConversion = keyConversion;
        this.callBackConsumer = callbackConsumer;
        this.eventHandler = eventHandler;
        this.cacheDescriptor = cacheDescriptor;
        this.responseCallback = new ConsulResponseCallback<List<V>>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onComplete(ConsulResponse<List<V>> consulResponse) {
                if (consulResponse.isKnownLeader()) {
                    if (!ConsulCache.this.isRunning()) {
                        return;
                    }
                    Duration elapsedTime = ConsulCache.this.stopWatch.elapsed();
                    ConsulCache.this.updateIndex(consulResponse);
                    LOGGER.debug("Consul cache updated for {} (index={}), request duration: {} ms", new Object[]{cacheDescriptor, ConsulCache.this.latestIndex, elapsedTime.toMillis()});
                    ImmutableMap full = ConsulCache.this.convertToMap(consulResponse);
                    boolean changed = !full.equals(ConsulCache.this.lastResponse.get());
                    eventHandler.cachePollingSuccess(cacheDescriptor, changed, elapsedTime);
                    if (changed) {
                        ConsulCache.this.lastResponse.set(full);
                        ConsulCache.this.lastContact.set(consulResponse.getLastContact());
                        ConsulCache.this.isKnownLeader.set(consulResponse.isKnownLeader());
                    }
                    if (changed) {
                        Boolean locked = false;
                        if (ConsulCache.this.state.get() == State.starting) {
                            ConsulCache.this.listenersStartingLock.lock();
                            locked = true;
                        }
                        try {
                            for (Listener l : ConsulCache.this.listeners) {
                                try {
                                    l.notify(full);
                                }
                                catch (RuntimeException e) {
                                    LOGGER.warn("ConsulCache Listener's notify method threw an exception.", (Throwable)e);
                                }
                            }
                        }
                        finally {
                            if (locked.booleanValue()) {
                                ConsulCache.this.listenersStartingLock.unlock();
                            }
                        }
                    }
                    if (ConsulCache.this.state.compareAndSet(State.starting, State.started)) {
                        ConsulCache.this.initLatch.countDown();
                    }
                    Duration timeToWait = cacheConfig.getMinimumDurationBetweenRequests();
                    if ((consulResponse.getResponse() == null || consulResponse.getResponse().isEmpty()) && cacheConfig.getMinimumDurationDelayOnEmptyResult().compareTo(timeToWait) > 0) {
                        timeToWait = cacheConfig.getMinimumDurationDelayOnEmptyResult();
                    }
                    timeToWait = timeToWait.minus(elapsedTime);
                    ConsulCache.this.executorService.schedule(() -> ConsulCache.this.runCallback(), timeToWait.toMillis(), TimeUnit.MILLISECONDS);
                } else {
                    this.onFailure(new ConsulException("Consul cluster has no elected leader"));
                }
            }

            @Override
            public void onFailure(Throwable throwable) {
                if (!ConsulCache.this.isRunning()) {
                    return;
                }
                eventHandler.cachePollingError(cacheDescriptor, throwable);
                long delayMs = ConsulCache.computeBackOffDelayMs(cacheConfig);
                String message = String.format("Error getting response from consul for %s, will retry in %d %s", new Object[]{cacheDescriptor, delayMs, TimeUnit.MILLISECONDS});
                cacheConfig.getRefreshErrorLoggingConsumer().accept(LOGGER, message, throwable);
                ConsulCache.this.executorService.schedule(() -> ConsulCache.this.runCallback(), delayMs, TimeUnit.MILLISECONDS);
            }
        };
    }

    static long computeBackOffDelayMs(CacheConfig cacheConfig) {
        return cacheConfig.getMinimumBackOffDelay().toMillis() + Math.round(Math.random() * (double)cacheConfig.getMaximumBackOffDelay().minus(cacheConfig.getMinimumBackOffDelay()).toMillis());
    }

    public void start() {
        Preconditions.checkState((boolean)this.state.compareAndSet(State.latent, State.starting), (String)"Cannot transition from state %s to %s", (Object)((Object)this.state.get()), (Object)((Object)State.starting));
        this.eventHandler.cacheStart(this.cacheDescriptor);
        this.runCallback();
    }

    public void stop() {
        this.eventHandler.cacheStop(this.cacheDescriptor);
        State previous = this.state.getAndSet(State.stopped);
        if (this.stopWatch.isRunning()) {
            this.stopWatch.stop();
        }
        if (previous != State.stopped) {
            this.executorService.shutdownNow();
        }
    }

    @Override
    public void close() {
        this.stop();
    }

    private void runCallback() {
        if (this.isRunning()) {
            this.stopWatch.reset().start();
            this.callBackConsumer.consume(this.latestIndex.get(), this.responseCallback);
        }
    }

    private boolean isRunning() {
        return this.state.get() == State.started || this.state.get() == State.starting;
    }

    public boolean awaitInitialized(long timeout, TimeUnit unit) throws InterruptedException {
        return this.initLatch.await(timeout, unit);
    }

    public ImmutableMap<K, V> getMap() {
        return this.lastResponse.get();
    }

    public ConsulResponse<ImmutableMap<K, V>> getMapWithMetadata() {
        return new ConsulResponse<ImmutableMap<K, V>>(this.lastResponse.get(), this.lastContact.get(), this.isKnownLeader.get(), this.latestIndex.get());
    }

    @VisibleForTesting
    ImmutableMap<K, V> convertToMap(ConsulResponse<List<V>> response) {
        if (response == null || response.getResponse() == null || response.getResponse().isEmpty()) {
            return ImmutableMap.of();
        }
        ImmutableMap.Builder builder = ImmutableMap.builder();
        HashSet<K> keySet = new HashSet<K>();
        for (V v : response.getResponse()) {
            K key = this.keyConversion.apply(v);
            if (key != null) {
                if (!keySet.contains(key)) {
                    builder.put(key, v);
                } else {
                    LOGGER.warn("Duplicate service encountered. May differ by tags. Try using more specific tags? " + key.toString());
                }
            }
            keySet.add(key);
        }
        return builder.build();
    }

    private void updateIndex(ConsulResponse<List<V>> consulResponse) {
        if (consulResponse != null && consulResponse.getIndex() != null) {
            this.latestIndex.set(consulResponse.getIndex());
        }
    }

    protected static QueryOptions watchParams(BigInteger index, int blockSeconds, QueryOptions queryOptions) {
        Preconditions.checkArgument((!queryOptions.getIndex().isPresent() && !queryOptions.getWait().isPresent() ? 1 : 0) != 0, (Object)"Index and wait cannot be overridden");
        ImmutableQueryOptions.Builder builder = ImmutableQueryOptions.builder().from(ConsulCache.watchDefaultParams(index, blockSeconds)).token(queryOptions.getToken()).consistencyMode(queryOptions.getConsistencyMode()).near(queryOptions.getNear()).datacenter(queryOptions.getDatacenter());
        for (String tag : queryOptions.getTag()) {
            builder.addTag(tag);
        }
        return builder.build();
    }

    private static QueryOptions watchDefaultParams(BigInteger index, int blockSeconds) {
        if (index == null) {
            return QueryOptions.BLANK;
        }
        return QueryOptions.blockSeconds(blockSeconds, index).build();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean addListener(Listener<K, V> listener) {
        boolean added;
        Boolean locked = false;
        if (this.state.get() == State.starting) {
            this.listenersStartingLock.lock();
            locked = true;
        }
        try {
            added = this.listeners.add(listener);
            if (this.state.get() == State.started) {
                try {
                    listener.notify((Map)this.lastResponse.get());
                }
                catch (RuntimeException e) {
                    LOGGER.warn("ConsulCache Listener's notify method threw an exception.", (Throwable)e);
                }
            }
        }
        finally {
            if (locked.booleanValue()) {
                this.listenersStartingLock.unlock();
            }
        }
        return added;
    }

    public List<Listener<K, V>> getListeners() {
        return Collections.unmodifiableList(this.listeners);
    }

    public boolean removeListener(Listener<K, V> listener) {
        return this.listeners.remove(listener);
    }

    @VisibleForTesting
    protected State getState() {
        return this.state.get();
    }

    public static interface Listener<K, V> {
        public void notify(Map<K, V> var1);
    }

    protected static interface CallbackConsumer<V> {
        public void consume(BigInteger var1, ConsulResponseCallback<List<V>> var2);
    }

    static enum State {
        latent,
        starting,
        started,
        stopped;

    }
}

