/* Copyright (c) 2011, 2025, Oracle and/or its affiliates. */
/* All rights reserved.*/

/*
 * DESCRIPTION
 *   Represents the pl/sql used to construct the database metadata for
 *   a table. 
 *   
 * 
 *   NOTES
 *     1: Set 3 parameters when using a prepared statement
 *        input params: schema, table ( in order)
 *        output param: clob ( output clob containing db metadata for table)
 
 *     2: DO NOT use any -- style comments in the pl/sql code as this 
 *        comments out everything after the comment when passed to 
 *        a Java String. 
 *
 *     3: To run this code as a sql script: minor edits are required
 *        see embedded "uncomment" comments and comment out the input/output
 *        parameter lines above them.  
 *
*/

declare
  /*
   *  xml document tags
   *
   * variables represent tags of the xml returned by  dbms_metadata.get_xml()
   * the lengths are used to copy data of  <TABLE_T> node
   * from the xml clob
   */
  xmlHeader varchar2(100) := '<?xml version="1.0"?>';
  xmlHeaderLen integer := length(xmlHeader) +1;
  startRowRowSetTag varchar2(100) := '<ROWSET><ROW>' || chr(10);
  endRowRowSetTag varchar2(100) := '</ROW></ROWSET>';
  endRowRowSetLen integer := length(endRowRowSetTag);

  /* chr(10) is carriage return - used for formatting the output
   * Use UTF-8 encoding
   */
  l_xmlheader varchar2(100) :=  '<?xml version="1.0" encoding="UTF-8"?>'
                                   || chr(10);
  
                                   
  /* Root element */
  l_rootStr varchar2(200) := '<DATABASE>' || chr(10);
   

  /* local variables */
  l_xt xmltype;
  l_table_clob CLOB;
  l_type_clob CLOB;
  l_src_length integer;
  l_dest_offset integer;
  l_src_offset integer;

  /* STRMTABLE metadata for datapump formats */
  l_strm_clob CLOB; /* stores metadata for datapump stream  */  
  l_empty_strmtable  varchar2(100) := chr(10) || '   <STRMTABLE_T/>' || chr(10);   

  /* TABLE_T tags */
  l_table_t_start_tag        varchar2(100) := '<TABLE_T>';
  l_table_t_end_tag          varchar2(100) := '</TABLE_T>';

  /* FULL_TYPE_T tags for varray */
  l_full_type_t_start_tag    varchar2(100) := '<FULL_TYPE_T>';
  l_full_type_t_end_tag      varchar2(100) := '</FULL_TYPE_T>';

  /*  DB properties tags */
  l_db_props_start_tag       varchar2(100) := chr(10) || '  <DB_PROPERTIES>' || chr(10); 
  l_db_props_end_tag         varchar2(100) := '  </DB_PROPERTIES>' || chr(10);
  l_db_prop_start_tag        varchar2(100) := '    <DB_PROPERTY>';
  l_db_prop_end_tag          varchar2(100) := '    </DB_PROPERTY>';
  l_prop_name_start_tag      varchar2(100) := '      <PROPERTY_NAME>';
  l_prop_name_end_tag        varchar2(100) := '</PROPERTY_NAME>';
  l_prop_value_start_tag     varchar2(100) := '      <PROPERTY_VALUE>';
  l_prop_value_end_tag       varchar2(100) := '</PROPERTY_VALUE>';
  l_prop_name                varchar2(100);
  l_prop_value               varchar2(4000);
  l_propid_name              varchar2(100); 
  l_dbcontext_prop_name      varchar2(100); 
  l_charset_id               number;
  l_database_str varchar2(100) := chr(10) || '</DATABASE>';

  /* STRMTABLE_T tags */
  l_strmtable_t_start_tag    varchar2(100) := '<STRMTABLE_T>';
  l_strmtable_t_end_tag      varchar2(100) := '</STRMTABLE_T>';

  /* input/output parameters */
  p_table_name varchar2(128);  /* support long id */
  p_schema_name varchar2(128); /* support long id */
  p_banner varchar2(400);
  p_dest_clob CLOB;

  /* parameter to control debug behavior 
   * oracle JDBC driver does not support passing Boolean to plsql
   * http://docs.oracle.com/cd/B28359_01/java.111/b31224/apxtblsh.htm#i1005380
   * an Integer is used as a workaround */
  p_debug                    int := 0; /*0 false 1 true*/
    
  /* temp variables to calculate pad characters */    
  l_pad char(2) := 'x';               /* second char is pad */        
  l_npad nchar(2) := to_nchar('x');   /* second nchar is pad */
  l_pos integer;
  l_dmp varchar2(20); 
  l_ndmp nvarchar2(20);
  /* using a large value for port_str, since max length is port-specific */
  l_port_str varchar2(500);

  /* temp variable to store DBversion */
  l_dbver varchar2(100);


begin
   /* Get input parameters */
   p_schema_name := ?;
   p_table_name := ?;
   p_banner := ?; 
   p_debug  := ?;
   /*p_schema_name := '&1'; uncomment for sqlplus */   
   /*p_table_name := '&2';  uncomment for sqlplus */
   /*p_banner := '&3';  uncomment for sqlplus */
   /*p_debug := 1; uncomment for sqlplus */
   
   

   if (p_schema_name is null) then
     p_schema_name := USER;
   end if;

 
  /*
   *  Step 1: create dest Clob and write document header
   */
  sys.dbms_lob.createTemporary(p_dest_clob, true, sys.dbms_lob.call);
  dbms_lob.writeAppend(p_dest_clob, length(l_xmlheader), l_xmlheader);
  dbms_lob.writeAppend(p_dest_clob, length(p_banner), p_banner);
  dbms_lob.writeAppend(p_dest_clob, length(l_rootStr), l_rootStr);


  /* 
   * step 2: Get table metadata from dbms_metadata and write to dest CLOB
   */

  /* Append <ROWSET><ROW> first*/
  dbms_lob.writeAppend(p_dest_clob, length(startRowRowSetTag), startRowRowSetTag);


  l_table_clob := dbms_metadata.get_xml('TABLE', p_table_name, p_schema_name);

  /*
   * write table metadata to output clob 
   * calculate length and offsets for the <TABLE_T> fragment.
   * the offset is the position of tag <TABLE_T>, and the length is
   * position of </TABLE_T> + its length - position of <TABLE_T>     
   */
  l_src_offset := dbms_lob.instr(l_table_clob,l_table_t_start_tag);
  l_src_length := dbms_lob.instr(l_table_clob, l_table_t_end_tag) - 
                  l_src_offset + length(l_table_t_end_tag);

  /*
   * calculate the offset in the dest clob
   * data is appended to the end of the dest CLOB
   */
  l_dest_offset := dbms_lob.getLength(p_dest_clob)+1;

  /*
   * copy table metadata Clob into the destination CLOB
   * copy from <ROW><ROWSET> tags until(but not including) </ROW></ROWSET> tags
   */
  dbms_lob.copy(p_dest_clob,l_table_clob,
                l_src_length, l_dest_offset, l_src_offset);



  /*
   * step 3: Get varray types info and write to dest Clob 
   */ 

  /* get type info about varray columns
   * extract varray columns from the table metadata 
   */
  l_xt := xmltype(l_table_clob);
  for cur in (select *
               from xmltable
               ('/ROWSET/ROW/TABLE_T/COL_LIST/COL_LIST_ITEM'
                passing l_xt
                columns
                       name   varchar2(128)  path 'NAME',
                   type_num        number   path 'TYPE_NUM',
               type_md_name   varchar2(128)  path 'TYPEMD/SCHEMA_OBJ/NAME',
              type_md_owner   varchar2(128)  path 'TYPEMD/SCHEMA_OBJ/OWNER_NAME',
                type_md_num   varchar2(3)   path 'TYPEMD/SCHEMA_OBJ/TYPE_NUM'
              ) where type_num=123   ) loop 

     /* for each varray column get the type info */
     l_type_clob := dbms_metadata.get_xml('TYPE',
                                          cur.type_md_name,
                                          cur.type_md_owner);

    /* calculate length of type def data
     * calculate length and offsets for the <FULL_TYPE_T> fragment.
     * the offset is the position of tag <FULL_TYPE_T>, and the length is
     * position of </FULL_TYPE_T> + its length - position of <FULL_TYPE_T>  
     */
     l_src_offset := dbms_lob.instr(l_type_clob, l_full_type_t_start_tag);
     l_src_length := dbms_lob.instr(l_type_clob, l_full_type_t_end_tag) -
                     l_src_offset + length(l_full_type_t_end_tag);


     /* calculate the dest offset in the dest CLOB
      * the type info is appended to end the of the dest CLOB
      */
     l_dest_offset := dbms_lob.getLength(p_dest_clob)+1;    

    /* copy type info from src CLOB into dest CLOB */
     dbms_lob.copy(p_dest_clob,
                  l_type_clob,
                  l_src_length,
                  l_dest_offset,
                  l_src_offset );

  end loop;
  
  
  /* 
   * Step 4: write the end </ROW></ROWSET> tag
   */ 
  dbms_lob.writeAppend(p_dest_clob, endRowRowSetLen, endRowRowSetTag);


  /*======================================================================= 
   * Step 5: write database properties 
   *========================================================================
   */

  /*
   * write the result of the following query as an xml fragment into a CLOB
   *    select * from database_properties 
   *    where property_name like 'NLS_%' or property_name = 'DBTIMEZONE'
   * output: xml fragment
   * <DB_PROPERTIES>
   *   <DB_PROPERTY>
   *     <PROPERTY_NAME>property name  </PROPERTY_NAME>
   *     <PROPERTY_VALUE>property value </PROPERTY_VALUE>
   *   <DB_PROPERTY>
   * </DB_PROPERTIES>
   *
   * write <DB_PROPERTIES> tag
   */
  dbms_lob.writeAppend(p_dest_clob, length(l_db_props_start_tag), l_db_props_start_tag);

  /*
   *required properties for NLSContext, from oracle.hadoop.loader.metadata.Enum
   */

  for cur in (select property_name, property_value,
      decode (property_name,
			'NLS_CALENDAR', 'NLSCalendar',
			'NLS_DATE_FORMAT', 'DateFormat',
			'NLS_TIMESTAMP_FORMAT', 'TimestampFormat',
			'NLS_TIMESTAMP_TZ_FORMAT', 'TimestampTZFormat',
			'DBTIMEZONE', 'DBTimezone') as decode_name
   from SYS.DATABASE_PROPERTIES
      where property_name in 
      ('NLS_CALENDAR', 'NLS_CHARACTERSET',
      'NLS_COMP', 'NLS_CURRENCY',
      'NLS_DATE_FORMAT', 'NLS_DATE_LANGUAGE',
      'NLS_DUAL_CURRENCY', 'NLS_ISO_CURRENCY',
      'NLS_LANGUAGE', 'NLS_LENGTH_SEMANTICS',
      'NLS_NCHAR_CHARACTERSET', 'NLS_NCHAR_CONV_EXCP',
      'NLS_NUMERIC_CHARACTERS', 'NLS_RDBMS_VERSION',
      'NLS_SORT', 'NLS_TERRITORY',
      'NLS_TIMESTAMP_FORMAT',  'NLS_TIMESTAMP_TZ_FORMAT',
      'NLS_TIME_FORMAT', 'NLS_TIME_TZ_FORMAT',
      'DBTIMEZONE')) loop
   
    l_prop_name :=      l_db_prop_start_tag   || chr(10)
                     || l_prop_name_start_tag || cur.property_name
                     || l_prop_name_end_tag   || chr(10);

    l_prop_value :=     l_prop_value_start_tag || cur.property_value 
                     || l_prop_value_end_tag   || chr(10)
                     || l_db_prop_end_tag      || chr(10);

    dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
    dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);

    if cur.decode_name is not null 
     then 
        l_prop_name :=   l_db_prop_start_tag   || chr(10)
                      || l_prop_name_start_tag || cur.decode_name
                      || l_prop_name_end_tag   || chr(10);

        l_prop_value :=  l_prop_value_start_tag || cur.property_value 
                      || l_prop_value_end_tag   || chr(10)
                      || l_db_prop_end_tag      || chr(10);
         dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
         dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);
    end if;

 
    /*
     * get character set id and write it as a DB_PROPERTY
     */
    
    if (cur.property_name in ('NLS_CHARACTERSET', 'NLS_NCHAR_CHARACTERSET') )
    then
      l_charset_id := nls_charset_id(cur.property_value);
      
      l_propid_name := cur.property_name || '_ID'; 
      l_prop_name :=      l_db_prop_start_tag   || chr(10)
                       || l_prop_name_start_tag || l_propid_name
                       || l_prop_name_end_tag   || chr(10);

      l_prop_value :=     l_prop_value_start_tag || l_charset_id 
                       || l_prop_value_end_tag   || chr(10)
                       || l_db_prop_end_tag      || chr(10);

      dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
      dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);

      /*property names for DBContext*/

      if l_propid_name = 'NLS_CHARACTERSET_ID' then
         l_dbcontext_prop_name := 'CharSetID';
      elsif l_propid_name = 'NLS_NCHAR_CHARACTERSET_ID' then
         l_dbcontext_prop_name := 'NCharSetID';
      end if;

      l_prop_name :=   l_db_prop_start_tag   || chr(10)
                    || l_prop_name_start_tag || l_dbcontext_prop_name
                    || l_prop_name_end_tag   || chr(10);

      l_prop_value :=  l_prop_value_start_tag || l_charset_id
                    || l_prop_value_end_tag   || chr(10)
                    || l_db_prop_end_tag      || chr(10);

      dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
      dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);
      
    end if;
  end loop;

    /*
     * get DBVersion for DBContext
     */

    select version into l_dbver from SYS.PRODUCT_COMPONENT_VERSION 
      where product like 'Oracle%Database%';
      l_prop_name :=      l_db_prop_start_tag   || chr(10)
                       || l_prop_name_start_tag || 'DBVersion'
                       || l_prop_name_end_tag   || chr(10);

      l_prop_value :=     l_prop_value_start_tag || l_dbver 
                       || l_prop_value_end_tag   || chr(10)
                       || l_db_prop_end_tag      || chr(10);

      dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
      dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);

  /*========================================================================
   * Step 5b calculate pad characters - hack
   * we do not write CharPad NCharPad here because DBContext has 
   * different de-ser method from NLSContext. We relied on 
   * cs.convert(" ") and ncs.convert(" ") in DBContext to get them
   *========================================================================
   */
  /* CHAR, select and dump the pad char, number 2   */
  select dump(substr(l_pad,2)) into l_dmp from sys.dual;
  l_pos := 1 + instr(l_dmp, ':');
  l_dmp := trim(substr(l_dmp, l_pos));

  l_prop_name :=      l_db_prop_start_tag   || chr(10)
                   || l_prop_name_start_tag || 'CHAR_PAD'
                   || l_prop_name_end_tag   || chr(10);

  l_prop_value :=     l_prop_value_start_tag || l_dmp
                   || l_prop_value_end_tag   || chr(10)
                   || l_db_prop_end_tag      || chr(10);

  dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
  dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);

  /* NCHAR, select and dump the pad char, number 2  */
  select dump(substrc(l_npad,2)) into l_ndmp from sys.dual;
  l_pos := 1 + instrc(l_ndmp, to_nchar(':'));
  l_ndmp := trim(substrc(l_ndmp, l_pos));
  
  l_prop_name :=      l_db_prop_start_tag   || chr(10)
                   || l_prop_name_start_tag || 'NCHAR_PAD'
                   || l_prop_name_end_tag   || chr(10);

  l_prop_value :=     l_prop_value_start_tag || to_char(l_ndmp)
                   || l_prop_value_end_tag   || chr(10)
                   || l_db_prop_end_tag      || chr(10);

  dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
  dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);

  /*========================================================================
   * Step 5c : write the port_string ( platform/OS info) for datapump
   *========================================================================
   */
  
  select dbms_utility.port_string into l_port_str from sys.dual;

  l_prop_name :=      l_db_prop_start_tag   || chr(10)
                   || l_prop_name_start_tag || 'PORT_STRING'
                   || l_prop_name_end_tag   || chr(10);

  l_prop_value :=     l_prop_value_start_tag || l_port_str
                   || l_prop_value_end_tag   || chr(10)
                   || l_db_prop_end_tag      || chr(10);

  dbms_lob.writeAppend(p_dest_clob, length(l_prop_name), l_prop_name);
  dbms_lob.writeAppend(p_dest_clob, length(l_prop_value), l_prop_value);
  
  /*write </DB_PROPERTIES> tag */
  dbms_lob.writeAppend(p_dest_clob, length(l_db_props_end_tag), l_db_props_end_tag);


  

  /*========================================================================
   * Step 6: get stream metadata for datapump formats
   * for OLH we use default values for all params except for 
   * force_no_encrpt( TRUE) which indicates the column data is not encrypted.
   * since OLH cannot meaningfully generate encrypted data.
   *
   * copy-pasted doc from dbmsmeta.sql for convenience
   *
   * GET_DPSTRM_MD: Retrieve stream (i.e., table) metadata for DataPump data 
   *                layer.
   * PARAMETERS:
   *     schema    - the table's schema
   *     name      - the table's name
   *     mdversion - the version of metadata to be extracted, one of
   *               - COMPATIBLE (default) - version of metadata
   *                       corresponds to database compatibility level
   *               - LATEST - corresponds to database version
   *                        - a specific database version
   *     dpapiversion - the direct path API version (this value is stored
   *                    in the returned XML document)
   *     doc    - An XML document containing the stream metadata
   *              for the table/partition.
   *     network_link - The name of a database link to the database
   *                   whose data is to be retrieved.  If NULL (default),
   *                   metadata is retrieved from the database on which the 
   *                   caller is running.
   *     force_lob_be - if TRUE, clear bit 0x0200 of
   *                    COL_LIST/COL_LIST_ITEM/LOB_PROPERTY, i.e., force the
   *                    metadata  to make the lob appear big endian.
   *     force_no_encrypt - if TRUE, clear encryption bits in col$ properties:
   *                       0x04000000 =  67108864 = Column is encrypted
   *                       0x20000000 = 536870912 = Column is encrypted without
   *                       salt
   *                       This is necessary when users do not specify an
   *                       encryption_password and the data is written to the 
   *                       dumpfile in clear text although the col properity 
   *                       retains the encrypt property.
   *         get_dpstrm_md (
   *               IN VARCHAR2,
   *               IN VARCHAR2,
   *               IN VARCHAR2 DEFAULT 'COMPATIBLE',
   *               IN NUMBER DEFAULT 3,
   *               IN OUT NOCOPY CLOB,
   *               IN VARCHAR2 DEFAULT NULL,
   *               IN BOOLEAN DEFAULT FALSE,
   *               IN BOOLEAN DEFAULT FALSE);
   *========================================================================
   */

  begin
    /* Allocate temporary lob */
    sys.dbms_lob.createTemporary(l_strm_clob, true, sys.dbms_lob.call);

    /*get datapump stream metadata  */
    dbms_metadata.get_dpstrm_md(p_schema_name,
                                p_table_name,
                               'COMPATIBLE',   /* DEFAULT */
                                3,             /* DEFAULT */  
                                l_strm_clob,
                                null,          /* DEFAULT */
                                true,          /* DEFAULT */
                                false          /* NON-DEFAULT */
             );

    if (l_strm_clob is null) then
     /* happens if table/schema does not exist
      * but write an empty element <STRMTABLE_T/>
      * to be consistent with empty <TABLE_T/> for table metadata
      */
      dbms_lob.writeAppend( p_dest_clob,
                           length(l_empty_strmtable),
                           l_empty_strmtable); 
    else 

      /* dbms_metadata.get_dpstrm_md() produces a well-formed xml document
       * To generate a well-formed xml document for our metadata we need to
       * to strip out the xml header and <ROW><ROWSET> tags from the 
       * the stream metadata. 
       */



       /* calculate length and offsets for the <STRMTABLE_T> fragment.
        * the offset is the position of tag <STRMTABLE_T>, and the length is
        * position of </STRMTABLE_T> + its length - position of <STRMTABLE_T>
        */

       l_src_offset := dbms_lob.instr(l_strm_clob, l_strmtable_t_start_tag);
       l_src_length := dbms_lob.instr(l_strm_clob, l_strmtable_t_end_tag)-
                       l_src_offset + length(l_strmtable_t_end_tag);

       /* calculate the dest offset in the dest CLOB
        * the strm info is appended to end the of the output dest CLOB
        */
       l_dest_offset := dbms_lob.getLength(p_dest_clob) + 1;    

       /* copy dpstream info from src CLOB into dest CLOB */
       dbms_lob.copy(p_dest_clob,
                    l_strm_clob,
                    l_src_length,
                    l_dest_offset,
                    l_src_offset );
       
    end if;
   
     
     /*  free the temp lob */
    if (dbms_lob.istemporary(l_strm_clob) <> 0) then
      dbms_lob.freetemporary(l_strm_clob);
    end if;

   end;
    




  /*========================================================================
   * Step 7: write Document end tag 
   *========================================================================
   */
  dbms_lob.writeAppend(p_dest_clob, length(l_database_str), l_database_str);

  /*========================================================================
   * Step 8: cleanup and check dest xml 
   *========================================================================
   */
  if (dbms_lob.isTemporary(l_table_clob) <> 0 ) then
    dbms_lob.freeTemporary(l_table_clob);
  end if;

  /* p_debug = 1, then the database checks if the XMLType is wellformed */
  if p_debug = 1 then
    l_xt := XMLType(p_dest_clob);
  end if;

 
  /* 
   * Assign result to OUT parameter
   */
   ? := p_dest_clob; 
   
  /*dbms_output.put_line(p_dest_clob); uncomment for sqlplus */

end;
