/*
 * Decompiled with CFR 0.152.
 */
package com.okta.sdk.impl.cache;

import com.okta.sdk.cache.Cache;
import com.okta.sdk.impl.util.SoftHashMap;
import com.okta.sdk.lang.Assert;
import com.okta.sdk.lang.Duration;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCache<K, V>
implements Cache<K, V> {
    private final Logger logger = LoggerFactory.getLogger(DefaultCache.class);
    private final Map<K, Entry<V>> map;
    private volatile Duration timeToLive;
    private volatile Duration timeToIdle;
    private final String name;
    private final AtomicLong accessCount;
    private final AtomicLong hitCount;
    private final AtomicLong missCount;

    public DefaultCache(String name) {
        this(name, new SoftHashMap());
    }

    public DefaultCache(String name, Map<K, Entry<V>> backingMap) {
        this(name, backingMap, null, null);
    }

    public DefaultCache(String name, Map<K, Entry<V>> backingMap, Duration timeToLive, Duration timeToIdle) {
        Assert.notNull((Object)name, (String)"Cache name cannot be null.");
        Assert.notNull(backingMap, (String)"Backing map cannot be null.");
        DefaultCache.assertTtl(timeToLive);
        DefaultCache.assertTti(timeToIdle);
        this.name = name;
        this.map = backingMap;
        this.timeToLive = timeToLive;
        this.timeToIdle = timeToIdle;
        this.accessCount = new AtomicLong(0L);
        this.hitCount = new AtomicLong(0L);
        this.missCount = new AtomicLong(0L);
    }

    protected static void assertTtl(Duration ttl) {
        if (ttl != null) {
            Assert.isTrue((ttl.getValue() > 0L ? 1 : 0) != 0, (String)"timeToLive duration must be greater than zero");
        }
    }

    protected static void assertTti(Duration tti) {
        if (tti != null) {
            Assert.isTrue((tti.getValue() > 0L ? 1 : 0) != 0, (String)"timeToIdle duration must be greater than zero");
        }
    }

    public V get(K key) {
        Duration sinceLastAccess;
        Duration sinceCreation;
        this.accessCount.incrementAndGet();
        Entry<V> entry = this.map.get(key);
        if (entry == null) {
            this.missCount.incrementAndGet();
            return null;
        }
        long nowMillis = System.currentTimeMillis();
        Duration ttl = this.timeToLive;
        Duration tti = this.timeToIdle;
        if (ttl != null && (sinceCreation = new Duration(nowMillis - entry.getCreationTimeMillis(), TimeUnit.MILLISECONDS)).isGreaterThan(ttl)) {
            this.map.remove(key);
            this.missCount.incrementAndGet();
            this.logger.trace("Removing {} from cache due to TTL, sinceCreation: {}", key, (Object)sinceCreation);
            return null;
        }
        if (tti != null && (sinceLastAccess = new Duration(nowMillis - entry.getLastAccessTimeMillis(), TimeUnit.MILLISECONDS)).isGreaterThan(tti)) {
            this.map.remove(key);
            this.missCount.incrementAndGet();
            this.logger.trace("Removing {} from cache due to TTI, sinceLastAccess: {}", key, (Object)sinceLastAccess);
            return null;
        }
        ((Entry)entry).lastAccessTimeMillis = nowMillis;
        this.hitCount.incrementAndGet();
        return entry.getValue();
    }

    public V put(K key, V value) {
        Entry<V> newEntry = new Entry<V>(value);
        Entry<V> previous = this.map.put(key, newEntry);
        if (previous != null) {
            return (V)((Entry)previous).value;
        }
        return null;
    }

    public V remove(K key) {
        this.accessCount.incrementAndGet();
        Entry<V> previous = this.map.remove(key);
        if (previous != null) {
            this.hitCount.incrementAndGet();
            return (V)((Entry)previous).value;
        }
        this.missCount.incrementAndGet();
        return null;
    }

    public Duration getTimeToLive() {
        return this.timeToLive;
    }

    public void setTimeToLive(Duration timeToLive) {
        DefaultCache.assertTtl(timeToLive);
        this.timeToLive = timeToLive;
    }

    public Duration getTimeToIdle() {
        return this.timeToIdle;
    }

    public void setTimeToIdle(Duration timeToIdle) {
        DefaultCache.assertTti(timeToIdle);
        this.timeToIdle = timeToIdle;
    }

    public long getAccessCount() {
        return this.accessCount.get();
    }

    public long getHitCount() {
        return this.hitCount.get();
    }

    public long getMissCount() {
        return this.missCount.get();
    }

    public double getHitRatio() {
        double accessCount = this.getAccessCount();
        if (accessCount > 0.0) {
            double hitCount = this.getHitCount();
            return hitCount / accessCount;
        }
        return 0.0;
    }

    public void clear() {
        this.map.clear();
    }

    public int size() {
        return this.map.size();
    }

    public String getName() {
        return this.name;
    }

    public String toString() {
        return "    {\n      \"name\": \"" + this.name + "\",\n" + "      \"size\": " + this.map.size() + ",\n" + "      \"accessCount\": " + this.getAccessCount() + ",\n" + "      \"hitCount\": " + this.getHitCount() + ",\n" + "      \"missCount\": " + this.getMissCount() + ",\n" + "      \"hitRatio\": " + this.getHitRatio() + "\n" + "    }";
    }

    public static class Entry<V> {
        private final V value;
        private final long creationTimeMillis;
        private volatile long lastAccessTimeMillis;

        public Entry(V value) {
            this.value = value;
            this.lastAccessTimeMillis = this.creationTimeMillis = System.currentTimeMillis();
        }

        public V getValue() {
            return this.value;
        }

        public long getCreationTimeMillis() {
            return this.creationTimeMillis;
        }

        public long getLastAccessTimeMillis() {
            return this.lastAccessTimeMillis;
        }
    }
}

