package com.okta.sdk.resource.model;

import java.util.Objects;
import java.util.Arrays;
import java.io.Serializable;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonTypeName;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiModel;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;

/**
 * The Identity Provider object&#39;s &#x60;type&#x60; property identifies the social or enterprise Identity Provider
 * used for authentication. Each Identity Provider uses a specific protocol, therefore the &#x60;protocol&#x60; property
 * must correspond with the IdP &#x60;type&#x60;. If the protocol is OAuth 2.0-based, the Protocol object&#39;s
 * &#x60;scopes&#x60; property must also correspond with the scopes supported by the IdP &#x60;type&#x60;. For policy
 * actions supported by each IdP type, see [IdP type policy
 * actions](https://developer.okta.com/docs/api/openapi/okta-management/management/tag/IdentityProvider/#tag/IdentityProvider/operation/createIdentityProvider!path&#x3D;policy&amp;t&#x3D;request).
 * | Type | Description | Corresponding protocol | Corresponding protocol scopes | | ------------------ |
 * -----------------------------------------------------------------------------------------------------------------------------------------------------
 * | ---------------------- | -------------------------------------------------------------------- | |
 * &#x60;AMAZON&#x60; |
 * [Amazon](https://developer.amazon.com/settings/console/registration?return_to&#x3D;/)&amp;nbsp;as the Identity
 * Provider | OpenID Connect | &#x60;profile&#x60;, &#x60;profile:user_id&#x60; | | &#x60;APPLE&#x60; |
 * [Apple](https://developer.apple.com/sign-in-with-apple/)&amp;nbsp;as the Identity Provider | OpenID Connect |
 * &#x60;names&#x60;, &#x60;email&#x60;, &#x60;openid&#x60; | | &#x60;DISCORD&#x60; |
 * [Discord](https://discord.com/login)&amp;nbsp;as the Identity Provider | OAuth 2.0 | &#x60;identify&#x60;,
 * &#x60;email&#x60; | | &#x60;FACEBOOK&#x60; | [Facebook](https://developers.facebook.com)&amp;nbsp;as the Identity
 * Provider | OAuth 2.0 | &#x60;public_profile&#x60;, &#x60;email&#x60; | | &#x60;GITHUB&#x60; |
 * [GitHub](https://github.com/join)&amp;nbsp;as the Identity Provider | OAuth 2.0 | &#x60;user&#x60; | |
 * &#x60;GITLAB&#x60; | [GitLab](https://gitlab.com/users/sign_in)&amp;nbsp;as the Identity Provider | OpenID Connect |
 * &#x60;openid&#x60;, &#x60;read_user&#x60;, &#x60;profile&#x60;, &#x60;email&#x60; | | &#x60;GOOGLE&#x60; |
 * [Google](https://accounts.google.com/signup)&amp;nbsp;as the Identity Provider | OpenID Connect | &#x60;openid&#x60;,
 * &#x60;email&#x60;, &#x60;profile&#x60; | | &#x60;LINKEDIN&#x60; |
 * [LinkedIn](https://developer.linkedin.com/)&amp;nbsp;as the Identity Provider | OAuth 2.0 |
 * &#x60;r_emailaddress&#x60;, &#x60;r_liteprofile&#x60; | | &#x60;LOGINGOV&#x60; |
 * [Login.gov](https://developers.login.gov/)&amp;nbsp;as the Identity Provider | OpenID Connect | &#x60;email&#x60;,
 * &#x60;profile&#x60;, &#x60;profile:name&#x60; | | &#x60;LOGINGOV_SANDBOX&#x60; | [Login.gov&#39;s identity
 * sandbox](https://developers.login.gov/testing/)&amp;nbsp;as the Identity Provider | OpenID Connect |
 * &#x60;email&#x60;, &#x60;profile&#x60;, &#x60;profile:name&#x60; | | &#x60;MICROSOFT&#x60; | [Microsoft Enterprise
 * SSO](https://azure.microsoft.com/)&amp;nbsp;as the Identity Provider | OpenID Connect | &#x60;openid&#x60;,
 * &#x60;email&#x60;, &#x60;profile&#x60;, &#x60;https://graph.microsoft.com/User.Read&#x60; | | &#x60;OIDC&#x60; | IdP
 * provider that supports [OpenID Connect](https://openid.net/specs/openid-connect-core-1_0.html) | OpenID Connect |
 * &#x60;openid&#x60;, &#x60;email&#x60;, &#x60;profile&#x60; | | &#x60;PAYPAL&#x60; |
 * [Paypal](https://www.paypal.com/signin)&amp;nbsp;as the Identity Provider | OpenID Connect | &#x60;openid&#x60;,
 * &#x60;email&#x60;, &#x60;profile&#x60; | | &#x60;PAYPAL_SANDBOX&#x60; | [Paypal
 * Sandbox](https://developer.paypal.com/tools/sandbox/)&amp;nbsp;as the Identity Provider | OpenID Connect |
 * &#x60;openid&#x60;, &#x60;email&#x60;, &#x60;profile&#x60; | | &#x60;SALESFORCE&#x60; |
 * [SalesForce](https://login.salesforce.com/)&amp;nbsp;as the Identity Provider | OAuth 2.0 | &#x60;id&#x60;,
 * &#x60;email&#x60;, &#x60;profile&#x60; | | &#x60;SAML2&#x60; | Enterprise IdP provider that supports the [SAML 2.0
 * Web Browser SSO Profile](https://docs.oasis-open.org/security/saml/v2.0/saml-profiles-2.0-os.pdf) | SAML 2.0 | | |
 * &#x60;SPOTIFY&#x60; | [Spotify](https://developer.spotify.com/)&amp;nbsp;as the Identity Provider | OpenID Connect |
 * &#x60;user-read-email&#x60;, &#x60;user-read-private&#x60; | | &#x60;X509&#x60; | [Smart Card
 * IdP](https://tools.ietf.org/html/rfc5280) | Mutual TLS | | | &#x60;XERO&#x60; |
 * [Xero](https://www.xero.com/us/signup/api/)&amp;nbsp;as the Identity Provider | OpenID Connect | &#x60;openid&#x60;,
 * &#x60;profile&#x60;, &#x60;email&#x60; | | &#x60;YAHOO&#x60; | [Yahoo](https://login.yahoo.com/)&amp;nbsp;as the
 * Identity Provider | OpenID Connect | &#x60;openid&#x60;, &#x60;profile&#x60;, &#x60;email&#x60; | |
 * &#x60;YAHOOJP&#x60; | [Yahoo Japan](https://login.yahoo.co.jp/config/login)&amp;nbsp;as the Identity Provider |
 * OpenID Connect | &#x60;openid&#x60;, &#x60;profile&#x60;, &#x60;email&#x60; |
 */
public enum IdentityProviderType {

    AMAZON("AMAZON"),

    APPLE("APPLE"),

    DISCORD("DISCORD"),

    FACEBOOK("FACEBOOK"),

    GITHUB("GITHUB"),

    GITLAB("GITLAB"),

    GOOGLE("GOOGLE"),

    LINKEDIN("LINKEDIN"),

    LOGINGOV("LOGINGOV"),

    LOGINGOV_SANDBOX("LOGINGOV_SANDBOX"),

    MICROSOFT("MICROSOFT"),

    OIDC("OIDC"),

    PAYPAL("PAYPAL"),

    PAYPAL_SANDBOX("PAYPAL_SANDBOX"),

    SALESFORCE("SALESFORCE"),

    SAML2("SAML2"),

    SPOTIFY("SPOTIFY"),

    X509("X509"),

    XERO("XERO"),

    YAHOO("YAHOO"),

    YAHOOJP("YAHOOJP"),

    UNKNOWN_DEFAULT_OPEN_API("unknown_default_open_api");

    private String value;

    IdentityProviderType(String value) {
        this.value = value;
    }

    @JsonValue
    public String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return String.valueOf(value);
    }

    @JsonCreator
    public static IdentityProviderType fromValue(String value) {
        for (IdentityProviderType b : IdentityProviderType.values()) {
            if (b.value.equals(value)) {
                return b;
            }
        }
        return UNKNOWN_DEFAULT_OPEN_API;
    }

    /**
     * Convert the instance into URL query string.
     *
     * @param prefix
     *            prefix of the query string
     *
     * @return URL query string
     */
    public String toUrlQueryString(String prefix) {
        if (prefix == null) {
            prefix = "";
        }

        return String.format("%s=%s", prefix, this.toString().toString());
    }
}
