/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.agent.service.module;

import com.newrelic.agent.Agent;
import com.newrelic.agent.bridge.ManifestUtils;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.deps.com.google.common.cache.CacheBuilder;
import com.newrelic.agent.deps.com.google.common.collect.ImmutableList;
import com.newrelic.agent.deps.com.google.common.collect.ImmutableMap;
import com.newrelic.agent.service.ServiceFactory;
import com.newrelic.agent.service.module.EmbeddedJars;
import com.newrelic.agent.service.module.Jar;
import com.newrelic.agent.service.module.JarInfo;
import com.newrelic.agent.service.module.ShaChecksums;
import com.newrelic.weave.weavepackage.WeavePackageConfig;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;
import java.util.logging.Level;

class JarCollectorServiceProcessor {
    static final String SHA1_CHECKSUM_KEY = "sha1Checksum";
    static final String SHA512_CHECKSUM_KEY = "sha512Checksum";
    static final String JAR_EXTENSION = ".jar";
    static final String JAR_PROTOCOL = "jar";
    static final String UNKNOWN_VERSION = " ";
    private static final int MAX_MAP_SIZE = 1000;
    private static final JarInfo NON_JAR = new JarInfo(null, null);
    private static final JarInfo JAR_ERROR = new JarInfo(null, null);
    private static final List<String> ATTRIBUTES_TO_COLLECT = ImmutableList.of(Attributes.Name.IMPLEMENTATION_VENDOR.toString(), Attributes.Name.IMPLEMENTATION_VENDOR_ID.toString());
    private final boolean skipTempJars;
    private final List<String> ignoreJars;
    private final Map<URL, JarInfo> sentJars;
    private final Map<String, URL> sha1ChecksumToURL = CacheBuilder.newBuilder().maximumSize(10000L).build().asMap();
    private static final File TEMP_DIRECTORY = new File(System.getProperty("java.io.tmpdir"));

    public JarCollectorServiceProcessor() {
        this(ServiceFactory.getConfigService().getDefaultAgentConfig().getIgnoreJars());
    }

    JarCollectorServiceProcessor(List<String> ignoreJars) {
        this.ignoreJars = ignoreJars;
        this.sentJars = CacheBuilder.newBuilder().maximumSize(1000L).weakKeys().build().asMap();
        AgentConfig config = ServiceFactory.getConfigService().getDefaultAgentConfig();
        this.skipTempJars = (Boolean)config.getValue("jar_collector.skip_temp_jars", true);
        if (!this.skipTempJars) {
            Agent.LOG.finest("Jar collector: temporary jars will be transmitted to the host");
        }
    }

    protected synchronized List<Jar> processModuleData(Collection<URL> urlsToProcess, boolean sendAll) {
        urlsToProcess = new HashSet<URL>(urlsToProcess);
        ArrayList<Jar> jars = new ArrayList<Jar>();
        if (sendAll) {
            urlsToProcess.addAll(this.sentJars.keySet());
        } else {
            urlsToProcess.removeAll(new HashSet<URL>(this.sentJars.keySet()));
        }
        Map<URL, JarInfo> processedUrls = this.processUrls(urlsToProcess, jars);
        this.sentJars.putAll(processedUrls);
        return jars;
    }

    private Map<URL, JarInfo> processUrls(Collection<URL> urls, List<Jar> jars) {
        HashMap<URL, JarInfo> jarDetails = new HashMap<URL, JarInfo>();
        for (URL address : urls) {
            JarInfo jar = NON_JAR;
            try {
                if (this.skipTempJars && JarCollectorServiceProcessor.isTempFile(address)) {
                    Agent.LOG.log(Level.FINE, "Skipping temp jar file {0}", address.toString());
                } else {
                    Agent.LOG.log(Level.FINEST, "Processing jar file {0}", address.toString());
                    jar = this.processUrl(address, jars);
                }
            }
            catch (Exception e) {
                Agent.LOG.log(Level.FINEST, "While processing {0}: {1}: {2}", address, e.getClass().getSimpleName(), e.getMessage());
            }
            jarDetails.put(address, jar);
        }
        return jarDetails;
    }

    static boolean isTempFile(URL address) throws URISyntaxException {
        if (!"file".equals(address.getProtocol())) {
            return false;
        }
        return JarCollectorServiceProcessor.isTempFile(new File(address.toURI()));
    }

    static boolean isTempFile(File file) {
        if (null == (file = file.getParentFile())) {
            return false;
        }
        if (TEMP_DIRECTORY.equals(file)) {
            return true;
        }
        return JarCollectorServiceProcessor.isTempFile(file);
    }

    private JarInfo processUrl(URL url, List<Jar> jars) {
        try {
            if (!url.getFile().endsWith(JAR_EXTENSION)) {
                return NON_JAR;
            }
            Agent.LOG.log(Level.FINEST, "URL has file path {0}.", url.getFile());
            return this.handleJar(url, jars);
        }
        catch (Exception e) {
            Agent.LOG.log(Level.FINEST, e, "Error processing the file path : {0}", e.getMessage());
            return JAR_ERROR;
        }
    }

    private JarInfo handleJar(URL url, List<Jar> jars) {
        JarInfo jarInfo = JarCollectorServiceProcessor.getJarInfoSafe(url);
        String sha1checksum = jarInfo.attributes.get(SHA1_CHECKSUM_KEY);
        if (null != sha1checksum) {
            this.sha1ChecksumToURL.put(sha1checksum, url);
        }
        this.addJarAndVersion(url, jarInfo, jars);
        return jarInfo;
    }

    Map<String, URL> getSha1ChecksumToURL() {
        return this.sha1ChecksumToURL;
    }

    static JarInfo getJarInfoSafe(URL url) {
        JarInfo jarInfo;
        HashMap<String, String> attributes = new HashMap<String, String>();
        try {
            String sha1Checksum = ShaChecksums.computeSha(url);
            attributes.put(SHA1_CHECKSUM_KEY, sha1Checksum);
        }
        catch (Exception ex) {
            Agent.LOG.log(Level.FINE, "Error getting jar file sha1 checksum : {0}", ex.getMessage());
            Agent.LOG.log(Level.FINEST, ex, "{0}", ex.getMessage());
            attributes.put("sha1error", JarCollectorServiceProcessor.getErrorMessage(ex));
        }
        try {
            String sha512Checksum = ShaChecksums.computeSha512(url);
            attributes.put(SHA512_CHECKSUM_KEY, sha512Checksum);
        }
        catch (Exception ex) {
            Agent.LOG.log(Level.FINE, "Error getting jar file sha512 checksum : {0}", ex.getMessage());
            Agent.LOG.log(Level.FINEST, ex, "{0}", ex.getMessage());
            attributes.put("sha512error", JarCollectorServiceProcessor.getErrorMessage(ex));
        }
        try {
            jarInfo = JarCollectorServiceProcessor.getJarInfo(url, attributes);
        }
        catch (Exception e) {
            Agent.LOG.log(Level.FINEST, e, "Trouble getting version from {0} jar. Adding jar without version.", url.getFile());
            jarInfo = new JarInfo(UNKNOWN_VERSION, attributes);
        }
        return jarInfo;
    }

    private static String getErrorMessage(Exception ex) {
        String message = ex.getMessage();
        if (null == message) {
            message = ex.toString();
        }
        return ex.getClass().getName() + " : " + message;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static JarInfo getJarInfo(URL url, Map<String, String> attributes) throws IOException {
        JarInputStream jarFile;
        block13: {
            JarInfo jarInfo;
            block14: {
                jarFile = EmbeddedJars.getJarInputStream(url);
                Throwable throwable = null;
                try {
                    JarCollectorServiceProcessor.getExtraAttributes(jarFile, attributes);
                    Map<String, String> pom = JarCollectorServiceProcessor.getPom(jarFile);
                    if (pom == null) break block13;
                    attributes.putAll(pom);
                    jarInfo = new JarInfo(pom.get("version"), attributes);
                    if (jarFile == null) return jarInfo;
                    if (throwable == null) break block14;
                }
                catch (Exception ex) {
                    Agent.LOG.log(Level.FINEST, ex, "{0}", ex.getMessage());
                    break block13;
                }
                catch (Throwable throwable3) {
                    throwable = throwable3;
                    throw throwable3;
                }
                catch (Throwable throwable4) {
                    throw throwable4;
                }
                try {
                    jarFile.close();
                    return jarInfo;
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                    return jarInfo;
                }
            }
            jarFile.close();
            return jarInfo;
        }
        String version = JarCollectorServiceProcessor.getVersion(jarFile);
        if (version != null) return new JarInfo(version, attributes);
        version = UNKNOWN_VERSION;
        return new JarInfo(version, attributes);
    }

    private static Map<String, String> getPom(JarInputStream jarFile) throws IOException {
        Properties pom = null;
        JarEntry entry = null;
        while ((entry = jarFile.getNextJarEntry()) != null) {
            if (!entry.getName().startsWith("META-INF/maven") || !entry.getName().endsWith("pom.properties")) continue;
            if (pom != null) {
                return null;
            }
            Properties props = new Properties();
            props.load(jarFile);
            pom = props;
        }
        return pom;
    }

    static void getExtraAttributes(JarInputStream jarFile, Map<String, String> map) {
        Manifest manifest = jarFile.getManifest();
        if (manifest == null) {
            return;
        }
        Attributes attributes = manifest.getMainAttributes();
        for (String name : ATTRIBUTES_TO_COLLECT) {
            String value = attributes.getValue(name);
            if (null == value) continue;
            map.put(name, value);
        }
    }

    static String getVersion(JarInputStream jarFile) {
        Manifest manifest = jarFile.getManifest();
        if (manifest == null) {
            return null;
        }
        return ManifestUtils.getVersionFromManifest((Manifest)manifest);
    }

    boolean addJarAndVersion(URL url, JarInfo jarInfo, List<Jar> jars) {
        if (jarInfo == null) {
            jarInfo = JarInfo.MISSING;
        }
        boolean added = false;
        String jarFile = null;
        try {
            jarFile = JarCollectorServiceProcessor.parseJarName(url);
            if (this.shouldAttemptAdd(jarFile)) {
                jars.add(new Jar(jarFile, jarInfo));
                added = true;
            }
        }
        catch (URISyntaxException e) {
            Agent.LOG.log(Level.FINEST, e, "{0}", e.getMessage());
        }
        if (added) {
            Agent.LOG.log(Level.FINER, "Adding the jar {0} with version {1}.", jarFile, jarInfo.version);
        } else {
            Agent.LOG.log(Level.FINER, "Not taking version {0} for jar {1}.", jarInfo.version, jarFile);
        }
        return added;
    }

    static String parseJarName(URL url) throws URISyntaxException {
        if ("file".equals(url.getProtocol())) {
            File file = new File(url.toURI());
            return file.getName().trim();
        }
        Agent.LOG.log(Level.FINEST, "Parsing jar file name from {0}", url);
        String path = url.getFile();
        int end = path.lastIndexOf(JAR_EXTENSION);
        if (end > 0) {
            int start = (path = path.substring(0, end)).lastIndexOf(File.separator);
            if (start > -1) {
                return path.substring(start + 1) + JAR_EXTENSION;
            }
            return path + JAR_EXTENSION;
        }
        throw new URISyntaxException(url.getPath(), "Unable to parse the jar file name from a URL");
    }

    private boolean shouldAttemptAdd(String jarFile) {
        return !this.ignoreJars.contains(jarFile);
    }

    static Collection<Jar> getWeaveJars(Map<File, WeavePackageConfig> weaveConfigurations) {
        ArrayList<Jar> jars = new ArrayList<Jar>();
        for (Map.Entry<File, WeavePackageConfig> entry : weaveConfigurations.entrySet()) {
            String sha1Checksum = "UNKNOWN";
            try {
                sha1Checksum = ShaChecksums.computeSha(entry.getKey());
            }
            catch (Exception ex) {
                Agent.LOG.log(Level.FINE, "Error getting weave file checksum : {0}", ex.getMessage());
                Agent.LOG.log(Level.FINEST, ex, "{0}", ex.getMessage());
            }
            ImmutableMap<String, String> attributes = ImmutableMap.of("weaveFile", entry.getValue().getSource(), SHA1_CHECKSUM_KEY, sha1Checksum);
            JarInfo info = new JarInfo(Float.toString(entry.getValue().getVersion()), attributes);
            Jar jar = new Jar(entry.getValue().getName(), info);
            jars.add(jar);
        }
        return jars;
    }
}

