/*
 * Decompiled with CFR 0.152.
 */
package com.networknt.url;

import com.networknt.url.HttpURL;
import com.networknt.utility.StringUtils;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLDecoder;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class URLNormalizer
implements Serializable {
    private static final long serialVersionUID = 7236478212865008971L;
    private static final Logger logger = LoggerFactory.getLogger(URLNormalizer.class);
    private static final Pattern PATTERN_PERCENT_ENCODED_CHAR = Pattern.compile("(%[0-9a-f]{2})", 2);
    private static final Pattern PATTERN_PATH_LAST_SEGMENT = Pattern.compile("(.*/)(index\\.html|index\\.htm|index\\.shtml|index\\.php|default\\.html|default\\.htm|home\\.html|home\\.htm|index\\.php5|index\\.php4|index\\.php3|index\\.cgi|placeholder\\.html|default\\.asp)$", 2);
    private static final Pattern PATTERN_DOMAIN = Pattern.compile("^[a-z0-9]+([\\-\\.]{1}[a-z0-9]+)*\\.[a-z]{2,5}$", 2);
    private static final Pattern PATTERN_SCHEMA = Pattern.compile("(.*?)(://.*)$", 2);
    private String url;

    public URLNormalizer(URL url) {
        this(Objects.toString(url, null));
    }

    public URLNormalizer(String url) {
        if (StringUtils.isBlank(url)) {
            throw new IllegalArgumentException("URL argument cannot be null.");
        }
        this.url = url.trim();
        try {
            new URL(this.url);
        }
        catch (MalformedURLException e) {
            throw new RuntimeException("Invalid URL: " + url, e);
        }
    }

    public URLNormalizer lowerCaseSchemeHost() {
        URL u = this.toURL();
        this.url = Pattern.compile(u.getProtocol(), 2).matcher(this.url).replaceFirst(u.getProtocol().toLowerCase());
        this.url = Pattern.compile(u.getHost(), 2).matcher(this.url).replaceFirst(u.getHost().toLowerCase());
        return this;
    }

    public URLNormalizer upperCaseEscapeSequence() {
        if (this.url.contains("%")) {
            StringBuffer sb = new StringBuffer();
            Matcher m4 = PATTERN_PERCENT_ENCODED_CHAR.matcher(this.url);
            while (m4.find()) {
                m4.appendReplacement(sb, m4.group(1).toUpperCase());
            }
            this.url = m4.appendTail(sb).toString();
        }
        return this;
    }

    public URLNormalizer decodeUnreservedCharacters() {
        if (this.url.contains("%")) {
            StringBuffer sb = new StringBuffer();
            Matcher m4 = PATTERN_PERCENT_ENCODED_CHAR.matcher(this.url);
            try {
                while (m4.find()) {
                    String enc = m4.group(1).toUpperCase();
                    if (!this.isEncodedUnreservedCharacter(enc)) continue;
                    m4.appendReplacement(sb, URLDecoder.decode(enc, StandardCharsets.UTF_8.toString()));
                }
            }
            catch (UnsupportedEncodingException e) {
                logger.debug("UTF-8 is not supported by your system. URL will remain unchanged:" + this.url, e);
            }
            this.url = m4.appendTail(sb).toString();
        }
        return this;
    }

    public URLNormalizer encodeNonURICharacters() {
        this.url = this.toURI().toASCIIString();
        return this;
    }

    public URLNormalizer encodeSpaces() {
        String path = StringUtils.substringBefore(this.url, "?");
        path = StringUtils.replace(path, " ", "%20", -1);
        String qs = StringUtils.substringAfter(this.url, "?");
        if (StringUtils.isNotBlank(qs)) {
            qs = StringUtils.replace(qs, " ", "+", -1);
            this.url = path + "?" + qs;
        } else {
            this.url = path;
        }
        return this;
    }

    public URLNormalizer removeDefaultPort() {
        URL u = this.toURL();
        if ("http".equalsIgnoreCase(u.getProtocol()) && u.getPort() == 80) {
            this.url = this.url.replaceFirst(":80", "");
        } else if ("https".equalsIgnoreCase(u.getProtocol()) && u.getPort() == 443) {
            this.url = this.url.replaceFirst(":443", "");
        }
        return this;
    }

    public URLNormalizer addDirectoryTrailingSlash() {
        String urlRoot = HttpURL.getRoot(this.url);
        String path = this.toURL().getPath();
        String urlRootAndPath = urlRoot + path;
        String name = StringUtils.substringAfterLast(path, "/");
        if (StringUtils.isNotBlank(name) && !name.contains(".")) {
            String newPath = path + "/";
            String newUrlRootAndPath = urlRoot + newPath;
            this.url = StringUtils.replaceOnce(this.url, urlRootAndPath, newUrlRootAndPath);
        }
        return this;
    }

    public URLNormalizer addDomainTrailingSlash() {
        String urlRoot = HttpURL.getRoot(this.url);
        String path = this.toURL().getPath();
        if (StringUtils.isNotBlank(path)) {
            return this;
        }
        String urlRootAndPath = urlRoot + "/";
        this.url = StringUtils.replaceOnce(this.url, urlRoot, urlRootAndPath);
        return this;
    }

    @Deprecated
    public URLNormalizer addTrailingSlash() {
        return this.addDirectoryTrailingSlash();
    }

    public URLNormalizer removeTrailingSlash() {
        String urlRoot = HttpURL.getRoot(this.url);
        String path = this.toURL().getPath();
        String urlRootAndPath = urlRoot + path;
        if (path.endsWith("/")) {
            String newPath = StringUtils.removeEnd(path, "/");
            String newUrlRootAndPath = urlRoot + newPath;
            this.url = StringUtils.replaceOnce(this.url, urlRootAndPath, newUrlRootAndPath);
        }
        return this;
    }

    public URLNormalizer removeDotSegments() {
        String path = this.toURL().getPath().trim();
        StringBuilder in = new StringBuilder(path);
        StringBuilder out = new StringBuilder();
        while (in.length() > 0) {
            if (URLNormalizer.startsWith(in, "../")) {
                this.deleteStart(in, "../");
                continue;
            }
            if (URLNormalizer.startsWith(in, "./")) {
                this.deleteStart(in, "./");
                continue;
            }
            if (URLNormalizer.startsWith(in, "/./")) {
                this.replaceStart(in, "/./", "/");
                continue;
            }
            if (URLNormalizer.equalStrings(in, "/.")) {
                this.replaceStart(in, "/.", "/");
                continue;
            }
            if (URLNormalizer.startsWith(in, "/../")) {
                this.replaceStart(in, "/../", "/");
                this.removeLastSegment(out);
                continue;
            }
            if (URLNormalizer.equalStrings(in, "/..")) {
                this.replaceStart(in, "/..", "/");
                this.removeLastSegment(out);
                continue;
            }
            if (URLNormalizer.equalStrings(in, "..")) {
                this.deleteStart(in, "..");
                continue;
            }
            if (URLNormalizer.equalStrings(in, ".")) {
                this.deleteStart(in, ".");
                continue;
            }
            int nextSlashIndex = in.indexOf("/", 1);
            if (nextSlashIndex > -1) {
                out.append(in.substring(0, nextSlashIndex));
                in.delete(0, nextSlashIndex);
                continue;
            }
            out.append((CharSequence)in);
            in.setLength(0);
        }
        this.url = StringUtils.replaceOnce(this.url, path, out.toString());
        return this;
    }

    private static boolean equalStrings(StringBuilder b, String str) {
        return b.length() == str.length() && b.indexOf(str) == 0;
    }

    private static boolean startsWith(StringBuilder b, String str) {
        return b.indexOf(str) == 0;
    }

    private void replaceStart(StringBuilder b, String toreplace, String replacement) {
        this.deleteStart(b, toreplace);
        b.insert(0, replacement);
    }

    private void deleteStart(StringBuilder b, String str) {
        b.delete(0, str.length());
    }

    private void removeLastSegment(StringBuilder b) {
        int index = b.lastIndexOf("/");
        if (index == -1) {
            b.setLength(0);
        } else {
            b.setLength(index);
        }
    }

    public URLNormalizer removeDirectoryIndex() {
        String path = this.toURL().getPath();
        if (PATTERN_PATH_LAST_SEGMENT.matcher(path).matches()) {
            this.url = StringUtils.replaceOnce(this.url, path, StringUtils.substringBeforeLast(path, "/") + "/");
        }
        return this;
    }

    public URLNormalizer removeFragment() {
        this.url = this.url.replaceFirst("(.*?)(#.*)", "$1");
        return this;
    }

    public URLNormalizer replaceIPWithDomainName() {
        URL u = this.toURL();
        if (!PATTERN_DOMAIN.matcher(u.getHost()).matches()) {
            try {
                InetAddress addr = InetAddress.getByName(u.getHost());
                String host = addr.getHostName();
                if (!u.getHost().equalsIgnoreCase(host)) {
                    this.url = this.url.replaceFirst(u.getHost(), host);
                }
            }
            catch (UnknownHostException e) {
                logger.debug("Cannot resolve IP to host for :" + u.getHost(), e);
            }
        }
        return this;
    }

    public URLNormalizer unsecureScheme() {
        String schema;
        Matcher m4 = PATTERN_SCHEMA.matcher(this.url);
        if (m4.find() && "https".equalsIgnoreCase(schema = m4.group(1))) {
            this.url = m4.replaceFirst(StringUtils.stripEnd(schema, "Ss") + "$2");
        }
        return this;
    }

    public URLNormalizer secureScheme() {
        String schema;
        Matcher m4 = PATTERN_SCHEMA.matcher(this.url);
        if (m4.find() && "http".equalsIgnoreCase(schema = m4.group(1))) {
            this.url = m4.replaceFirst(schema + "s$2");
        }
        return this;
    }

    public URLNormalizer removeDuplicateSlashes() {
        String urlRoot = HttpURL.getRoot(this.url);
        String path = this.toURL().getPath();
        String urlRootAndPath = urlRoot + path;
        String newPath = path.replaceAll("/{2,}", "/");
        String newUrlRootAndPath = urlRoot + newPath;
        this.url = StringUtils.replaceOnce(this.url, urlRootAndPath, newUrlRootAndPath);
        return this;
    }

    public URLNormalizer removeWWW() {
        String host = this.toURL().getHost();
        String newHost = StringUtils.removeStartIgnoreCase(host, "www.");
        this.url = StringUtils.replaceOnce(this.url, host, newHost);
        return this;
    }

    public URLNormalizer addWWW() {
        String host = this.toURL().getHost();
        if (!host.toLowerCase().startsWith("www.")) {
            this.url = StringUtils.replaceOnce(this.url, host, "www." + host);
        }
        return this;
    }

    public URLNormalizer sortQueryParameters() {
        String[] params;
        if (!this.url.contains("?")) {
            return this;
        }
        ArrayList<String> keyValues = new ArrayList<String>();
        String queryString = StringUtils.substringAfter(this.url, "?");
        Object fragment = StringUtils.substringAfter(queryString, "#");
        if (StringUtils.isNotBlank((CharSequence)fragment)) {
            fragment = "#" + (String)fragment;
        }
        queryString = StringUtils.substringBefore(queryString, "#");
        for (String param : params = StringUtils.split(queryString, '&')) {
            keyValues.add(param);
        }
        Collections.sort(keyValues);
        String sortedQueryString = StringUtils.join(keyValues, '&');
        if (StringUtils.isNotBlank(sortedQueryString)) {
            this.url = StringUtils.substringBefore(this.url, "?") + "?" + sortedQueryString + (String)fragment;
        }
        return this;
    }

    public URLNormalizer removeEmptyParameters() {
        String[] params;
        if (!this.url.contains("?")) {
            return this;
        }
        ArrayList<String> keyValues = new ArrayList<String>();
        String queryString = StringUtils.substringAfter(this.url, "?");
        for (String param : params = StringUtils.split(queryString, '&')) {
            if (!param.contains("=") || !StringUtils.isNotBlank(StringUtils.substringAfter(param, "=")) || !StringUtils.isNotBlank(StringUtils.substringBefore(param, "="))) continue;
            keyValues.add(param);
        }
        String cleanQueryString = StringUtils.join(keyValues, '&');
        if (StringUtils.isNotBlank(cleanQueryString)) {
            this.url = StringUtils.substringBefore(this.url, "?") + "?" + cleanQueryString;
        }
        return this;
    }

    public URLNormalizer removeTrailingQuestionMark() {
        if (this.url.endsWith("?") && StringUtils.countMatches((CharSequence)this.url, "?") == 1) {
            this.url = StringUtils.removeEnd(this.url, "?");
        }
        return this;
    }

    public URLNormalizer removeSessionIds() {
        if (StringUtils.containsIgnoreCase(this.url, ";jsessionid=")) {
            this.url = this.url.replaceFirst("(;jsessionid=([A-F0-9]+)((\\.\\w+)*))", "");
        } else {
            Object u = StringUtils.substringBefore(this.url, "?");
            String q = StringUtils.substringAfter(this.url, "?");
            if (StringUtils.containsIgnoreCase(this.url, "PHPSESSID=")) {
                q = q.replaceFirst("(&|^)(PHPSESSID=[0-9a-zA-Z]*)", "");
            } else if (StringUtils.containsIgnoreCase(this.url, "ASPSESSIONID")) {
                q = q.replaceFirst("(&|^)(ASPSESSIONID[a-zA-Z]{8}=[a-zA-Z]*)", "");
            }
            if (!StringUtils.isBlank(q)) {
                u = (String)u + "?" + StringUtils.removeStart(q, "&");
            }
            this.url = u;
        }
        return this;
    }

    public URLNormalizer removeTrailingHash() {
        if (this.url.endsWith("#") && StringUtils.countMatches((CharSequence)this.url, "#") == 1) {
            this.url = StringUtils.removeEnd(this.url, "#");
        }
        return this;
    }

    public String toString() {
        return this.url;
    }

    public URI toURI() {
        if (StringUtils.isBlank(this.url)) {
            return null;
        }
        return HttpURL.toURI(this.url);
    }

    public URL toURL() {
        if (StringUtils.isBlank(this.url)) {
            return null;
        }
        try {
            return new URL(this.url);
        }
        catch (MalformedURLException e) {
            logger.info("URL does not appear to be valid and cannot be parsed:" + this.url, e);
            return null;
        }
    }

    private boolean isEncodedUnreservedCharacter(String enc) {
        if (enc.compareTo("%41") >= 0 && enc.compareTo("%5A") <= 0 || enc.compareTo("%61") >= 0 && enc.compareTo("%7A") <= 0) {
            return true;
        }
        if (enc.compareTo("%30") >= 0 && enc.compareTo("%39") <= 0) {
            return true;
        }
        return URLNormalizer.equalsAny(enc, "%2D", "%2E", "%5F", "%7E");
    }

    private static boolean equalsAny(Object source2, Object ... targets) {
        if (targets == null) {
            return source2 == null;
        }
        for (Object object : targets) {
            if (!Objects.equals(source2, object)) continue;
            return true;
        }
        return false;
    }
}

