/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.concurrency.limits.limiter;

import com.netflix.concurrency.limits.Limiter;
import com.netflix.concurrency.limits.MetricRegistry;
import com.netflix.concurrency.limits.internal.Preconditions;
import com.netflix.concurrency.limits.limiter.AbstractLimiter;
import com.netflix.concurrency.limits.limiter.SimpleLimiter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractPartitionedLimiter<ContextT>
extends AbstractLimiter<ContextT> {
    private static final Logger LOG = LoggerFactory.getLogger(AbstractPartitionedLimiter.class);
    private static final String PARTITION_TAG_NAME = "partition";
    private final Map<String, Partition> partitions;
    private final Partition unknownPartition;
    private final List<Function<ContextT, String>> partitionResolvers;
    private final ReentrantLock lock = new ReentrantLock();
    private final AtomicInteger delayedThreads = new AtomicInteger();
    private final int maxDelayedThreads;

    public AbstractPartitionedLimiter(Builder<?, ContextT> builder) {
        super(builder);
        Preconditions.checkArgument(!((Builder)builder).partitions.isEmpty(), "No partitions specified");
        Preconditions.checkArgument(((Builder)builder).partitions.values().stream().map(Partition::getPercent).reduce(0.0, Double::sum) <= 1.0, "Sum of percentages must be <= 1.0");
        this.partitions = new HashMap<String, Partition>(((Builder)builder).partitions);
        this.partitions.forEach((name, partition) -> partition.createMetrics(builder.registry));
        this.unknownPartition = new Partition("unknown");
        this.unknownPartition.createMetrics(builder.registry);
        this.partitionResolvers = ((Builder)builder).partitionResolvers;
        this.maxDelayedThreads = ((Builder)builder).maxDelayedThreads;
        this.onNewLimit(this.getLimit());
    }

    private Partition resolvePartition(ContextT context) {
        for (Function<ContextT, String> resolver : this.partitionResolvers) {
            Partition partition;
            String name = resolver.apply(context);
            if (name == null || (partition = this.partitions.get(name)) == null) continue;
            return partition;
        }
        return this.unknownPartition;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Optional<Limiter.Listener> acquire(ContextT context) {
        final Partition partition = this.resolvePartition(context);
        try {
            this.lock.lock();
            if (this.shouldBypass(context)) {
                Optional<Limiter.Listener> optional = this.createBypassListener();
                return optional;
            }
            if (this.getInflight() >= this.getLimit() && partition.isLimitExceeded()) {
                this.lock.unlock();
                if (partition.backoffMillis > 0L && this.delayedThreads.get() < this.maxDelayedThreads) {
                    try {
                        this.delayedThreads.incrementAndGet();
                        TimeUnit.MILLISECONDS.sleep(partition.backoffMillis);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                    finally {
                        this.delayedThreads.decrementAndGet();
                    }
                }
                Optional<Limiter.Listener> e = this.createRejectedListener();
                return e;
            }
            partition.acquire();
            final Limiter.Listener listener = this.createListener();
            Optional<Limiter.Listener> optional = Optional.of(new Limiter.Listener(){

                @Override
                public void onSuccess() {
                    listener.onSuccess();
                    AbstractPartitionedLimiter.this.releasePartition(partition);
                }

                @Override
                public void onIgnore() {
                    listener.onIgnore();
                    AbstractPartitionedLimiter.this.releasePartition(partition);
                }

                @Override
                public void onDropped() {
                    listener.onDropped();
                    AbstractPartitionedLimiter.this.releasePartition(partition);
                }
            });
            return optional;
        }
        finally {
            if (this.lock.isHeldByCurrentThread()) {
                this.lock.unlock();
            }
        }
    }

    private void releasePartition(Partition partition) {
        try {
            this.lock.lock();
            partition.release();
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    protected void onNewLimit(int newLimit) {
        super.onNewLimit(newLimit);
        this.partitions.forEach((name, partition) -> partition.updateLimit(newLimit));
    }

    Partition getPartition(String name) {
        return this.partitions.get(name);
    }

    static class Partition {
        private final String name;
        private double percent = 0.0;
        private int limit = 0;
        private int busy = 0;
        private long backoffMillis = 0L;
        private MetricRegistry.SampleListener inflightDistribution;

        Partition(String name) {
            this.name = name;
        }

        Partition setPercent(double percent) {
            this.percent = percent;
            return this;
        }

        Partition setBackoffMillis(long backoffMillis) {
            this.backoffMillis = backoffMillis;
            return this;
        }

        void updateLimit(int totalLimit) {
            this.limit = (int)Math.max(1.0, Math.ceil((double)totalLimit * this.percent));
        }

        boolean isLimitExceeded() {
            return this.busy >= this.limit;
        }

        void acquire() {
            ++this.busy;
            this.inflightDistribution.addSample(this.busy);
        }

        void release() {
            --this.busy;
        }

        int getLimit() {
            return this.limit;
        }

        public int getInflight() {
            return this.busy;
        }

        double getPercent() {
            return this.percent;
        }

        void createMetrics(MetricRegistry registry) {
            this.inflightDistribution = registry.distribution("inflight", AbstractPartitionedLimiter.PARTITION_TAG_NAME, this.name);
            registry.gauge("limit.partition", this::getLimit, AbstractPartitionedLimiter.PARTITION_TAG_NAME, this.name);
        }

        public String toString() {
            return "Partition [pct=" + this.percent + ", limit=" + this.limit + ", busy=" + this.busy + "]";
        }
    }

    public static abstract class Builder<BuilderT extends AbstractLimiter.BypassLimiterBuilder<BuilderT, ContextT>, ContextT>
    extends AbstractLimiter.BypassLimiterBuilder<BuilderT, ContextT> {
        private List<Function<ContextT, String>> partitionResolvers = new ArrayList<Function<ContextT, String>>();
        private final Map<String, Partition> partitions = new LinkedHashMap<String, Partition>();
        private int maxDelayedThreads = 100;

        public BuilderT partitionResolver(Function<ContextT, String> contextToPartition) {
            this.partitionResolvers.add(contextToPartition);
            return (BuilderT)((AbstractLimiter.BypassLimiterBuilder)this.self());
        }

        public BuilderT partition(String name, double percent) {
            Preconditions.checkArgument(name != null, "Partition name may not be null");
            Preconditions.checkArgument(percent >= 0.0 && percent <= 1.0, "Partition percentage must be in the range [0.0, 1.0]");
            this.partitions.computeIfAbsent(name, Partition::new).setPercent(percent);
            return (BuilderT)((AbstractLimiter.BypassLimiterBuilder)this.self());
        }

        public BuilderT partitionRejectDelay(String name, long duration, TimeUnit units) {
            this.partitions.computeIfAbsent(name, Partition::new).setBackoffMillis(units.toMillis(duration));
            return (BuilderT)((AbstractLimiter.BypassLimiterBuilder)this.self());
        }

        public BuilderT maxDelayedThreads(int maxDelayedThreads) {
            this.maxDelayedThreads = maxDelayedThreads;
            return (BuilderT)((AbstractLimiter.BypassLimiterBuilder)this.self());
        }

        protected boolean hasPartitions() {
            return !this.partitions.isEmpty();
        }

        public Limiter<ContextT> build() {
            return this.hasPartitions() && !this.partitionResolvers.isEmpty() ? new AbstractPartitionedLimiter<ContextT>(this){} : new SimpleLimiter(this);
        }
    }
}

