/*
 * @(#) SpeechRecognizer.java 2015. 1.
 *
 * Copyright 2015 Naver Corp. All rights Reserved.
 * Naver PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.naver.speech.clientapi;

import java.util.ArrayList;

import com.naver.speech.clientapi.SpeechConfig.EndPointDetectType;

import android.content.Context;

/**
 * 이 클래스는 네이버 음성인식 서비스를 사용할 수 있도록 클라이언트 역할을 제공한다.
 * 네이버 음성인식은 음성인식 서버와 클라이언트 간의 통신으로 구성되어 있고,
 * 클라이언트는 음성 입력을 받아 서버로 전송하고, 서버로부터 인식 결과를 받는 역할을 동시에 수행한다.
 * 따라서 반드시 현재 어플리케이션이 android.permission.RECORD_AUDIO 와 android.permission.INTERNET 사용 권한을 가지고 있는지 확인해야 한다.
 * <p>
 * 이 클래스는 실행 전 반드시 initialize() 메소드를 호출하여야 하고, 종료 후 반드시 release() 메소드를 호출하여야 한다.
 * 이는 음성인식으로 인한 자원을 초기화하거나, 해제함으로써 다른 프로세스에 악영향을 주는 것을 방지하기 위함이다.
 * 시나리오에 따라서 달라질 수 있지만, 적어도 {@link android.app.Activity#onResume()} 콜백이 호출될 때에는 initialize()를,
 * {@link android.app.Activity#onPause()} 콜백이 호출될 때에는 release()를 호출하는 것을 권장한다.
 * <p>
 * 이 클래스는 recognize() 메소드를 호출함으로써 음성인식을 시작한다. 음성인식 수행 과정은 아래와 같은 스테이트머신에 기반한다. (다이어그램 참고)
 * <p>
 * <img src="../../../../../../../../docs/images/state_diagram.png" width="480px" height="475px" alt="ast state diagram">
 * <p>
 * 각 스테이트로 천이할 때 마다 콜백 메소드가 호출되며, {@link SpeechRecognitionListener} 를 상속받아서 콜백 메소드들을 오버라이딩 해야 한다.
 * 이를 통해서 음성인식 시작, 중간 결과, 최종 결과 등 다양한 동작을 어플리케이션에 이용할 수 있다.
 * 
 * @see SpeechRecognitionListener
 * @see SpeechRecognizer#initialize()
 * @see SpeechRecognizer#release()
 */
public class SpeechRecognizer {

	protected SpeechRecognitionListener speechRecognitionListener;

	private AudioCapture mAudioCapture;

	private static final String CLIENT_LIB_NAME = "NaverspeechJNI";
	private static final String CLIENT_LIB_VER = "1.0.6";
	
	//////////////////////////////////////////////////////////////
	//
	// Error code
	/**
	 * 네트워크 초기화 오류, 에러값은 10
	 */
	public static final int ERROR_NETWORK_INITIALIZE   = 10;
	/**
	 * 네트워크 메모리 해체 오류, 에러값은 11
	 */
	public static final int ERROR_NETWORK_FINALIZE     = 11;
	/**
	 * 네트워크 읽기 오류, 에러값은 12
	 */
	public static final int ERROR_NETWORK_READ         = 12;
	/**
	 * 네트워크 쓰기 오류, 에러값은 13
	 */
	public static final int ERROR_NETWORK_WRITE        = 13;
	/**
	 * 인식 서버 오류, 에러값은 14
	 */
	public static final int ERROR_NETWORK_NACK         = 14;
	/**
	 * 전송/응답 패킷 오류, 에러값은 15
	 */
	public static final int ERROR_PACKET               = 15;
	/**
	 * 오디오 초기화 오류, 에러값은 20
	 * <br>
	 * 오디오 입력은 기본적으로 안드로이드의 {@link android.media.AudioRecord} 를 사용하도록 설정되어 있다.
	 */
	public static final int ERROR_AUDIO_INITIALIZE     = 20;
	/**
	 * 오디오 메모리 해제 오류, 에러값은 21
	 * <br>
	 * 오디오 입력은 기본적으로 안드로이드의 {@link android.media.AudioRecord} 를 사용하도록 설정되어 있다.
	 */
	public static final int ERROR_AUDIO_FINIALIZE      = 21;
	/**
	 * 오디오 읽기 오류, 에러값은 22
	 */
	public static final int ERROR_AUDIO_RECORD         = 22;
	/**
	 * 인증 권한 오류, 에러값은 30
	 */
	public static final int ERROR_SECURITY             = 30;
	/**
	 * 인식 결과 없음, 에러값은 40
	 */
	public static final int ERROR_NO_REULST            = 40;
	/**
	 * 타임아웃 오류, 에러값은 41
	 * <br>
	 * 일정 시간동안 인식 서버가 음성을 받지 못하였을 경우 발생한다.  
	 */
	public static final int ERROR_TIMEOUT              = 41;
	/**
	 * 인식기 인스턴스 오류, 에러값은 42
	 */
	public static final int ERROR_NULL_CLIENT          = 42;
	/**
	 * 내부 이벤트 오류, 에러값은 50
	 * <br>
	 * 인식기 내부에서 규정되어 있지 않은 이벤트가 감지되었을 때 발생하는 오류이다.
	 */
	public static final int ERROR_UNKOWN_EVENT         = 50;
	/**
	 * 프로토콜 버전 오류, 에러값은 60
	 */
	public static final int ERROR_VERSION              = 60;
	/**
	 * 클라이언트 프로퍼티 오류, 에러값은 61
	 */
	public static final int ERROR_CLIENTINFO           = 61;
	/**
	 * 음성인식 서버의 풀(pool) 부족, 에러값은 62
	 */
	public static final int ERROR_SERVER_POOL          = 62;
	/**
	 * 음성인식 서버 세션 만료, 에러값은 63
	 */
	public static final int ERROR_SESSION_EXPIRED      = 63;
	/**
	 * HELLO 패킷을 수신하지 않음, 에러값은 64
	 */
	public static final int ERROR_NO_HELLO             = 64;
	/**
	 * 음성 패킷 사이즈 초과, 에러값은 65
	 */
	public static final int ERROR_SPEECH_SIZE_EXCEEDED = 65;
	/**
	 * 인증 time stamp 불량, 에러값은 66
	 */
	public static final int ERROR_EXCEED_TIME_LIMIT    = 66;
	/**
	 * 올바른 Service Type이 아님, 에러값은 67
	 */
	public static final int ERROR_WRONG_SERVICE_TYPE   = 67;
	/**
	 * 올바른 Language Type이 아님, 에러값은 68
	 */
	public static final int ERROR_WRONG_LANGUAGE_TYPE  = 68;
	/**
	 * OpenAPI 인증 에러(Client ID 또는 AndroidManifest.xml의 package가 개발자센터에 등록한 값과 다름), 에러값은 70
	 */
	public static final int ERROR_OPENAPI_AUTH         = 70;
	/**
	 * 정해진 Quota를 다 소진함, 에러값은 71
	 */
	public static final int ERROR_QUOTA_OVERFLOW       = 71;

	//////////////////////////////////////////////////////////////

	/**
	 * 생성자
	 * @param context 애플리케이션의 메인 액티비티
	 * @param clientId 개발자센터에서 "내 애플리케이션" 등록할 때 발급된 Client ID
	 * @throws SpeechRecognitionException 예외가 발생하는 경우는 아래와 같습니다.<br>
	 * 1. activity 파라미터가 올바른 MainActivity의 인스턴스가 아닙니다.<br>
	 * 2. AndroidManifest.xml에서 package를 올바르게 등록하지 않았습니다.<br>
	 * 3. package를 올바르게 등록했지만 과도하게 긴 경우, 256바이트 이하면 좋습니다.<br>
	 * 4. clientId가 null인 경우<br>
	 * 개발하면서 예외가 발생하지 않았다면 실서비스에서도 예외는 발생하지 않습니다. 개발 초기에만 주의하시면 됩니다.
	 */
	public SpeechRecognizer(Context context, String clientId) throws SpeechRecognitionException {
		this.mAudioCapture = new AudioCapture();

		String errMsg = setupJNI(CLIENT_LIB_VER, android.os.Build.MODEL, android.os.Build.VERSION.RELEASE, clientId, context);
		if (errMsg != null) {
			throw new SpeechRecognitionException(errMsg);
		}
	}

	/**
	 * 음성인식 자원을 초기화해준다. {@link android.app.Activity#onResume()} 콜백 호출 시점에 함께 호출되는 것을 권장한다.
	 */
	public void initialize() {
		initializeJNI();
	}

	/**
	 * 음성인식 자원을 해제해준다. {@link android.app.Activity#onPause()} 콜백 호출 시점에 함께 호출되는 것을 권장한다.
	 */
	public void release() {
		releaseJNI();
	}

	/**
	 * 커스텀 리스너를 등록한다.
	 * @param callback 커스텀 리스너.
	 * @see SpeechRecognitionListener
	 */
	public void setSpeechRecognitionListener(SpeechRecognitionListener callback) {
		this.speechRecognitionListener = callback;
	}

	public boolean selectEPDTypeInHybrid(EndPointDetectType epdType) {
		return selectEPDTypeInHybridJNI(epdType.toInteger());
	}

	/**
	 * 음성인식 정지를 요청한다. 이 메소드를 호출하는 즉시 SpeechRecognizer의 스테이트가 EndPointDetected로 천이 된다.
	 * 즉, 이 메소드를 호출함으로써 음성인식이 즉시 종료되는 것이 아니라, 서버로부터 인식 최종 결과가 오기를 기다렸다가, 결과를 받은 후 onResult 콜백 메소드를 호출하고 종료된다.
	 */
	public void stop() {
		if(isRunning())
			sendUserEPDJNI();
	}

	/**
	 * 음성인식을 즉시 정지한다.{@link SpeechRecognizer#stop()} 메소드와 달리,
	 * SpeechRecognizer의 모든 동작을 즉시 정지시킨다. 
	 */
	public void cancel() {
		stopAudioRecording();
		stopListeningJNI();
	}

	/**
	 * 현재 음성인식의 수행 여부를 반환한다.
	 * @return 음성인식이 수행 중일 경우 true를 반환한다.
	 */
	public boolean isRunning() {
		return isRunningJNI();
	}

	/**
	 * 음성인식을 시작한다. 이후에 SpeechRecognizer의 상태 천이에 따라 이벤트가 발생하며, 이를 커스텀 리스너에서 감지한다.
	 * @param config 음성인식 엔진 종류 설정
	 * @see SpeechRecognizer
	 * @see SpeechRecognitionListener
	 *
	 * @throws SpeechRecognitionException 음성인식을 중복 실행하거나, 클라이언트 정보가 null 이거나, 음성인식 대상 언어가 null 일 때 예외 발생.
	 */
	public void recognize(SpeechConfig config) throws SpeechRecognitionException {

		if (isRunning())
			throw new SpeechRecognitionException("Speech Recognizer is already running");

		if (config.getServiceType() == null)
			throw new SpeechRecognitionException("ServiceType is null");

		if (config.getLanguageType() == null)
			throw new SpeechRecognitionException("LanguageType is null");

		if (config.getEndPointDetectType() == null)
			throw new SpeechRecognitionException("EndPointDetectType is null");

		startListeningJNI(config.getServiceType().toInteger(),
						  config.getLanguageType().toInteger(),
						  config.isQuestionDetection(),
						  config.getEndPointDetectType().toInteger());
	}

	// JNI reflection
	private int startAudioRecording() {
		try {
			mAudioCapture.beforeStart();
		} catch (Exception e) {
			e.printStackTrace();
			return AudioCapture.ERROR;
		}
		return AudioCapture.OK;
	}

	private int stopAudioRecording() {
		try {
			mAudioCapture.beforeFinish();
		} catch (Exception e) {
			e.printStackTrace();
			return AudioCapture.ERROR;
		}
		return AudioCapture.OK;
	}

	private short[] record() {
		try {
			short[] speech = mAudioCapture.record();
			return speech;
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 
	 */
	protected void onInactive() {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onInactive();
		}
	}

	/**
	 * 
	 */
	protected void onReady() {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onReady();
		}
	}

	/**
	 * @param speech
	 */
	protected void onRecord(short[] speech) {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onRecord(speech);
		}
	}

	/**
	 * @param result
	 */
	protected void onPartialResult(String result) {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onPartialResult(result);
		}
	}

	/**
	 * 
	 */
	protected void onEndPointDetected() {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onEndPointDetected();
		}
	}

	/**
	 * @param result
	 */
	protected void onResult(Object[] result) {
		if (null != speechRecognitionListener) {
			SpeechRecognitionResult speechRecognitionResult = new SpeechRecognitionResult();
			ArrayList<String> results = new ArrayList<String>();
			results.add((String)result[0]);
			results.add((String)result[1]);
			results.add((String)result[2]);
			results.add((String)result[3]);
			results.add((String)result[4]);
			speechRecognitionResult.setResults(results);

			speechRecognitionResult.setGender(Integer.parseInt((String) result[5]));
			speechRecognitionListener.onResult(speechRecognitionResult);
		}
	}

	/**
	 * @param errorCode
	 */
	protected void onError(int errorCode) {
		if (null != speechRecognitionListener) {
			speechRecognitionListener.onError(errorCode);
		}
	}

	protected void onEndPointDetectTypeSelected(int epdType) {
		if (null != speechRecognitionListener) {
			switch(epdType) {
			case 0:
				speechRecognitionListener.onEndPointDetectTypeSelected(EndPointDetectType.AUTO);
				break;
			case 1:
				speechRecognitionListener.onEndPointDetectTypeSelected(EndPointDetectType.MANUAL);
				break;
			default:
				break;
			}
		}
	}

	static {
		System.loadLibrary(CLIENT_LIB_NAME + "-" + CLIENT_LIB_VER);
	}

	private native static String setupJNI(String client_lib_ver, String device, String os, String cliendId, Context context);

	private native void initializeJNI();
	private native void releaseJNI();

	private native boolean selectEPDTypeInHybridJNI(int epdType);
	private native void sendUserEPDJNI();

	private native void startListeningJNI(int serviceType, int languageType, boolean questionDetection, int epdType);
	private native void stopListeningJNI();

	private native boolean isRunningJNI();
}
