/*
 * @(#) SpeechRecognitionListener.java 2015. 1.
 *
 * Copyright 2015 Naver Corp. All rights Reserved.
 * Naver PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.naver.speech.clientapi;

import com.naver.speech.clientapi.SpeechConfig.EndPointDetectType;

/**
 * SpeechRecognizer 클래스에서 발생한 이벤트를 감지하여 해당하는 콜벡 메소드를 호출한다.
 * 커스텀 리스너는 선언되어 있는 콜백 메소드를 오버라이딩하여, 애플리케이션 메인 스레드에서 임의의 동작을 하도록 구현한다.
 * 커스텀 리스너를 사용하려면 반드시 {@link SpeechRecognizer#setSpeechRecognitionListener(SpeechRecognitionListener)} 메소드를 통해 등록되어야 한다.
 * 
 * @see SpeechRecognizer
 */
public interface SpeechRecognitionListener {

	/**
	 * 음성인식이 완료된 상태일 때 호출된다.
	 */
	public void onInactive();

	/**
	 * 음성인식 준비가 완료된 상태일 때 호출된다.
	 */
	public void onReady();

	/**
	 * 음성 입력을 받은 즉시 호출된다.
	 * @param speech 음성 신호가 저장된 버퍼.
	 */
	public void onRecord(short[] speech);

	/**
	 * 음성인식 서버로부터 인식 중간 결과를 받을 경우 호출된다. 중간 결과는 없거나 여러번 있을 수 있다.
	 * @param result 음성인식 중간 결과 문자열.
	 */
	public void onPartialResult(String result);

	/**
	 * 음성 발성이 종료되었음을 인지하였을 때 호출된다.
	 * 이는 음성인식 서버가 음성을 통해서 스스로 인지 후 결정하며,
	 * 유저가 직접 {@link SpeechRecognizer#stop()} 메소드를 호출하여 명시적으로 음성이 종료되었음을 서버에 알릴 수 있다.
	 * @see SpeechRecognizer#stop()
	 */
	public void onEndPointDetected();

	/**
	 * 음성인식 최종 결과를 받을 경우 호출된다.
	 * @param result 음성인식 최종 결과이다. 문자열 타입의 배열로 타입 변환하여 사용할 수 있다.
	 * 이 배열을 통해 음성 인식 결과를 전달받을 수 있고, 추가적으로 네이버 음성인식 내에서 정의한 음소 기호 세트를 전달 받을 수 있다.
	 * <p>
	 * 만약 단순하게 사용하기를 원한다면 results[0]만 사용하면 된다. results[0]이 음성인식 최종 결과이기 때문이다.
	 * 하지만 추가적인 정보를 사용하고 싶다면, 아래의 데이터에 접근하여 사용할 수 있다.
	 * <dl>
	 * <dt>results[1]</dt>
	 * <dd>
	 * 네이버 음성인식 내에서 정의한 음소 기호로 표기된 인식 결과이다.
	 * </dd>
	 * <dt>results[2]</dt>
	 * <dd>
	 * result[0]에 저장된 인식 결과의 출력 시간이다. 단어 단위로 인식 시간이 저장되며, 공백(' ')을 구분자로 하여 하나의 문자열로 저장된다.
	 * 이 때 인식의 시작 시간과 종료 시간이 함께 저장되기 때문에, 시간 정보의 수는 인식 결과의 단어 수보다 항상 2개 더 많아진다.
	 * 즉, 인식된 단어가 6개일 경우 시간 정보는 시작, 종료 시간을 포함하여 총 8개가 저장된다.
	 * </dd>
	 * <dt>results[3]</dt>
	 * <dd>
	 * result[1]에 저장된 음소 기호 결과의 출력 시간이다. 음소열 단위로 인식 시간이 저장되며, 공백(' ')을 구분자로 하여 하나의 문자열로 저장된다.
	 * 이 때 인식의 시작과 종료 시간이 함께 저장되기 때문에, 시간 정보의 수는 인식 결과의 음소열 수보다 항상 2개 더 많아진다.
	 * 즉, 인식된 음소열이 10개일 경우 시간 정보는 시작, 종료 시간을 포함하여 총 12개가 저장된다.
	 * </dd>
	 * </dl>
	 */
	public void onResult(SpeechRecognitionResult result);

	/**
	 * 음성인식중에 오류가 발생하였거나, 결과가 없을 경우 호출된다.
	 * @param errorCode 에러의 의미 코드.
	 * @see SpeechRecognizer#ERROR_AUDIO_FINIALIZE
	 * @see SpeechRecognizer#ERROR_AUDIO_INITIALIZE
	 * @see SpeechRecognizer#ERROR_AUDIO_RECORD
	 * @see SpeechRecognizer#ERROR_NETWORK_FINALIZE
	 * @see SpeechRecognizer#ERROR_NETWORK_INITIALIZE
	 * @see SpeechRecognizer#ERROR_NETWORK_NACK
	 * @see SpeechRecognizer#ERROR_NETWORK_READ
	 * @see SpeechRecognizer#ERROR_NETWORK_WRITE
	 * @see SpeechRecognizer#ERROR_NO_REULST
	 * @see SpeechRecognizer#ERROR_NULL_CLIENT
	 * @see SpeechRecognizer#ERROR_PACKET
	 * @see SpeechRecognizer#ERROR_SECURITY
	 * @see SpeechRecognizer#ERROR_TIMEOUT
	 * @see SpeechRecognizer#ERROR_UNKOWN_EVENT
	 * @see SpeechRecognizer#ERROR_VERSION
	 * @see SpeechRecognizer#ERROR_CLIENTINFO
	 * @see SpeechRecognizer#ERROR_SERVER_POOL
	 * @see SpeechRecognizer#ERROR_SESSION_EXPIRED
	 * @see SpeechRecognizer#ERROR_NO_HELLO
	 * @see SpeechRecognizer#ERROR_SPEECH_SIZE_EXCEEDED
	 * @see SpeechRecognizer#ERROR_EXCEED_TIME_LIMIT
	 */
	public void onError(int errorCode);

	/**
	 *
	 * @param epdType
	 */
	public void onEndPointDetectTypeSelected(EndPointDetectType epdType);
}
