package com.nanorep.accessibility.voice.engines.textToSpeech

import android.content.Context
import android.os.Build
import android.os.Bundle
import android.speech.tts.TextToSpeech
import android.speech.tts.UtteranceProgressListener
import android.util.Log
import com.nanorep.accessibility.voice.RecognitionErrorCodes.ERROR_READING
import com.nanorep.accessibility.voice.TEXT_TO_VOICE_NOT_AVAILABLE
import com.nanorep.sdkcore.model.Parser
import com.nanorep.sdkcore.utils.NRError
import java.util.*

interface TTSServiceListener {
    fun onReady() {}
    fun onError(error: NRError) {}
    fun onStart() {}
    fun onEnd(hasFinished: Boolean) {}
}

interface AsyncParser : Parser<Any, (String?)->Unit> {

    override fun parse(data: Any): (String?)->Unit {
        return {}
    }

    fun parse(data: Any, callback: (String?)->Unit) : Unit
}


/**
 * @param parser Parses the object to be read
 * @param isActive Defines whether to enabling reading the object at the read method
 * @param onFinished A method that is triggered at the progressListener when done reading - working for the default ProgressListener only
 */
interface TTSController : TTSConfigProvider {
    var parser: AsyncParser?
    var serviceListener: TTSServiceListener?
    var isActive: Boolean
    fun read(data: Any?) {}
    fun stop() {}
    fun clear() {}
    fun isSpeaking(): Boolean? {
        return false
    }
}

interface TTSConfigProvider {
    var speechRate: Float?
    var language: Locale?
    var pitch: Float?
    var volume: Float?
}

/**
 * Default TTSController class that encapsulates the Android's engine
 */
class TTSEngine(val context: Context) : TTSController {

    /**
     * The android's Text To Speech engine
     */
    private var innerTextToSpeech: TextToSpeech? = null

    /**
     * The engine is being supported at the device / Locale or not
     */
    private var supported = true

    /**
     * Set to true when the engine has been initialized properly
     */
    private var ready: Boolean = false

    /**
     * Parameters for the speak request
     */
    private var params: Any? = null

    /**
     * Listens to the speak progress.
     * currently internal implementation only. Calls TTSServiceListener status methods
     */
    private val progressListener: UtteranceProgressListener? by lazy {
        ProgressListener()
    }

    override var serviceListener: TTSServiceListener? = null

    override var parser: AsyncParser? = null

    override var isActive: Boolean = false
        set(value) {
            field = value
            stop()
        }

    private val TextToSpeech.myLanguage: Locale?
        get() {
            try {
                return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP)
                    innerTextToSpeech?.voice?.locale
                else innerTextToSpeech?.language
            } catch (e: Throwable) {
            }
            return Locale.getDefault()
        }

    override var language: Locale?
        set(value) {
            innerTextToSpeech?.takeIf { it.myLanguage != value }?.apply {
                value?.takeIf { assertLanguageSupported(it) }?.let { this.language = it }
            }
        }
        get() {
            return innerTextToSpeech?.myLanguage
        }

    override var speechRate: Float? = null
        set(value) {
            field = value?.also {
                innerTextToSpeech?.setSpeechRate(it)
            }
        }

    override var pitch: Float? = null
        set(value) {
            field = value?.also {
                innerTextToSpeech?.setPitch(it)
            }
        }

    override var volume: Float? = null
        set(value) {
            field = value?.also {
                params = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    Bundle().apply { putFloat(TextToSpeech.Engine.KEY_PARAM_VOLUME, it) }
                } else {
                    getMapParams().apply {
                        put(TextToSpeech.Engine.KEY_PARAM_VOLUME, it.toString())
                    }
                }
            }
        }

    private val onInitListener = TextToSpeech.OnInitListener { status ->
        val error = when (status) {

            TextToSpeech.SUCCESS -> null

            TextToSpeech.LANG_NOT_SUPPORTED -> "language not supported"

            TextToSpeech.LANG_MISSING_DATA -> "language not available"

            else -> "some error occurred"

        }

        error?.run {
            serviceListener?.onError(NRError(TEXT_TO_VOICE_NOT_AVAILABLE, error))
        } ?: kotlin.run {
            // we can set language to the TextToSpeech object
            ttsReady()
        }
    }

    init {
        innerTextToSpeech = TextToSpeech(context.applicationContext, onInitListener)
    }

    private fun ttsReady() {

        if (ready) return

        this.ready = true

        innerTextToSpeech?.takeIf { it.language == null }?.setLanguage(language)?.takeIf { it == TextToSpeech.LANG_MISSING_DATA }?.run {
            Log.e("TTS", "This Language has missing data")
        }

        assertServiceAvailable()

        if (supported) {
            initProgressListener()
            serviceListener?.onReady()
        }
    }

    private fun assertServiceAvailable() {
        // Here the language is still Locale.getDefault()
        supported = innerTextToSpeech?.engines?.isNotEmpty() ?: false
    }

    /**
     *  This method checks is the language is supported by the engine
     *  NOTE: Before the engine is initialized it always returns false ( because TextToSpeech.LANG_NOT_SUPPORTED )
     */
    private fun assertLanguageSupported(language: Locale): Boolean {

        val cond = innerTextToSpeech?.let { tts ->

            ((Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP || tts.availableLanguages?.isNotEmpty() == true)
                    && (tts.isLanguageAvailable(language) != TextToSpeech.LANG_NOT_SUPPORTED))
        } ?: false

        return cond.also {
            if (!it && ready)
                Log.e("TTS", "This Language is not supported")
        }

    }

    private fun initProgressListener() {
        innerTextToSpeech?.setOnUtteranceProgressListener(progressListener)
    }


    inner class ProgressListener : UtteranceProgressListener() {

        var handleStopped: ((pausePosition: Int) -> Unit)? = null // TBD would be used when trying to resume the reading from pause

        override fun onRangeStart(utteranceId: String?, start: Int, end: Int, frame: Int) {
            super.onRangeStart(utteranceId, start, end, frame)
        }

        override fun onDone(utteranceId: String?) {
            isActive = false
            serviceListener?.onEnd(true)
            Log.d("TTS", "Done speaking")
        }

        override fun onError(utteranceId: String?) {
            Log.d("TTS", "some error occurred")
            serviceListener?.onError(NRError(ERROR_READING, reason = utteranceId))
        }

        override fun onStop(utteranceId: String?, interrupted: Boolean) {
            serviceListener?.onEnd(false)
        }

        override fun onStart(utteranceId: String?) {
            Log.d("TTS", "Started speaking")
            serviceListener?.onStart()
        }
    }

    fun setListener(serviceListener: TTSServiceListener?) = apply {
        this.serviceListener = serviceListener
        if (!supported) {
            this.serviceListener?.onError(NRError(TEXT_TO_VOICE_NOT_AVAILABLE, "language not supported"))
        }
    }

    override fun read(data: Any?) {
        // Log.v("TTS", "read(): ready = $ready, isActive = $isActive, data = $data")

        if (!ready || !isActive || data == null) return

        parser?.parse(data){ text ->
            if (params == null) {
                params = getParams()
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                innerTextToSpeech?.speak(text, TextToSpeech.QUEUE_FLUSH, params as Bundle, MESSEGE_ID)
            } else {
                innerTextToSpeech?.speak(text, TextToSpeech.QUEUE_FLUSH, params as HashMap<String, String>)
            }
        }
    }

    override fun isSpeaking(): Boolean? {
        return innerTextToSpeech?.isSpeaking
    }

    override fun stop() {
        innerTextToSpeech?.stop() ?: Log.e("TTS", "TextToSpeech engine is null")
    }

    override fun clear() {
        innerTextToSpeech?.shutdown()
        innerTextToSpeech?.setOnUtteranceProgressListener(null)
        serviceListener = null
        innerTextToSpeech = null
    }

    private fun getParams(): Any {
        return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            Bundle()
        } else {
            getMapParams()
        }
    }

    private fun getMapParams(): HashMap<String, String> {
        return hashMapOf(TextToSpeech.Engine.KEY_PARAM_UTTERANCE_ID to MESSEGE_ID)
    }

    companion object {
        const val MESSEGE_ID = "message"
        const val END = -1
    }
}
