/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.exception;

import java.util.List;
import java.util.stream.Collectors;

import static org.mule.munit.common.util.Preconditions.checkNotNull;
import org.mule.munit.common.util.Throwables;

/**
 * The goal of this class is to find {@link DeploymentException} in a exception cause
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class DeploymentExceptionThrower {

  private static final String DEPLOYMENT_EXCEPTION_CANONICAL_NAME = "org.mule.runtime.deployment.model.api.DeploymentException";

  /**
   * It will throw a {@link DeploymentException} if a Mule Deployment exception is found in the causal chain. If not the original
   * exception will be thrown.
   *
   * @param e exception to analyze
   * @throws DeploymentException if a Mule Deployment exception is found
   * @throws Throwable           the original exception
   */
  public static void throwIfMatches(RuntimeException e) throws DeploymentException, RuntimeException {
    checkNotNull(e, "The exception must no be null");

    List<Throwable> deploymentExceptions = findCauses(e);
    if (!deploymentExceptions.isEmpty()) {
      throw new DeploymentException(deploymentExceptions.get(0));
    }
    throw e;
  }


  private static List<Throwable> findCauses(Throwable e) {
    List<Throwable> causalChain = Throwables.getCausalChain(e);
    return causalChain.stream()
        .filter(DeploymentExceptionThrower::isDeploymentException)
        .collect(Collectors.toList());
  }

  /**
   * Checks if the given throwable is a Mule DeploymentException instance.
   *
   * @param t the throwable to check
   * @return true if the throwable is a DeploymentException, false otherwise
   */
  public static boolean isDeploymentException(Throwable t) {
    if (t == null) {
      return false;
    }
    try {
      Class<?> deploymentExceptionClass =
          Class.forName(DEPLOYMENT_EXCEPTION_CANONICAL_NAME, false, t.getClass().getClassLoader());
      return deploymentExceptionClass.isInstance(t);
    } catch (ClassNotFoundException ex) {
      return false;
    }
  }

}
