/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage.model;

import org.mule.munit.remote.coverage.server.ComponentLocation;
import org.mule.munit.remote.coverage.server.LocationPart;

import java.io.Serializable;
import java.util.*;

import static java.util.Collections.unmodifiableList;
import static java.util.Optional.*;

/**
 * A component location describes where the component is defined in the configuration of the artifact.
 * 
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageComponentLocationReport implements ComponentLocation, Serializable {

  private static final long serialVersionUID = 4958158607813720623L;

  private final String name;
  private final LinkedList<CoverageLocationPartReport> parts;
  private volatile String location;

  /**
   * @param name the name of the global element in which the specific component is located.
   * @param parts the set of parts to locate the component.
   */
  public CoverageComponentLocationReport(Optional<String> name, List<CoverageLocationPartReport> parts) {
    this.name = name.orElse(null);
    this.parts = new LinkedList<>(parts);
  }

  /**
   * {@inheritDoc}
   */
  public Optional<String> getName() {
    return ofNullable(name);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<LocationPart> getParts() {
    return unmodifiableList(parts);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<String> getFileName() {
    return parts.getLast().getFileName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<Integer> getLineInFile() {
    return parts.getLast().getLineInFile();
  }

  /**
   * @return a string representation of the {@link CoverageComponentLocationReport}.
   */
  @Override
  public String getLocation() {
    if (location == null) {
      synchronized (this) {
        if (location == null) {
          StringBuilder locationBuilder = new StringBuilder();
          for (CoverageLocationPartReport part : parts) {
            locationBuilder.append("/").append(part.getPartPath());
          }
          location = locationBuilder.replace(0, 1, "").toString();
        }
      }
    }
    return location;
  }



  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    CoverageComponentLocationReport that = (CoverageComponentLocationReport) o;

    if (getName() != null ? !getName().equals(that.getName()) : that.getName() != null) {
      return false;
    }
    if (!getParts().equals(that.getParts())) {
      return false;
    }
    return getLocation() != null ? getLocation().equals(that.getLocation()) : that.getLocation() == null;
  }

  @Override
  public int hashCode() {
    int result = getName() != null ? getName().hashCode() : 0;
    result = 31 * result + getParts().hashCode();
    result = 31 * result + (getLocation() != null ? getLocation().hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "DefaultComponentLocation{" +
        "name='" + name + '\'' +
        ", parts=" + parts +
        ", location='" + getLocation() + '\'' +
        '}';
  }

  @Override
  public Optional<Integer> getStartColumn() {
    // TODO Auto-generated method stub
    return Optional.empty();
  }

  @Override
  public OptionalInt getLine() {
    // TODO Auto-generated method stub
    return OptionalInt.empty();
  }

  @Override
  public OptionalInt getColumn() {
    // TODO Auto-generated method stub
    return OptionalInt.empty();
  }
}
