/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.coverage;

import static org.mule.munit.common.util.Preconditions.checkArgument;
import static org.mule.munit.common.util.Preconditions.checkNotNull;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.mule.munit.remote.coverage.model.CoverageComponentLocationReport;
import org.mule.munit.remote.coverage.model.ApplicationCoverageReport;
import org.mule.munit.remote.coverage.model.MuleFlow;
import org.mule.munit.remote.coverage.model.MuleLocation;
import org.mule.munit.remote.coverage.model.MuleResource;
import org.mule.munit.remote.coverage.server.LocationPart;

/**
 * It knows how to build an {@link ApplicationCoverageReport}
 * 
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class ApplicationCoverageReportBuilder {

  private Set<String> suitePaths;
  private Set<String> flowsToIgnore = emptySet();
  private Set<String> filesToIgnore = emptySet();

  private Set<CoverageComponentLocationReport> allLocations;
  private Set<CoverageComponentLocationReport> coveredLocations;

  public ApplicationCoverageReportBuilder(Set<CoverageComponentLocationReport> allLocations,
                                          Set<CoverageComponentLocationReport> coveredLocations, Set<String> suitePaths) {
    checkNotNull(allLocations, "The set of locations must not be null");
    checkNotNull(coveredLocations, "The set of covered locations must not be null");
    checkNotNull(suitePaths, "The suite path list  must not be null");
    checkArgument(!suitePaths.isEmpty(), "The suite path list  must not be empty");

    this.allLocations = allLocations;
    this.coveredLocations = coveredLocations;

    this.suitePaths = suitePaths;
  }

  public void setFlowsToIgnore(Set<String> flowsToIgnore) {
    checkNotNull(flowsToIgnore, "The set of flows to ignore must not be null");
    this.flowsToIgnore = flowsToIgnore;
  }

  public void setFilesToIgnore(Set<String> filesToIgnore) {
    checkNotNull(filesToIgnore, "The set of files to ignore must not be null");
    this.filesToIgnore = filesToIgnore;
  }

  public Set<String> getSuitePaths() {
    return suitePaths;
  }

  public Set<String> getFlowsToIgnore() {
    return flowsToIgnore;
  }

  public Set<String> getFilesToIgnore() {
    return filesToIgnore;
  }

  public Set<CoverageComponentLocationReport> getCoveredLocations() {
    return coveredLocations;
  }

  public ApplicationCoverageReport build() {
    return new ApplicationCoverageReport(createMuleResources());
  }

  private List<MuleResource> createMuleResources() {
    Map<String, Set<CoverageComponentLocationReport>> locationsPerFile = new LocationGrouper(allLocations).groupByFile();
    return locationsPerFile.entrySet().stream()
        .filter(location -> isNotIgnoredFile(location.getKey()))
        .filter(location -> isNotSuiteFile(location.getKey()))
        .map(location -> toMuleResource(location.getKey(), location.getValue()))
        .collect(toList());
  }

  private MuleResource toMuleResource(String fileName, Set<CoverageComponentLocationReport> fileLocations) {
    MuleResource muleResource = new MuleResource(fileName);
    muleResource.setFlows(createMuleFlows(fileLocations));
    return muleResource;
  }

  private List<MuleFlow> createMuleFlows(Set<CoverageComponentLocationReport> locations) {
    Map<LocationPart, Set<CoverageComponentLocationReport>> locationsPerFlow = new LocationGrouper(locations).groupByFlow();
    return locationsPerFlow.entrySet().stream()
        .filter(location -> isNotIgnoredFlow(location.getKey()))
        .map(location -> toMuleFlow(location.getKey(), location.getValue()))
        .collect(toList());
  }

  private MuleFlow toMuleFlow(LocationPart locationPart, Set<CoverageComponentLocationReport> flowLocations) {
    MuleFlow muleFlow = new MuleFlow(locationPart.getPartPath());

    muleFlow.setType(locationPart.getPartIdentifier().get().getType().toString());
    muleFlow.setLocations(flowLocations.stream().map(l -> toMuleLocation(l)).collect(toList()));
    muleFlow.setCoveredLocations(flowLocations.stream()
        .filter(l -> coveredLocations.contains(l))
        .map(l -> toMuleLocation(l)).collect(toList()));
    return muleFlow;
  }

  private MuleLocation toMuleLocation(CoverageComponentLocationReport coverageComponentLocation) {
    return new MuleLocation(coverageComponentLocation.getLocation(), coverageComponentLocation.getLineInFile().orElse(-1));
  }

  private boolean isNotIgnoredFlow(LocationPart locationPart) {
    String flowName = locationPart.getPartPath();
    return !flowsToIgnore.contains(flowName);
  }

  private boolean isNotIgnoredFile(String fileName) {
    return !filesToIgnore.contains(fileName);
  }

  private boolean isNotSuiteFile(String fileName) {
    return !suitePaths.contains(fileName);
  }

}
