/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote.container;

import static org.mule.munit.common.util.StackTraceUtil.getStackTrace;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.munit.remote.properties.deploy.MuleRuntimeDeploymentProperties.isAtLeastMinMuleVersion;
import static org.mule.munit.remote.runtime.utils.Product.MULE_EE;

import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.apache.commons.lang3.StringUtils.EMPTY;

import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.util.FileUtils;
import org.apache.commons.io.IOUtils;
import org.mule.munit.common.util.IllegalPortDefinitionException;
import org.mule.munit.common.util.RunnerPortProvider;
import org.mule.munit.remote.MinMuleVersionConfig;
import org.mule.munit.remote.coverage.model.ApplicationCoverageReport;
import org.mule.munit.remote.MuleDxMunitLock;
import org.mule.munit.remote.api.client.RunnerClient;
import org.mule.munit.remote.api.configuration.CoverageConfiguration;
import org.mule.munit.remote.api.configuration.DebuggerConfiguration;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.container.model.SuiteDeployment;
import org.mule.munit.remote.container.model.SuiteRun;
import org.mule.munit.remote.coverage.CoverageManager;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.properties.Parameterization;
import org.mule.munit.remote.properties.deploy.DeploymentProperties;
import org.mule.munit.remote.properties.deploy.MuleRuntimeDeploymentProperties;
import org.mule.munit.remote.properties.deploy.TemporaryFolderProperties;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import com.google.gson.Gson;

import org.mule.munit.remote.runtime.utils.MuleApplicationModel;
import org.mule.munit.remote.runtime.utils.MuleApplicationModelJsonSerializer;
import org.mule.munit.remote.runtime.utils.Product;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

/**
 * Dispatches a Suite Run to {@link Container} with a given {@link RunConfiguration}
 *
 * @author Mulesoft Inc.
 * @since 2.2.0
 */
public class ContainerSuiteRunDispatcher implements SuiteRunDispatcher {

  public static final String MUNIT_SUITE_TIMEOUT_PROPERTY = "munit.suite.timeout";
  private static final Long DEFAULT_TIMEOUT = 30 * 60 * 1000L;

  private transient Logger LOGGER = LogManager.getLogger(this.getClass());

  private final Container container;
  public static final String MULE_ARTIFACT_JSON = "mule-artifact.json";
  private final RunConfiguration runConfig;
  private final MuleApplicationModel originalMuleApplicationModel;
  private final Set<SuiteRun> suiteRuns;
  private final Collection<DeploymentProperties> deploymentProperties;
  private final File applicationDirectory;
  private final int munitRunnerPort;
  private final CoverageManager coverageManager;
  private final List<SuiteRunDispatcherListener> suiteRunDispatcherListeners = new ArrayList<>();
  private Set<String> minMuleVersions;

  public ContainerSuiteRunDispatcher(Container container, RunConfiguration runConfiguration, Set<SuiteRun> suiteRuns,
                                     File applicationDirectory, List<String> minMuleVersions) {
    this.container = container;
    this.runConfig = runConfiguration;
    this.suiteRuns = suiteRuns;
    this.applicationDirectory = applicationDirectory;
    this.originalMuleApplicationModel = getOriginalApplicationModel();
    this.deploymentProperties = getDeploymentProperties();
    this.munitRunnerPort = getPort();
    this.coverageManager = buildCoverageManager(runConfig.getCoverageConfiguration());
    this.minMuleVersions = new HashSet<>(minMuleVersions);
  }

  protected Integer getPort() throws ContainerFactoryException {
    try {
      return new RunnerPortProvider().getPort();
    } catch (IllegalPortDefinitionException e) {
      throw new ContainerFactoryException(e.getMessage(), e);
    }
  }

  private MuleApplicationModel getOriginalApplicationModel() {
    try {
      File muleArtifactJsonFile = getMuleArtifactJsonFile();
      return new MuleApplicationModelJsonSerializer()
          .deserialize(IOUtils.toString(muleArtifactJsonFile.toURI(), Charset.defaultCharset()));
    } catch (Exception e) {
      throw new IllegalStateException(e);
    }
  }


  private Collection<DeploymentProperties> getDeploymentProperties() {
    List<DeploymentProperties> deploymentProperties = new ArrayList<>();
    deploymentProperties.add(new MuleRuntimeDeploymentProperties(runConfig));
    return deploymentProperties;
  }

  private File getMuleArtifactJsonFile() {
    return Paths.get(applicationDirectory.toURI()).resolve(META_INF.value()).resolve(MULE_ARTIFACT.value())
        .resolve(MULE_ARTIFACT_JSON).toFile();
  }

  @Override
  public void runSuites(RemoteRunEventListener listener) throws DeploymentException {
    if (suiteRuns.stream().noneMatch(this::shouldRunSuite)) {
      LOGGER.info("Suite run for suites " + suiteRuns + " will not start since no suites will run");
      return;
    }
    try {
      if (!container.isJvmVersionSupported()) {
        LOGGER.info("The current java version is not supported by the runtime version [" + container.getMuleContainerVersion()
            + "].");
        return;
      }
      getCoverageManager().ifPresent(CoverageManager::startCoverageServer);
      container.start();
      container.deployDomain();
      Collection<SuiteDeployment> suiteDeployments = generateSuiteDeployments();
      Map<String, Object> initialProperties = getInitialProperties(suiteDeployments, runConfig.getClearParameters());
      for (SuiteDeployment suiteDeployment : suiteDeployments) {
        performSuiteDeploy(listener, suiteDeployment, initialProperties);
      }
    } catch (SkipAfterFailureException e) {
      LOGGER.debug("Skipped running suites since skipAfterFailure is on");
    } finally {
      try {
        container.stop();
      } finally {
        getCoverageManager().ifPresent(CoverageManager::stopCoverageServer);
        getCoverageManager().ifPresent(manager -> sendCoverageReport(manager, listener));
        suiteRunDispatcherListeners.forEach(SuiteRunDispatcherListener::onFinish);
      }
    }
  }

  private Collection<SuiteDeployment> generateSuiteDeployments() {
    List<SuiteDeployment> suiteDeployments = new ArrayList<>();
    suiteDeployments.addAll(getParameterizedDeployments());
    getDesignTimeSuitesDeployment().ifPresent(suiteDeployments::add);
    getRuntimeSuitesDeployment().ifPresent(suiteDeployments::add);
    return suiteDeployments;
  }

  private void performSuiteDeploy(RemoteRunEventListener listener, SuiteDeployment suiteDeployment,
                                  Map<String, Object> initialProperties)
      throws SkipAfterFailureException {
    Set<SuiteRun> suiteRuns = suiteDeployment.getSuiteRuns();
    if (suiteRuns.stream().noneMatch(this::shouldRunSuite)) {
      LOGGER.info("Suite run for suites " + suiteRuns + " will not start since no suites will run");
      return;
    }
    Map<String, Object> systemProperties = getSystemProperties(initialProperties, suiteDeployment);
    try {
      MuleApplicationModel applicationModel = originalMuleApplicationModel;
      applicationModel = removeSuitesThatWontRun(suiteRuns, applicationModel);
      // if minMuleVersions is not empty is because mtf user specified the minMuleVersions in the pom
      if (minMuleVersions.isEmpty()) {
        //The only case in which we should avoid running the default config is when mtf user set MMVConfig as RUNTIME
        if (runConfig.getMinMuleVersionConfig() == null
            || !runConfig.getMinMuleVersionConfig().equals(MinMuleVersionConfig.RUNTIME)) {
          minMuleVersions.add(applicationModel.getMinMuleVersion());
        }
        if (runConfig.getMinMuleVersionConfig() != null
            && (runConfig.getMinMuleVersionConfig().equals(MinMuleVersionConfig.BOTH)
                || runConfig.getMinMuleVersionConfig().equals(MinMuleVersionConfig.RUNTIME))) {
          minMuleVersions.add(container.getMuleContainerVersion());
        }
      }
      for (String version : minMuleVersions) {
        updateMMV(applicationModel, version);
        systemProperties.putAll((new TemporaryFolderProperties()).get());
        LOGGER.info("Suite will run with MMV " + version);
        deploySuite(suiteDeployment, systemProperties, listener, suiteRuns);
      }

    } catch (DeploymentException e) {
      if (suiteRuns.size() > 1) {
        listener.notifyContainerFailure(getStackTrace(e));
      } else {
        SuiteRun suiteRun = suiteRuns.iterator().next();
        listener.notifyContainerFailure(suiteRun.getSuitePath(), getParameterizationName(suiteRun), getStackTrace(e));
      }
    } catch (Exception e) {
      try {
        container.undeployApplication(systemProperties);
      } catch (Exception ex) {
        LOGGER.error("Exception undeploying application.", ex);
      }
      if (e.getClass().equals(SkipAfterFailureException.class)) {
        throw new SkipAfterFailureException();
      }
    }
  }

  private void deploySuite(SuiteDeployment suiteDeployment, Map<String, Object> systemProperties, RemoteRunEventListener listener,
                           Set<SuiteRun> suiteRuns)
      throws SkipAfterFailureException, DeploymentException {
    container.deployApplication(suiteDeployment.isEnableXmlValidations(), systemProperties);
    waitForDebuggerClient();
    for (SuiteRun suiteRun : suiteRuns) {
      if (shouldRunSuite(suiteRun)) {
        String suitePath = suiteRun.getSuitePath();
        String parameterizationName = getParameterizationName(suiteRun);
        boolean success = runSuite(runConfig, listener, suitePath, parameterizationName);
        if (!success && runConfig.isSkipAfterFailure()) {
          throw new SkipAfterFailureException();
        }
      } else {
        showSuiteSkippedMessage(suiteRun);
      }
    }
    try {
      container.undeployApplication(systemProperties);
    } catch (Exception e) {
      LOGGER.error("Exception undeploying application.", e);
    }
  }

  private boolean runSuite(RunConfiguration runConfig, RemoteRunEventListener listener, String suitePath,
                           String parameterizationName) {
    Future<Boolean> clientTask = null;
    try {
      RunnerClient runnerClient = getRunnerClient(listener);
      runnerClient.sendSuiteRunInfo(runConfig.getRunToken(), suitePath, parameterizationName, runConfig.getTestNames(),
                                    runConfig.getTestNamesWithSuite(),
                                    runConfig.getTags());
      ExecutorService executor = Executors.newCachedThreadPool();
      clientTask = executor.submit(runnerClient::receiveAndNotify);
      return clientTask.get(getSuiteTimeout(), MILLISECONDS);
    } catch (RuntimeException | IOException | InterruptedException e) {
      listener.notifyContainerFailure(suitePath, parameterizationName, getStackTrace(e));
      return false;
    } catch (ExecutionException e) {
      listener.notifyContainerFailure(suitePath, parameterizationName, getStackTrace(e.getCause()));
      return false;
    } catch (TimeoutException e) {
      String timeoutMessage = "Suite timeout after " + getSuiteTimeout() + " milliseconds."
          + " To change the default timeout use the " + MUNIT_SUITE_TIMEOUT_PROPERTY + " property";
      LOGGER.error(timeoutMessage, e);
      listener.notifyUnexpectedError(timeoutMessage);
      return false;
    } finally {
      if (clientTask != null) {
        clientTask.cancel(true);
      }
    }
  }

  protected RunnerClient getRunnerClient(RemoteRunEventListener listener) throws IOException {
    return new RunnerClient(munitRunnerPort, listener);
  }

  /**
   * Suites that won't run will be removed
   */
  private MuleApplicationModel removeSuitesThatWontRun(Set<SuiteRun> suiteRuns, MuleApplicationModel applicationModel) {
    Set<String> skippedSuites = suiteRuns.stream().filter(this::shouldNotRunSuite).map(SuiteRun::getSuitePath).collect(toSet());
    if (!skippedSuites.isEmpty()) {
      LOGGER.info("Suites " + skippedSuites + " will be skipped");
    }
    Set<String> suitesToRun = suiteRuns.stream().filter(this::shouldRunSuite).map(SuiteRun::getSuitePath).collect(toSet());
    Set<String> newConfigs = applicationModel.getConfigs().stream()
        .filter(config -> isNonSuiteConfigFile(config) || suitesToRun.contains(config)).collect(toSet());
    try {
      MuleApplicationModel muleApplicationModel = overrideConfigs(applicationModel, newConfigs);
      String newMuleApplicationModelString = new MuleApplicationModelJsonSerializer().serialize(muleApplicationModel);
      FileUtils.write(getMuleArtifactJsonFile(), newMuleApplicationModelString, Charset.defaultCharset());
      return muleApplicationModel;
    } catch (IOException e) {
      throw new IllegalStateException("An error occurred while regenerating the Mule Application Model", e);
    }
  }

  protected void updateMMV(MuleApplicationModel applicationModel, String version) {
    try {
      applicationModel.setMinMuleVersion(version);
      String newMuleApplicationModelString = new MuleApplicationModelJsonSerializer().serialize(applicationModel);
      FileUtils.write(getMuleArtifactJsonFile(), newMuleApplicationModelString, Charset.defaultCharset());
    } catch (IOException e) {
      throw new IllegalStateException("An error occurred while regenerating the Mule Application Model", e);
    }
  }

  /**
   * Files that are not suites but are configuration files should always be deployed
   */
  private boolean isNonSuiteConfigFile(String config) {
    return !runConfig.getAllSuitePaths().contains(config);
  }

  private Map<String, Object> getSystemProperties(Map<String, Object> initialProperties, SuiteDeployment suiteDeployment) {
    Map<String, Object> systemProperties = new HashMap<>(initialProperties);
    deploymentProperties.forEach(properties -> systemProperties.putAll(properties.get()));
    systemProperties.putAll(suiteDeployment.getSystemProperties());
    return systemProperties;
  }

  private void showSuiteSkippedMessage(SuiteRun suiteRun) {
    String message = "Suite " + suiteRun.getSuitePath() + " will not be deployed: ";
    if (suiteRun.isIgnored()) {
      message += "Suite is ignored";
    } else if (!isCurrentRuntimeAtLeastMinVersion(suiteRun)) {
      message += format("Current runtime version [%s] is lower that suite's minMuleVersion [%s]", getRuntimeVersion(),
                        suiteRun.getMinMuleVersion().orElse("N/A"));
    } else if (!isSuiteFiltered(suiteRun)) {
      message += "Suite was filtered from running";
    } else if (!isCurrentRuntimeProductAllowedToRun(suiteRun)) {
      message += format("Suite is expected to run only with runtime product [%s] and the current runtime is [%s]",
                        suiteRun.getRequiredProduct(), getContainerRuntimeProduct());
    }
    LOGGER.info(message);
  }

  private boolean shouldRunSuite(SuiteRun suiteRun) {
    return !suiteRun.isIgnored() && isCurrentRuntimeAtLeastMinVersion(suiteRun) &&
        isSuiteFiltered(suiteRun) && isCurrentRuntimeProductAllowedToRun(suiteRun);
  }

  private boolean shouldNotRunSuite(SuiteRun suiteRun) {
    return !shouldRunSuite(suiteRun);
  }

  private Boolean isCurrentRuntimeAtLeastMinVersion(SuiteRun suiteRun) {
    return suiteRun.getMinMuleVersion().map(minMuleVersion -> isAtLeastMinMuleVersion(getRuntimeVersion(), minMuleVersion))
        .orElse(true);
  }

  private boolean isSuiteFiltered(SuiteRun suiteRun) {
    return runConfig.getSuitePaths().contains(suiteRun.getSuitePath());
  }

  private boolean isCurrentRuntimeProductAllowedToRun(SuiteRun suiteRun) {
    Product containerRuntimeProduct = getContainerRuntimeProduct();
    Product currentSuiteRequiredProduct = suiteRun.getRequiredProduct();

    return !(currentSuiteRequiredProduct == MULE_EE && containerRuntimeProduct != MULE_EE);
  }

  private class SkipAfterFailureException extends Exception {

  }

  /**
   * This step is to ensure that the deployment is successful whether parameterized suites are run or not
   * 
   * @param clearParameters
   */
  private Map<String, Object> getInitialProperties(Collection<SuiteDeployment> suiteDeployments, Boolean clearParameters) {
    Map<String, Object> initialProperties = new HashMap<>();
    if (!clearParameters) {
      suiteDeployments.forEach(suite -> initialProperties.putAll(suite.getSystemProperties()));
    }
    return initialProperties;
  }

  public static MuleApplicationModel overrideConfigs(MuleApplicationModel originalMuleApplicationModel, Set<String> configs) {
    MuleApplicationModel.MuleApplicationModelBuilder builder = new MuleApplicationModel.MuleApplicationModelBuilder();
    builder.setConfigs(configs);
    builder.setMinMuleVersion(originalMuleApplicationModel.getMinMuleVersion());
    builder.setName(originalMuleApplicationModel.getName());
    builder.setRedeploymentEnabled(true);
    builder.setRequiredProduct(originalMuleApplicationModel.getRequiredProduct());
    builder.setSecureProperties(originalMuleApplicationModel.getSecureProperties());
    builder.withBundleDescriptorLoader(originalMuleApplicationModel.getBundleDescriptorLoader());
    builder.withClassLoaderModelDescriptorLoader(originalMuleApplicationModel.getClassLoaderModelLoaderDescriptor());
    return builder.build();
  }

  private List<SuiteDeployment> getParameterizedDeployments() {
    return suiteRuns.stream().filter(suiteRun -> suiteRun.getParameterization().isPresent())
        .map(this::createParameterizedSuiteDeployment).collect(toList());
  }

  private SuiteDeployment createParameterizedSuiteDeployment(SuiteRun suiteRun) {
    Parameterization parameterization = suiteRun.getParameterization().get();
    return SuiteDeployment.builder().withSuiteRuns(Collections.singleton(suiteRun))
        .withSystemProperties(parameterization.getParameters())
        .withEnableXmlValidations(!suiteRun.isDesignTime()).build();
  }

  private Set<SuiteRun> getDesignTimeSuites() {
    return suiteRuns.stream().filter(SuiteRun::isDesignTime).collect(Collectors.toSet());
  }

  private Optional<SuiteDeployment> getDesignTimeSuitesDeployment() {
    Set<SuiteRun> designTimeSuites =
        getDesignTimeSuites().stream().filter(suiteRun -> !suiteRun.getParameterization().isPresent()).collect(toSet());
    if (designTimeSuites.isEmpty()) {
      return Optional.empty();
    }
    return of(SuiteDeployment.builder().withSuiteRuns(designTimeSuites).withEnableXmlValidations(false).build());
  }

  private Optional<SuiteDeployment> getRuntimeSuitesDeployment() {
    Set<SuiteRun> runtimeSuites = suiteRuns.stream().filter(suiteRun -> !suiteRun.isDesignTime())
        .filter(suiteRun -> !suiteRun.getParameterization().isPresent()).collect(toSet());
    if (runtimeSuites.isEmpty()) {
      return empty();
    }
    return of(SuiteDeployment.builder().withSuiteRuns(runtimeSuites).withEnableXmlValidations(true).build());
  }

  private String getParameterizationName(SuiteRun suiteRun) {
    return suiteRun.getParameterization().map(Parameterization::getParameterizationName).orElse(EMPTY);
  }

  private String getRuntimeVersion() {
    return container.getMuleContainerVersion();
  }

  private Product getContainerRuntimeProduct() {
    String containerRuntimeProduct = runConfig.getContainerConfiguration().getProduct();
    if (containerRuntimeProduct == null) {
      return null;
    }
    if (containerRuntimeProduct.equals(org.mule.runtime.module.embedded.api.Product.MULE_FRAMEWORK.name())) {
      // Use EE as the actual runtime to run the MuleFramework within
      return Product.MULE_EE;
    } else {
      return Product.valueOf(containerRuntimeProduct);
    }
  }

  private CoverageManager buildCoverageManager(CoverageConfiguration coverageConfiguration) {
    if (coverageConfiguration == null || !coverageConfiguration.isRunCoverage()) {
      return null;
    }
    Integer coverageServerPort = coverageConfiguration.getCoveragePort();
    Boolean randomizeCoveragePort = coverageConfiguration.isRandomizeCoveragePort();

    CoverageManager coverageManager =
        new CoverageManager(randomizeCoveragePort, coverageServerPort, coverageConfiguration.isRunCoverage(),
                            coverageConfiguration.getSuitePaths(), coverageConfiguration.getCoverageServerTimeout());
    coverageManager.setIgnoreFlows(ofNullable(coverageConfiguration.getIgnoredFlowNames()).orElse(emptySet()));
    coverageManager.setIgnoreFiles(ofNullable(coverageConfiguration.getIgnoredFiles()).orElse(emptySet()));
    return coverageManager;
  }

  private Optional<CoverageManager> getCoverageManager() {
    return Optional.ofNullable(coverageManager);
  }

  private void waitForDebuggerClient() {
    DebuggerConfiguration debuggerConfiguration = runConfig.getDebuggerConfiguration();
    if (debuggerConfiguration == null || debuggerConfiguration.getDebuggerPort() == 0) {
      return;
    }

    MuleDxMunitLock lock = new MuleDxMunitLock(debuggerConfiguration.getLockKey());
    lock.lock(debuggerConfiguration.getLockTries());
  }

  public ContainerSuiteRunDispatcher addSuiteRunDispatcherLister(SuiteRunDispatcherListener suiteRunDispatcherListener) {
    this.suiteRunDispatcherListeners.add(suiteRunDispatcherListener);
    return this;
  }

  private void sendCoverageReport(CoverageManager coverageManager, RemoteRunEventListener listener) {
    Optional<ApplicationCoverageReport> coverageReport = coverageManager.generateCoverageReport();
    if (coverageReport.isPresent()) {
      String coverageReportJson = new Gson().toJson(coverageReport.get());
      listener.notifyCoverageReport(coverageReportJson);
    }
  }

  private Long getSuiteTimeout() {
    return Long.valueOf(System.getProperty(MUNIT_SUITE_TIMEOUT_PROPERTY, DEFAULT_TIMEOUT.toString()));
  }

}
