/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.remote;


import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.mule.munit.common.properties.MUnitUserPropertiesManager;
import org.mule.munit.common.protocol.listeners.RemoteRunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListener;
import org.mule.munit.common.protocol.listeners.RunEventListenerContainer;
import org.mule.munit.common.util.StackTraceUtil;
import org.mule.munit.remote.api.configuration.NotifierConfiguration;
import org.mule.munit.remote.api.configuration.RunConfiguration;
import org.mule.munit.remote.api.configuration.RunConfigurationParser;
import org.mule.munit.remote.classloading.ClassLoaderUtils;
import org.mule.munit.remote.container.ContainerFactoryException;
import org.mule.munit.remote.container.EmbeddedContainerFactory;
import org.mule.munit.remote.container.SuiteRunDispatcher;
import org.mule.munit.remote.exception.DeploymentException;
import org.mule.munit.remote.notifier.NotifierReflectionFactory;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;


/**
 * This is the main entry point to run a set of MUnit Test Suite files.
 *
 *
 * If could be run from any command line on its own VM provided it receives the proper parameters defined in @link
 * {@link RunConfigurationParser}.
 *
 * The final goal of the class is to run each MUnit Test Suite file provided and report back the run results and the coverage
 * results.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class RemoteRunner {

  protected transient Logger logger = LogManager.getLogger(this.getClass());

  public static final String MUNIT_DEPLOYMENT_USERNAME = "munit.deployment.username";
  public static final String MUNIT_DEPLOYMENT_PASSWORD = "munit.deployment.password";

  public static final String SYSTEM_PROPERTIES_FILE = "munit.system.properties.file";
  public static final String MUNIT_DEBUG_LOG_CLASSPATH = "munit.debug.log.classpath";

  private EmbeddedContainerFactory containerFactory;

  public static void main(String args[]) throws ParseException, IOException, URISyntaxException {
    loadSystemProperties();
    RunConfiguration runConfig = new RunConfigurationParser().parse(args);
    RemoteRunner runner = new RemoteRunner();

    runner.setContainerFactory(new EmbeddedContainerFactory());
    runner.run(runConfig);
    shutDown();
  }

  public static void mainStarter(String runConfig) throws ParseException, IOException, URISyntaxException {
    //    main((String[]) args.toArray(new String[0]));
    String[] args = new String[] {"-run_configuration", runConfig};
    main(args);
  };

  public void setContainerFactory(EmbeddedContainerFactory containerFactory) {
    this.containerFactory = containerFactory;
  }

  public void run(RunConfiguration runConfig) throws IOException {
    logger.info("Run Started");
    logClassPathIfNecessary();

    RemoteRunEventListener listener = buildRunnerListener(runConfig.getNotifierConfigurations());
    listener.notifyRunStart();

    try {
      logger.info("Mule container is starting, it could take a few minutes");
      SuiteRunDispatcher suiteRunDispatcher = buildSuiteRunDispatcher(runConfig);

      suiteRunDispatcher.runSuites(listener);

    } catch (ContainerFactoryException e) {
      listener.notifyUnexpectedError(StackTraceUtil.getStackTrace(e));
      listener.notifyRunFinish();
      logger.info("Done");
      return;
    } catch (DeploymentException e) {
      listener.notifyContainerFailure(StackTraceUtil.getStackTrace(e));
    } catch (Throwable e) {
      e.printStackTrace();
      listener.notifyUnexpectedError(StackTraceUtil.getStackTrace(e));
    }

    listener.notifyRunFinish();
    logger.info("Done");
  }

  protected SuiteRunDispatcher buildSuiteRunDispatcher(RunConfiguration runConfig) throws ContainerFactoryException {
    return new SuiteRunDispatcherFactory(runConfig, containerFactory).create();
  }

  protected RemoteRunEventListener buildRunnerListener(List<NotifierConfiguration> configurations) {
    List<RunEventListener> listeners = new NotifierReflectionFactory().createNotifiers(configurations);
    RunEventListenerContainer container = new RunEventListenerContainer();
    listeners.forEach(container::addNotificationListener);
    return container;
  }

  private void logClassPathIfNecessary() {
    if (Boolean.valueOf(System.getProperty(MUNIT_DEBUG_LOG_CLASSPATH, "false"))) {
      logger.info("logging classpath ...");
      new ClassLoaderUtils().getClassPath().forEach(System.out::println);
      logger.info("logging classpath DONE");
    }
  }

  private static void loadSystemProperties() {
    loadSystemPropertiesFileIfPresent();
  }

  private static void loadSystemPropertiesFileIfPresent() {
    String filePath = System.getProperty(SYSTEM_PROPERTIES_FILE);
    try {
      if (filePath != null) {
        Properties properties = new Properties();
        properties.load(new FileInputStream(new File(filePath)));
        MUnitUserPropertiesManager.loadPropertiesToSystem(properties);
      }
    } catch (IOException e) {
      // TODO handle this properly
    }
  }

  private static void shutDown() {
    System.exit(0);
  }
}
