/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.extension.maven.internal.generator;

import static com.google.common.collect.Sets.difference;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;

/**
 * Generates a new {@link MuleApplicationModel} with new parameters based on an original
 *
 * @since 1.0.0
 * @author Mulesoft Inc.
 */
public class ApplicationMuleArtifactJsonGenerator {

  protected static final String EXPORTED_RESOURCES = "exportedResources";
  protected static final String EXPORTED_PACKAGES = "exportedPackages";
  protected static final String INCLUDE_TEST_DEPENDENCIES = "includeTestDependencies";

  private Set<String> configs;
  private Set<String> exportedResources;
  private Set<String> exportedPackages;

  public ApplicationMuleArtifactJsonGenerator(Set<String> configs, Set<String> exportedResources, Set<String> exportedPackages) {
    this.configs = configs;
    this.exportedResources = exportedResources;
    this.exportedPackages = exportedPackages;
  }

  public MuleApplicationModel generate(MuleApplicationModel originalMuleApplicationModel) {
    MuleApplicationModel.MuleApplicationModelBuilder builder = new MuleApplicationModel.MuleApplicationModelBuilder();
    builder.setConfigs(configs);
    builder.setMinMuleVersion(originalMuleApplicationModel.getMinMuleVersion());
    builder.setName(originalMuleApplicationModel.getName());
    builder.setRedeploymentEnabled(true);
    builder.setRequiredProduct(originalMuleApplicationModel.getRequiredProduct());
    builder.setSecureProperties(originalMuleApplicationModel.getSecureProperties());
    builder.withBundleDescriptorLoader(originalMuleApplicationModel.getBundleDescriptorLoader());
    builder.withClassLoaderModelDescriptorLoader(getClassLoaderModelLoaderDescriptor(originalMuleApplicationModel));
    return builder.build();
  }

  private MuleArtifactLoaderDescriptor getClassLoaderModelLoaderDescriptor(MuleApplicationModel originalMuleApplicationModel) {
    MuleArtifactLoaderDescriptor descriptor = originalMuleApplicationModel.getClassLoaderModelLoaderDescriptor();
    Map<String, Object> newAttributes = new HashMap<>(descriptor.getAttributes());
    overrideValues(newAttributes, EXPORTED_RESOURCES, exportedResources);
    // noinspection unchecked
    overrideValues(newAttributes, EXPORTED_PACKAGES,
                   cleanse(exportedPackages, (Collection<String>) newAttributes.get(EXPORTED_PACKAGES)));
    newAttributes.put(INCLUDE_TEST_DEPENDENCIES, Boolean.TRUE.toString());
    return new MuleArtifactLoaderDescriptor(descriptor.getId(), newAttributes);
  }

  private void overrideValues(Map<String, Object> attributes, String key, Set<String> newValues) {
    attributes.put(key, newValues);
  }

  private Set<String> cleanse(Set<String> values, Collection<String> remove) {
    if (remove != null && remove.size() > 0) {
      return new HashSet<>(difference(values, new HashSet<>(remove)));
    }

    return values;
  }
}
